package physx.particles;

import physx.NativeObject;
import physx.common.PxBase;
import physx.common.PxVec4;

/**
 * The shared base class for all particle buffers, can be instantiated directly to simulate granular and fluid particles.
 * <p>
 * See #PxPhysics::createParticleBuffer.
 * <p>
 * A particle buffer is a container that specifies per-particle attributes of a set of particles that will be used during the simulation 
 * of a particle system. It exposes direct access to the underlying GPU buffers and is independent of the scene and particle system. Particle
 * buffers can be added/removed from a particle system at any time between simulation steps, and transferred from one particle system to another.
 */
public class PxParticleBuffer extends PxBase {

    protected PxParticleBuffer() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxParticleBuffer wrapPointer(long address) {
        return address != 0L ? new PxParticleBuffer(address) : null;
    }
    
    public static PxParticleBuffer arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxParticleBuffer(long address) {
        super(address);
    }

    // Attributes

    /**
     * Index of this buffer in the particle system it is assigned to.
     */
    public int getBufferIndex() {
        checkNotNull();
        return _getBufferIndex(address);
    }
    private static native int _getBufferIndex(long address);

    /**
     * Unique index that does not change over the lifetime of a PxParticleBuffer.
     */
    public int getBufferUniqueId() {
        checkNotNull();
        return _getBufferUniqueId(address);
    }
    private static native int _getBufferUniqueId(long address);

    // Functions

    /**
     * Get positions and inverse masses for this particle buffer.
     * @return A pointer to a device buffer containing the positions and inverse mass packed as PxVec4(pos.x, pos.y, pos.z, inverseMass).
     */
    public PxVec4 getPositionInvMasses() {
        checkNotNull();
        return PxVec4.wrapPointer(_getPositionInvMasses(address));
    }
    private static native long _getPositionInvMasses(long address);

    /**
     * Get velocities for this particle buffer.
     * @return A pointer to a device buffer containing the velocities packed as PxVec4(vel.x, vel.y, vel.z, 0.0f).
     */
    public PxVec4 getVelocities() {
        checkNotNull();
        return PxVec4.wrapPointer(_getVelocities(address));
    }
    private static native long _getVelocities(long address);

    /**
     * Get phases for this particle buffer.
     * <p>
     * See #PxParticlePhase
     * @return A pointer to a device buffer containing the per-particle phases for this particle buffer.
     */
    public NativeObject getPhases() {
        checkNotNull();
        return NativeObject.wrapPointer(_getPhases(address));
    }
    private static native long _getPhases(long address);

    /**
     * Get particle volumes for this particle buffer.
     * <p>
     * See #PxParticleVolume
     * @return A pointer to a device buffer containing the #PxParticleVolume s for this particle buffer.
     */
    public PxParticleVolume getParticleVolumes() {
        checkNotNull();
        return PxParticleVolume.wrapPointer(_getParticleVolumes(address));
    }
    private static native long _getParticleVolumes(long address);

    /**
     * Set the number of active particles for this particle buffer.
     * @param nbActiveParticles The number of active particles.
     * <p>
     * The number of active particles can be &lt;= PxParticleBuffer::getMaxParticles(). The particle system will simulate the first
     * x particles in the #PxParticleBuffer, where x is the number of active particles.
     */
    public void setNbActiveParticles(int nbActiveParticles) {
        checkNotNull();
        _setNbActiveParticles(address, nbActiveParticles);
    }
    private static native void _setNbActiveParticles(long address, int nbActiveParticles);

    /**
     * Get the number of active particles for this particle buffer.
     * @return The number of active particles.
     */
    public int getNbActiveParticles() {
        checkNotNull();
        return _getNbActiveParticles(address);
    }
    private static native int _getNbActiveParticles(long address);

    /**
     * Get the maximum number particles this particle buffer can hold.
     * <p>
     * The maximum number of particles is specified when creating a #PxParticleBuffer. See #PxPhysics::createParticleBuffer.
     * @return The maximum number of particles.
     */
    public int getMaxParticles() {
        checkNotNull();
        return _getMaxParticles(address);
    }
    private static native int _getMaxParticles(long address);

    /**
     * Get the number of particle volumes in this particle buffer.
     * @return The number of #PxParticleVolume s for this particle buffer.
     */
    public int getNbParticleVolumes() {
        checkNotNull();
        return _getNbParticleVolumes(address);
    }
    private static native int _getNbParticleVolumes(long address);

    /**
     * Set the number of #PxParticleVolume s for this particle buffer.
     * @param nbParticleVolumes The number of particle volumes in this particle buffer.
     */
    public void setNbParticleVolumes(int nbParticleVolumes) {
        checkNotNull();
        _setNbParticleVolumes(address, nbParticleVolumes);
    }
    private static native void _setNbParticleVolumes(long address, int nbParticleVolumes);

    /**
     * Get the maximum number of particle volumes this particle buffer can hold.
     * <p>
     * See #PxParticleVolume.
     * @return The maximum number of particle volumes this particle buffer can hold.
     */
    public int getMaxParticleVolumes() {
        checkNotNull();
        return _getMaxParticleVolumes(address);
    }
    private static native int _getMaxParticleVolumes(long address);

    /**
     * Set the #PxParticleRigidFilterPair s for collision filtering of particles in this buffer with rigid bodies.
     * <p>
     * See #PxParticleRigidFilterPair
     * @param filters A device buffer containing #PxParticleRigidFilterPair s.
     * @param nbFilters The number of particle-rigid body collision filtering pairs.
     */
    public void setRigidFilters(PxParticleRigidFilterPair filters, int nbFilters) {
        checkNotNull();
        _setRigidFilters(address, filters.getAddress(), nbFilters);
    }
    private static native void _setRigidFilters(long address, long filters, int nbFilters);

    /**
     * Set the particle-rigid body attachments for particles in this particle buffer.
     * <p>
     * See #PxParticleRigidAttachment
     * @param attachments A device buffer containing #PxParticleRigidAttachment s.
     * @param nbAttachments The number of particle-rigid body attachments.
     */
    public void setRigidAttachments(PxParticleRigidAttachment attachments, int nbAttachments) {
        checkNotNull();
        _setRigidAttachments(address, attachments.getAddress(), nbAttachments);
    }
    private static native void _setRigidAttachments(long address, long attachments, int nbAttachments);

    /**
     * Get the start index for the first particle of this particle buffer in the complete list of
     * particles of the particle system this buffer is used in.
     * <p>
     * The return value is only correct if the particle buffer is assigned to a particle system and at least
     * one call to simulate() has been performed.
     * @return The index of the first particle in the complete particle list.
     */
    public int getFlatListStartIndex() {
        checkNotNull();
        return _getFlatListStartIndex(address);
    }
    private static native int _getFlatListStartIndex(long address);

    /**
     * Raise dirty flags on this particle buffer to communicate that the corresponding data has been updated
     * by the user.
     * @param flags The flag corresponding to the data that is dirty.
     * <p>
     * See #PxParticleBufferFlag.
     */
    public void raiseFlags(PxParticleBufferFlagEnum flags) {
        checkNotNull();
        _raiseFlags(address, flags.value);
    }
    private static native void _raiseFlags(long address, int flags);

    /**
     * Release this buffer and deallocate all the memory.
     */
    public void release() {
        checkNotNull();
        _release(address);
    }
    private static native void _release(long address);

}
