package physx.particles;

import physx.common.PxCudaContextManager;
import physx.common.PxVec3;
import physx.physics.PxActor;
import physx.physics.PxFilterData;
import physx.physics.PxRigidActor;

/**
 * The shared base class for all particle systems
 * <p>
 * A particle system simulates a bunch of particles that interact with each other. The interactions can be simple collisions 
 * with friction (granular material) ore more complex like fluid interactions, cloth, inflatables etc.
 */
public class PxParticleSystem extends PxActor {

    protected PxParticleSystem() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxParticleSystem wrapPointer(long address) {
        return address != 0L ? new PxParticleSystem(address) : null;
    }
    
    public static PxParticleSystem arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxParticleSystem(long address) {
        super(address);
    }

    // Functions

    /**
     * Sets the solver iteration counts for the body.
     * <p>
     * The solver iteration count determines how accurately joints and contacts are resolved.
     * If you are having trouble with jointed bodies oscillating and behaving erratically, then
     * setting a higher position iteration count may improve their stability.
     * <p>
     * If intersecting bodies are being depenetrated too violently, increase the number of velocity
     * iterations. More velocity iterations will drive the relative exit velocity of the intersecting
     * objects closer to the correct value given the restitution.
     * <p>
     * <b>Default:</b> 4 position iterations, 1 velocity iteration
     * @param minPositionIters Number of position iterations the solver should perform for this body. <b>Range:</b> [1,255]
     * @param minVelocityIters Number of velocity iterations the solver should perform for this body. <b>Range:</b> [1,255]
     * <p>
     * See #getSolverIterationCounts()
     */
    public void setSolverIterationCounts(int minPositionIters, int minVelocityIters) {
        checkNotNull();
        _setSolverIterationCounts(address, minPositionIters, minVelocityIters);
    }
    private static native void _setSolverIterationCounts(long address, int minPositionIters, int minVelocityIters);

    /**
     * Retrieves the collision filter settings.
     * @return The filter data
     */
    public PxFilterData getSimulationFilterData() {
        checkNotNull();
        return PxFilterData.wrapPointer(_getSimulationFilterData(address));
    }
    private static native long _getSimulationFilterData(long address);

    /**
     * Set collision filter settings
     * <p>
     * Allows to control with which objects the particle system collides
     * @param data The filter data
     */
    public void setSimulationFilterData(PxFilterData data) {
        checkNotNull();
        _setSimulationFilterData(address, data.getAddress());
    }
    private static native void _setSimulationFilterData(long address, long data);

    /**
     * Set particle flag
     * <p>
     * Allows to control self collision etc.
     * @param flag The flag to set
     * @param val The new value of the flag
     */
    public void setParticleFlag(PxParticleFlagEnum flag, boolean val) {
        checkNotNull();
        _setParticleFlag(address, flag.value, val);
    }
    private static native void _setParticleFlag(long address, int flag, boolean val);

    /**
     * Set particle flags
     * <p>
     * Allows to control self collision etc.
     * @param flags The flags to set
     */
    public void setParticleFlags(PxParticleFlags flags) {
        checkNotNull();
        _setParticleFlags(address, flags.getAddress());
    }
    private static native void _setParticleFlags(long address, long flags);

    /**
     * Retrieves the particle flags.
     * @return The particle flags
     */
    public PxParticleFlags getParticleFlags() {
        checkNotNull();
        return PxParticleFlags.wrapPointer(_getParticleFlags(address));
    }
    private static native long _getParticleFlags(long address);

    /**
     * Set the maximal depenetration velocity particles can reach
     * <p>
     * Allows to limit the particles' maximal depenetration velocity to avoid that collision responses lead to very high particle velocities
     * @param maxDepenetrationVelocity The maximal depenetration velocity
     */
    public void setMaxDepenetrationVelocity(float maxDepenetrationVelocity) {
        checkNotNull();
        _setMaxDepenetrationVelocity(address, maxDepenetrationVelocity);
    }
    private static native void _setMaxDepenetrationVelocity(long address, float maxDepenetrationVelocity);

    /**
     * Retrieves maximal depenetration velocity a particle can have.
     * @return The maximal depenetration velocity
     */
    public float getMaxDepenetrationVelocity() {
        checkNotNull();
        return _getMaxDepenetrationVelocity(address);
    }
    private static native float _getMaxDepenetrationVelocity(long address);

    /**
     * Set the maximal velocity particles can reach
     * <p>
     * Allows to limit the particles' maximal velocity to control the maximal distance a particle can move per frame
     * @param maxVelocity The maximal velocity
     */
    public void setMaxVelocity(float maxVelocity) {
        checkNotNull();
        _setMaxVelocity(address, maxVelocity);
    }
    private static native void _setMaxVelocity(long address, float maxVelocity);

    /**
     * Retrieves maximal velocity a particle can have.
     * @return The maximal velocity
     */
    public float getMaxVelocity() {
        checkNotNull();
        return _getMaxVelocity(address);
    }
    private static native float _getMaxVelocity(long address);

    /**
     * Return the cuda context manager
     * @return The cuda context manager
     */
    public PxCudaContextManager getCudaContextManager() {
        checkNotNull();
        return PxCudaContextManager.wrapPointer(_getCudaContextManager(address));
    }
    private static native long _getCudaContextManager(long address);

    /**
     * Set the rest offset for the collision between particles and rigids or soft bodies.
     * <p>
     * A particle and a rigid or soft body will come to rest at a distance equal to the sum of their restOffset values.
     * @param restOffset <b>Range:</b> (0, contactOffset)
     */
    public void setRestOffset(float restOffset) {
        checkNotNull();
        _setRestOffset(address, restOffset);
    }
    private static native void _setRestOffset(long address, float restOffset);

    /**
     * Return the rest offset
     * @return the rest offset
     * <p>
     * See #setRestOffset()
     */
    public float getRestOffset() {
        checkNotNull();
        return _getRestOffset(address);
    }
    private static native float _getRestOffset(long address);

    /**
     * Set the contact offset for the collision between particles and rigids or soft bodies
     * <p>
     * The contact offset needs to be larger than the rest offset.
     * Contact constraints are generated for a particle and a rigid or softbody below the distance equal to the sum of their contacOffset values.
     * @param contactOffset <b>Range:</b> (restOffset, PX_MAX_F32)
     */
    public void setContactOffset(float contactOffset) {
        checkNotNull();
        _setContactOffset(address, contactOffset);
    }
    private static native void _setContactOffset(long address, float contactOffset);

    /**
     * Return the contact offset
     * @return the contact offset
     * <p>
     * See #setContactOffset()
     */
    public float getContactOffset() {
        checkNotNull();
        return _getContactOffset(address);
    }
    private static native float _getContactOffset(long address);

    /**
     * Set the contact offset for the interactions between particles
     * <p>
     * The particle contact offset needs to be larger than the fluid rest offset and larger than the solid rest offset.
     * Interactions for two particles are computed if their distance is below twice the particleContactOffset value.
     * @param particleContactOffset <b>Range:</b> (Max(solidRestOffset, fluidRestOffset), PX_MAX_F32)
     */
    public void setParticleContactOffset(float particleContactOffset) {
        checkNotNull();
        _setParticleContactOffset(address, particleContactOffset);
    }
    private static native void _setParticleContactOffset(long address, float particleContactOffset);

    /**
     * Return the particle contact offset
     * @return the particle contact offset
     * <p>
     * See #setParticleContactOffset()
     */
    public float getParticleContactOffset() {
        checkNotNull();
        return _getParticleContactOffset(address);
    }
    private static native float _getParticleContactOffset(long address);

    /**
     * Set the solid rest offset
     * <p>
     * Two solid particles (or a solid and a fluid particle) will come to rest at a distance equal to twice the solidRestOffset value.
     * @param solidRestOffset  <b>Range:</b> (0, particleContactOffset)
     */
    public void setSolidRestOffset(float solidRestOffset) {
        checkNotNull();
        _setSolidRestOffset(address, solidRestOffset);
    }
    private static native void _setSolidRestOffset(long address, float solidRestOffset);

    /**
     * Return the solid rest offset
     * @return the solid rest offset
     * <p>
     * See #setSolidRestOffset()
     */
    public float getSolidRestOffset() {
        checkNotNull();
        return _getSolidRestOffset(address);
    }
    private static native float _getSolidRestOffset(long address);

    /**
     * Creates a rigid attachment between a particle and a rigid actor.
     * <p>
     * This method creates a symbolic attachment between the particle system and a rigid body for the purpose of island management.
     * The actual attachments will be contained in the particle buffers.
     * <p>
     * Be aware that destroying the rigid body before destroying the attachment is illegal and may cause a crash.
     * The particle system keeps track of these attachments but the rigid body does not.
     * @param actor The rigid actor used for the attachment
     */
    public void addRigidAttachment(PxRigidActor actor) {
        checkNotNull();
        _addRigidAttachment(address, actor.getAddress());
    }
    private static native void _addRigidAttachment(long address, long actor);

    /**
     * Removes a rigid attachment between a particle and a rigid body.
     * <p>
     * This method destroys a symbolic attachment between the particle system and a rigid body for the purpose of island management.
     * <p>
     * Be aware that destroying the rigid body before destroying the attachment is illegal and may cause a crash.
     * The particle system keeps track of these attachments but the rigid body does not.
     * @param actor The rigid body actor used for the attachment
     */
    public void removeRigidAttachment(PxRigidActor actor) {
        checkNotNull();
        _removeRigidAttachment(address, actor.getAddress());
    }
    private static native void _removeRigidAttachment(long address, long actor);

    /**
     * Enable continuous collision detection for particles
     * @param enable Boolean indicates whether continuous collision detection is enabled.
     */
    public void enableCCD(boolean enable) {
        checkNotNull();
        _enableCCD(address, enable);
    }
    private static native void _enableCCD(long address, boolean enable);

    /**
     * Creates combined particle flag with particle material and particle phase flags.
     * @param material A material instance to associate with the new particle group.
     * @param flags The particle phase flags.
     * @return The combined particle group index and phase flags.
     * <p>
     * See #PxParticlePhaseFlag
     */
    public int createPhase(PxParticleMaterial material, PxParticlePhaseFlags flags) {
        checkNotNull();
        return _createPhase(address, material.getAddress(), flags.getAddress());
    }
    private static native int _createPhase(long address, long material, long flags);

    /**
     * Returns number of particle materials
     * @return The number of particle materials
     */
    public int getNbParticleMaterials() {
        checkNotNull();
        return _getNbParticleMaterials(address);
    }
    private static native int _getNbParticleMaterials(long address);

    /**
     * Sets periodic boundary wrap value
     * @param boundary The periodic boundary wrap value
     * <p>
     * See #getPeriodicBoundary()
     */
    public void setPeriodicBoundary(PxVec3 boundary) {
        checkNotNull();
        _setPeriodicBoundary(address, boundary.getAddress());
    }
    private static native void _setPeriodicBoundary(long address, long boundary);

    /**
     * Gets periodic boundary wrap value
     * @return  boundary The periodic boundary wrap value
     * <p>
     * See #setPeriodicBoundary()
     */
    public PxVec3 getPeriodicBoundary() {
        checkNotNull();
        return PxVec3.wrapPointer(_getPeriodicBoundary(address));
    }
    private static native long _getPeriodicBoundary(long address);

    /**
     * Add an existing particle buffer to the particle system.
     * @param particleBuffer a PxParticleBuffer*.
     * <p>
     *    See #PxParticleBuffer.
     */
    public void addParticleBuffer(PxParticleBuffer particleBuffer) {
        checkNotNull();
        _addParticleBuffer(address, particleBuffer.getAddress());
    }
    private static native void _addParticleBuffer(long address, long particleBuffer);

    /**
     * Remove particle buffer from the particle system.
     * @param particleBuffer a PxParticleBuffer*.
     * <p>
     * See #PxParticleBuffer.
     */
    public void removeParticleBuffer(PxParticleBuffer particleBuffer) {
        checkNotNull();
        _removeParticleBuffer(address, particleBuffer.getAddress());
    }
    private static native void _removeParticleBuffer(long address, long particleBuffer);

    /**
     * Returns the GPU particle system index.
     * @return The GPU index, if the particle system is in a scene and PxSceneFlag::eSUPPRESS_READBACK is set, or 0xFFFFFFFF otherwise.
     */
    public int getGpuParticleSystemIndex() {
        checkNotNull();
        return _getGpuParticleSystemIndex(address);
    }
    private static native int _getGpuParticleSystemIndex(long address);

}
