package physx.physics;

import physx.common.PxBase;
import physx.geometry.PxBVHStructure;

/**
 * Class to aggregate actors into a single broad-phase entry.
 * <p>
 * A PxAggregate object is a collection of PxActors, which will exist as a single entry in the
 * broad-phase structures. This has 3 main benefits:
 * <p>
 * 1) it reduces "broad phase pollution" by allowing a collection of spatially coherent broad-phase 
 * entries to be replaced by a single aggregated entry (e.g. a ragdoll or a single actor with a 
 * large number of attached shapes).
 * <p>
 * 2) it reduces broad-phase memory usage
 * <p>
 * 3) filtering can be optimized a lot if self-collisions within an aggregate are not needed. For
 *    example if you don't need collisions between ragdoll bones, it's faster to simply disable
 *    filtering once and for all, for the aggregate containing the ragdoll, rather than filtering
 *    out each bone-bone collision in the filter shader.
 * @see PxPhysics#createAggregate
 */
public class PxAggregate extends PxBase {

    protected PxAggregate() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxAggregate wrapPointer(long address) {
        return address != 0L ? new PxAggregate(address) : null;
    }
    
    public static PxAggregate arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxAggregate(long address) {
        super(address);
    }

    // Functions

    /**
     * Adds an actor to the aggregate object.
     * <p>
     * A warning is output if the total number of actors is reached, or if the incoming actor already belongs
     * to an aggregate.
     * <p>
     * If the aggregate belongs to a scene, adding an actor to the aggregate also adds the actor to that scene.
     * <p>
     * If the actor already belongs to a scene, a warning is output and the call is ignored. You need to remove
     * the actor from the scene first, before adding it to the aggregate.
     * <p>
     * <b>Note:</b> When a BVH is provided the actor shapes are grouped together. 
     * The scene query pruning structure inside PhysX SDK will store/update one
     * bound per actor. The scene queries against such an actor will query actor
     * bounds and then make a local space query against the provided BVH, which is in actor's local space.
     * <p>
     * return true if success
     */
    public boolean addActor(PxActor actor) {
        checkNotNull();
        return _addActor(address, actor.getAddress());
    }
    private static native boolean _addActor(long address, long actor);

    /**
     * Adds an actor to the aggregate object.
     * <p>
     * A warning is output if the total number of actors is reached, or if the incoming actor already belongs
     * to an aggregate.
     * <p>
     * If the aggregate belongs to a scene, adding an actor to the aggregate also adds the actor to that scene.
     * <p>
     * If the actor already belongs to a scene, a warning is output and the call is ignored. You need to remove
     * the actor from the scene first, before adding it to the aggregate.
     * <p>
     * <b>Note:</b> When a BVH is provided the actor shapes are grouped together. 
     * The scene query pruning structure inside PhysX SDK will store/update one
     * bound per actor. The scene queries against such an actor will query actor
     * bounds and then make a local space query against the provided BVH, which is in actor's local space.
     * <p>
     * return true if success
     */
    public boolean addActor(PxActor actor, PxBVHStructure bvhStructure) {
        checkNotNull();
        return _addActor(address, actor.getAddress(), bvhStructure.getAddress());
    }
    private static native boolean _addActor(long address, long actor, long bvhStructure);

    /**
     * Removes an actor from the aggregate object.
     * <p>
     * A warning is output if the incoming actor does not belong to the aggregate. Otherwise the actor is
     * removed from the aggregate. If the aggregate belongs to a scene, the actor is reinserted in that
     * scene. If you intend to delete the actor, it is best to call #PxActor::release() directly. That way
     * the actor will be automatically removed from its aggregate (if any) and not reinserted in a scene.
     * <p>
     * return true if success
     */
    public boolean removeActor(PxActor actor) {
        checkNotNull();
        return _removeActor(address, actor.getAddress());
    }
    private static native boolean _removeActor(long address, long actor);

    /**
     * Returns the number of actors contained in the aggregate.
     * <p>
     * You can use #getActors() to retrieve the actor pointers.
     * @return Number of actors contained in the aggregate.
     * @see PxActor
     */
    public int getNbActors() {
        checkNotNull();
        return _getNbActors(address);
    }
    private static native int _getNbActors(long address);

    /**
     * Retrieves max amount of actors that can be contained in the aggregate.
     * <p>
     * <b>Note:</b> PxAggregate now supports an arbitrary number of actors. This method return PX_MAX_U32 and will be
     * removed in a future release.
     * @return Max actor size. 
     * @deprecated
     */
    @Deprecated
    public int getMaxNbActors() {
        checkNotNull();
        return _getMaxNbActors(address);
    }
    private static native int _getMaxNbActors(long address);

    /**
     * Retrieves the scene which this aggregate belongs to.
     * @return Owner Scene. NULL if not part of a scene.
     * @see PxScene
     */
    public PxScene getScene() {
        checkNotNull();
        return PxScene.wrapPointer(_getScene(address));
    }
    private static native long _getScene(long address);

    /**
     * Retrieves aggregate's self-collision flag.
     * @return self-collision flag
     */
    public boolean getSelfCollision() {
        checkNotNull();
        return _getSelfCollision(address);
    }
    private static native boolean _getSelfCollision(long address);

}
