package physx.physics;


/**
 * A fixed tendon that can be used to link multiple degrees of freedom of multiple articulation joints via length and limit constraints.
 * <p>
 * Fixed tendons allow the simulation of coupled relationships between joint degrees of freedom in an articulation. Fixed tendons do not allow
 * linking arbitrary joint axes of the articulation: The respective joints must all be directly connected to each other in the articulation structure,
 * i.e. each of the joints in the tendon must be connected by a single articulation link to another joint in the same tendon. This implies both that 
 * 1) fixed tendons can branch along a branching articulation; and 2) they cannot be used to create relationships between axes in a spherical joint with
 * more than one degree of freedom. Locked joint axes or fixed joints are currently not supported.
 */
public class PxArticulationFixedTendon extends PxArticulationTendon {

    protected PxArticulationFixedTendon() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxArticulationFixedTendon wrapPointer(long address) {
        return address != 0L ? new PxArticulationFixedTendon(address) : null;
    }
    
    public static PxArticulationFixedTendon arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxArticulationFixedTendon(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Creates an articulation tendon joint and adds it to the list of children in the parent tendon joint.
     * <p>
     * Creating a tendon joint is not allowed while the articulation is in a scene. In order to
     * add the joint, remove and then re-add the articulation to the scene.
     * @param parent The parent tendon joint. Can be NULL for the root tendon joint of a tendon.
     * @param axis The degree of freedom that this tendon joint is associated with.
     * @param coefficient A user-defined scale that the accumulated tendon length is scaled by.
     * @param recipCoefficient The scale that the tendon's response is multiplied by when applying to this tendon joint.
     * @param link The link (and the link's incoming joint in particular) that this tendon joint is associated with.
     * @return The newly-created tendon joint if creation was successful, otherwise a null pointer.
     * <p>
     * <b>Note:</b>
     * - The axis motion must not be configured as PxArticulationMotion::eLOCKED.
     * - The axis cannot be part of a fixed joint, i.e. joint configured as PxArticulationJointType::eFIX.
     * @see PxArticulationTendonJoint
     */
    public PxArticulationTendonJoint createTendonJoint(PxArticulationTendonJoint parent, PxArticulationAxisEnum axis, float coefficient, float recipCoefficient, PxArticulationLink link) {
        checkNotNull();
        return PxArticulationTendonJoint.wrapPointer(_createTendonJoint(address, parent.getAddress(), axis.value, coefficient, recipCoefficient, link.getAddress()));
    }
    private static native long _createTendonJoint(long address, long parent, int axis, float coefficient, float recipCoefficient, long link);

    /**
     * Returns the number of tendon joints in the tendon.
     * @return The number of tendon joints.
     */
    public int getNbTendonJoints() {
        checkNotNull();
        return _getNbTendonJoints(address);
    }
    private static native int _getNbTendonJoints(long address);

    /**
     * Sets the spring rest length of the tendon.
     * <p>
     * The accumulated "length" of a fixed tendon is a linear combination of the joint axis positions that the tendon is
     * associated with, scaled by the respective tendon joints' coefficients. As such, when the joint positions of all
     * joints are zero, the accumulated length of a fixed tendon is zero.
     * <p>
     * The spring of the tendon is not exerting any force on the articulation when the rest length is equal to the
     * tendon's accumulated length plus the tendon offset.
     * @param restLength The spring rest length of the tendon.
     * @see #getRestLength
     */
    public void setRestLength(float restLength) {
        checkNotNull();
        _setRestLength(address, restLength);
    }
    private static native void _setRestLength(long address, float restLength);

    /**
     * Gets the spring rest length of the tendon.
     * @return The spring rest length of the tendon.
     * @see #setRestLength
     */
    public float getRestLength() {
        checkNotNull();
        return _getRestLength(address);
    }
    private static native float _getRestLength(long address);

    /**
     * Sets the low and high limit on the length of the tendon.
     * @param parameter Struct with the low and high limit.
     * <p>
     * The limits, together with the damping and limit stiffness parameters, act on the accumulated length of the tendon.
     * @see PxArticulationTendonLimit
     * @see #getLimitParameters
     * @see #setRestLength
     */
    public void setLimitParameters(PxArticulationTendonLimit parameter) {
        checkNotNull();
        _setLimitParameters(address, parameter.getAddress());
    }
    private static native void _setLimitParameters(long address, long parameter);

    /**
     * Gets the low and high limit on the length of the tendon.
     * @return Struct with the low and high limit.
     * @see PxArticulationTendonLimit
     * @see #setLimitParameters
     */
    public PxArticulationTendonLimit getLimitParameters() {
        checkNotNull();
        return PxArticulationTendonLimit.wrapPointer(_getLimitParameters(address));
    }
    private static native long _getLimitParameters(long address);

}
