package physx.physics;

import physx.NativeObject;

/**
 * Defines a fixed-tendon joint on an articulation joint degree of freedom.
 */
public class PxArticulationTendonJoint extends NativeObject {

    protected PxArticulationTendonJoint() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxArticulationTendonJoint wrapPointer(long address) {
        return address != 0L ? new PxArticulationTendonJoint(address) : null;
    }
    
    public static PxArticulationTendonJoint arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxArticulationTendonJoint(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * user can assign this to whatever, usually to create a 1:1 relationship with a user object.
     */
    public NativeObject getUserData() {
        checkNotNull();
        return NativeObject.wrapPointer(_getUserData(address));
    }
    private static native long _getUserData(long address);

    /**
     * user can assign this to whatever, usually to create a 1:1 relationship with a user object.
     */
    public void setUserData(NativeObject value) {
        checkNotNull();
        _setUserData(address, value.getAddress());
    }
    private static native void _setUserData(long address, long value);

    // Functions

    /**
     * Sets the tendon joint coefficient.
     * @param axis The degree of freedom that the tendon joint operates on (must correspond to a degree of freedom of the associated link's incoming joint).
     * @param coefficient The scale that the axis' joint position is multiplied by when summing up the fixed tendon's length.
     * @param recipCoefficient The scale that the tendon's response is multiplied by when applying to this tendon joint.
     * <p>
     * <b>Note:</b> RecipCoefficient is commonly expected to be 1/coefficient, but it can be set to different values to tune behavior; for example, zero can be used to
     * have a joint axis only participate in the length computation of the tendon, but not have any tendon force applied to it.
     */
    public void setCoefficient(PxArticulationAxisEnum axis, float coefficient, float recipCoefficient) {
        checkNotNull();
        _setCoefficient(address, axis.value, coefficient, recipCoefficient);
    }
    private static native void _setCoefficient(long address, int axis, float coefficient, float recipCoefficient);

    /**
     * Gets the articulation link.
     * @return The articulation link (and its incoming joint in particular) that this tendon joint is associated with.
     */
    public PxArticulationLink getLink() {
        checkNotNull();
        return PxArticulationLink.wrapPointer(_getLink(address));
    }
    private static native long _getLink(long address);

    /**
     * Gets the parent tendon joint.
     * @return The parent tendon joint.
     */
    public PxArticulationTendonJoint getParent() {
        checkNotNull();
        return PxArticulationTendonJoint.wrapPointer(_getParent(address));
    }
    private static native long _getParent(long address);

    /**
     * Gets the tendon that the joint is a part of.
     * @return The tendon.
     * @see PxArticulationFixedTendon
     */
    public PxArticulationFixedTendon getTendon() {
        checkNotNull();
        return PxArticulationFixedTendon.wrapPointer(_getTendon(address));
    }
    private static native long _getTendon(long address);

    /**
     * Releases a tendon joint.
     * <p>
     * <b>Note:</b> Releasing a tendon joint is not allowed while the articulation is in a scene. In order to
     * release the joint, remove and then re-add the articulation to the scene.
     */
    public void release() {
        checkNotNull();
        _release(address);
    }
    private static native void _release(long address);

}
