package physx.physics;

import physx.NativeObject;

/**
 * Contact report pair information.
 * <p>
 * Instances of this class are passed to PxSimulationEventCallback.onContact(). If contact reports have been requested for a pair of shapes (see #PxPairFlag),
 * then the corresponding contact information will be provided through this structure.
 */
public class PxContactPair extends NativeObject {

    protected PxContactPair() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxContactPair wrapPointer(long address) {
        return address != 0L ? new PxContactPair(address) : null;
    }
    
    public static PxContactPair arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxContactPair(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * @param index Array index
     * @return WebIDL type: {@link PxShape}
     */
    public PxShape getShapes(int index) {
        checkNotNull();
        return PxShape.wrapPointer(_getShapes(address, index));
    }
    private static native long _getShapes(long address, int index);

    /**
     * @param index Array index
     * @param value WebIDL type: {@link PxShape}
     */
    public void setShapes(int index, PxShape value) {
        checkNotNull();
        _setShapes(address, index, value.getAddress());
    }
    private static native void _setShapes(long address, int index, long value);

    /**
     * Number of contact points stored in the contact stream
     */
    public byte getContactCount() {
        checkNotNull();
        return _getContactCount(address);
    }
    private static native byte _getContactCount(long address);

    /**
     * Number of contact points stored in the contact stream
     */
    public void setContactCount(byte value) {
        checkNotNull();
        _setContactCount(address, value);
    }
    private static native void _setContactCount(long address, byte value);

    /**
     * Number of contact patches stored in the contact stream
     */
    public byte getPatchCount() {
        checkNotNull();
        return _getPatchCount(address);
    }
    private static native byte _getPatchCount(long address);

    /**
     * Number of contact patches stored in the contact stream
     */
    public void setPatchCount(byte value) {
        checkNotNull();
        _setPatchCount(address, value);
    }
    private static native void _setPatchCount(long address, byte value);

    /**
     * Additional information on the contact report pair.
     */
    public PxContactPairFlags getFlags() {
        checkNotNull();
        return PxContactPairFlags.wrapPointer(_getFlags(address));
    }
    private static native long _getFlags(long address);

    /**
     * Additional information on the contact report pair.
     */
    public void setFlags(PxContactPairFlags value) {
        checkNotNull();
        _setFlags(address, value.getAddress());
    }
    private static native void _setFlags(long address, long value);

    /**
     * Flags raised due to the contact.
     * <p>
     * The events field is a combination of:
     * <p>
     * &lt;ul&gt;
     * &lt;li&gt;PxPairFlag::eNOTIFY_TOUCH_FOUND,&lt;/li&gt;
     * &lt;li&gt;PxPairFlag::eNOTIFY_TOUCH_PERSISTS,&lt;/li&gt;
     * &lt;li&gt;PxPairFlag::eNOTIFY_TOUCH_LOST,&lt;/li&gt;
     * &lt;li&gt;PxPairFlag::eNOTIFY_TOUCH_CCD,&lt;/li&gt;
     * &lt;li&gt;PxPairFlag::eNOTIFY_THRESHOLD_FORCE_FOUND,&lt;/li&gt;
     * &lt;li&gt;PxPairFlag::eNOTIFY_THRESHOLD_FORCE_PERSISTS,&lt;/li&gt;
     * &lt;li&gt;PxPairFlag::eNOTIFY_THRESHOLD_FORCE_LOST&lt;/li&gt;
     * &lt;/ul&gt;
     * <p>
     * See the documentation of #PxPairFlag for an explanation of each.
     * <p>
     * <b>Note:</b> eNOTIFY_TOUCH_CCD can get raised even if the pair did not request this event. However, in such a case it will only get
     * raised in combination with one of the other flags to point out that the other event occured during a CCD pass.
     */
    public PxPairFlags getEvents() {
        checkNotNull();
        return PxPairFlags.wrapPointer(_getEvents(address));
    }
    private static native long _getEvents(long address);

    /**
     * Flags raised due to the contact.
     * <p>
     * The events field is a combination of:
     * <p>
     * &lt;ul&gt;
     * &lt;li&gt;PxPairFlag::eNOTIFY_TOUCH_FOUND,&lt;/li&gt;
     * &lt;li&gt;PxPairFlag::eNOTIFY_TOUCH_PERSISTS,&lt;/li&gt;
     * &lt;li&gt;PxPairFlag::eNOTIFY_TOUCH_LOST,&lt;/li&gt;
     * &lt;li&gt;PxPairFlag::eNOTIFY_TOUCH_CCD,&lt;/li&gt;
     * &lt;li&gt;PxPairFlag::eNOTIFY_THRESHOLD_FORCE_FOUND,&lt;/li&gt;
     * &lt;li&gt;PxPairFlag::eNOTIFY_THRESHOLD_FORCE_PERSISTS,&lt;/li&gt;
     * &lt;li&gt;PxPairFlag::eNOTIFY_THRESHOLD_FORCE_LOST&lt;/li&gt;
     * &lt;/ul&gt;
     * <p>
     * See the documentation of #PxPairFlag for an explanation of each.
     * <p>
     * <b>Note:</b> eNOTIFY_TOUCH_CCD can get raised even if the pair did not request this event. However, in such a case it will only get
     * raised in combination with one of the other flags to point out that the other event occured during a CCD pass.
     */
    public void setEvents(PxPairFlags value) {
        checkNotNull();
        _setEvents(address, value.getAddress());
    }
    private static native void _setEvents(long address, long value);

    // Functions

    /**
     * Extracts the contact points from the stream and stores them in a convenient format.
     * @param userBuffer Array of PxContactPairPoint structures to extract the contact points to. The number of contacts for a pair is defined by #contactCount
     * @param bufferSize Number of PxContactPairPoint structures the provided buffer can store.
     * @return Number of contact points written to the buffer.
     * @see PxContactPairPoint
     */
    public int extractContacts(PxContactPairPoint userBuffer, int bufferSize) {
        checkNotNull();
        return _extractContacts(address, userBuffer.getAddress(), bufferSize);
    }
    private static native int _extractContacts(long address, long userBuffer, int bufferSize);

}
