package physx.physics;

/**
 * Flags which control the behavior of a material.
 * <p>
 * <b>See also:</b> PxMaterial 
 */
public enum PxMaterialFlagEnum {

    /**
     * If this flag is set, friction computations are always skipped between shapes with this material and any other shape.
     */
    eDISABLE_FRICTION(geteDISABLE_FRICTION()),
    /**
     * Whether to use strong friction.
     * The difference between "normal" and "strong" friction is that the strong friction feature
     * remembers the "friction error" between simulation steps. The friction is a force trying to
     * hold objects in place (or slow them down) and this is handled in the solver. But since the
     * solver is only an approximation, the result of the friction calculation can include a small
     * "error" - e.g. a box resting on a slope should not move at all if the static friction is in
     * action, but could slowly glide down the slope because of a small friction error in each 
     * simulation step. The strong friction counter-acts this by remembering the small error and
     * taking it to account during the next simulation step.
     * <p>
     * However, in some cases the strong friction could cause problems, and this is why it is
     * possible to disable the strong friction feature by setting this flag. One example is
     * raycast vehicles that are sliding fast across the surface, but still need a precise
     * steering behavior. It may be a good idea to reenable the strong friction when objects
     * are coming to a rest, to prevent them from slowly creeping down inclines.
     * <p>
     * Note: This flag only has an effect if the PxMaterialFlag::eDISABLE_FRICTION bit is 0.
     */
    eDISABLE_STRONG_FRICTION(geteDISABLE_STRONG_FRICTION()),
    /**
     * Whether to use the patch friction model.
     * This flag only has an effect if PxFrictionType::ePATCH friction model is used.
     * <p>
     * When using the patch friction model, up to 2 friction anchors are generated per patch. As the number of friction anchors
     * can be smaller than the number of contacts, the normal force is accumulated over all contacts and used to compute friction
     * for all anchors. Where there are more than 2 anchors, this can produce frictional behavior that is too strong (approximately 2x as strong
     * as analytical models suggest). 
     * <p>
     * This flag causes the normal force to be distributed between the friction anchors such that the total amount of friction applied does not 
     * exceed the analytical results.
     */
    eIMPROVED_PATCH_FRICTION(geteIMPROVED_PATCH_FRICTION());
    public final int value;
    
    PxMaterialFlagEnum(int value) {
        this.value = value;
    }

    private static native int _geteDISABLE_FRICTION();
    private static int geteDISABLE_FRICTION() {
        de.fabmax.physxjni.Loader.load();
        return _geteDISABLE_FRICTION();
    }

    private static native int _geteDISABLE_STRONG_FRICTION();
    private static int geteDISABLE_STRONG_FRICTION() {
        de.fabmax.physxjni.Loader.load();
        return _geteDISABLE_STRONG_FRICTION();
    }

    private static native int _geteIMPROVED_PATCH_FRICTION();
    private static int geteIMPROVED_PATCH_FRICTION() {
        de.fabmax.physxjni.Loader.load();
        return _geteIMPROVED_PATCH_FRICTION();
    }

    public static PxMaterialFlagEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxMaterialFlagEnum: " + value);
    }

}
