package physx.physics;

import physx.common.PxTransform;
import physx.common.PxVec3;

/**
 * PxRigidDynamic represents a dynamic rigid simulation object in the physics SDK.
 * <p>
 * <h3>Creation</h3>
 * Instances of this class are created by calling #PxPhysics::createRigidDynamic() and deleted with #release().
 * <p>
 * <h3>Visualizations</h3>
 * \li #PxVisualizationParameter::eACTOR_AXES
 * \li #PxVisualizationParameter::eBODY_AXES
 * \li #PxVisualizationParameter::eBODY_MASS_AXES
 * \li #PxVisualizationParameter::eBODY_LIN_VELOCITY
 * \li #PxVisualizationParameter::eBODY_ANG_VELOCITY
 * @see PxRigidBody
 * @see PxPhysics#createRigidDynamic
 */
public class PxRigidDynamic extends PxRigidBody {

    protected PxRigidDynamic() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxRigidDynamic wrapPointer(long address) {
        return address != 0L ? new PxRigidDynamic(address) : null;
    }
    
    public static PxRigidDynamic arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxRigidDynamic(long address) {
        super(address);
    }

    // Functions

    /**
     * Moves kinematically controlled dynamic actors through the game world.
     * <p>
     * You set a dynamic actor to be kinematic using the PxRigidBodyFlag::eKINEMATIC flag
     * with setRigidBodyFlag().
     * <p>
     * The move command will result in a velocity that will move the body into 
     * the desired pose. After the move is carried out during a single time step, 
     * the velocity is returned to zero. Thus, you must continuously call 
     * this in every time step for kinematic actors so that they move along a path.
     * <p>
     * This function simply stores the move destination until the next simulation
     * step is processed, so consecutive calls will simply overwrite the stored target variable.
     * <p>
     * The motion is always fully carried out. 
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already or if PxActorFlag::eDISABLE_SIMULATION is set.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping and will set the wake counter to #PxSceneDesc::wakeCounterResetValue.
     * @param destination The desired pose for the kinematic actor, in the global frame. <b>Range:</b> rigid body transform.
     * @see #getKinematicTarget
     */
    public void setKinematicTarget(PxTransform destination) {
        checkNotNull();
        _setKinematicTarget(address, destination.getAddress());
    }
    private static native void _setKinematicTarget(long address, long destination);

    /**
     * Get target pose of a kinematically controlled dynamic actor.
     * @param target Transform to write the target pose to. Only valid if the method returns true.
     * @return True if the actor is a kinematically controlled dynamic and the target has been set, else False.
     * @see #setKinematicTarget
     */
    public boolean getKinematicTarget(PxTransform target) {
        checkNotNull();
        return _getKinematicTarget(address, target.getAddress());
    }
    private static native boolean _getKinematicTarget(long address, long target);

    /**
     * Returns true if this body is sleeping.
     * <p>
     * When an actor does not move for a period of time, it is no longer simulated in order to save time. This state
     * is called sleeping. However, because the object automatically wakes up when it is either touched by an awake object,
     * or one of its properties is changed by the user, the entire sleep mechanism should be transparent to the user.
     * <p>
     * In general, a dynamic rigid actor is guaranteed to be awake if at least one of the following holds:
     * <p>
     * \li The wake counter is positive (see #setWakeCounter()).
     * \li The linear or angular velocity is non-zero.
     * \li A non-zero force or torque has been applied.
     * <p>
     * If a dynamic rigid actor is sleeping, the following state is guaranteed:
     * <p>
     * \li The wake counter is zero.
     * \li The linear and angular velocity is zero.
     * \li There is no force update pending.
     * <p>
     * When an actor gets inserted into a scene, it will be considered asleep if all the points above hold, else it will be treated as awake.
     * <p>
     * If an actor is asleep after the call to PxScene::fetchResults() returns, it is guaranteed that the pose of the actor 
     * was not changed. You can use this information to avoid updating the transforms of associated objects.
     * <p>
     * <b>Note:</b> A kinematic actor is asleep unless a target pose has been set (in which case it will stay awake until two consecutive 
     * simulation steps without a target pose being set have passed). The wake counter will get set to zero or to the reset value 
     * #PxSceneDesc::wakeCounterResetValue in the case where a target pose has been set to be consistent with the definitions above.
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already.
     * <p>
     * <b>Note:</b> It is not allowed to use this method while the simulation is running.
     * @return True if the actor is sleeping.
     * @see #isSleeping
     * @see #wakeUp
     * @see #putToSleep
     * @see #getSleepThreshold
     */
    public boolean isSleeping() {
        checkNotNull();
        return _isSleeping(address);
    }
    private static native boolean _isSleeping(long address);

    /**
     * Sets the mass-normalized kinetic energy threshold below which an actor may go to sleep.
     * <p>
     * Actors whose kinetic energy divided by their mass is below this threshold will be candidates for sleeping.
     * <p>
     * <b>Default:</b> 5e-5f * PxTolerancesScale::speed * PxTolerancesScale::speed
     * @param threshold Energy below which an actor may go to sleep. <b>Range:</b> [0, PX_MAX_F32)
     * @see #isSleeping
     * @see #getSleepThreshold
     * @see #wakeUp
     * @see #putToSleep
     * @see physx.common.PxTolerancesScale
     */
    public void setSleepThreshold(float threshold) {
        checkNotNull();
        _setSleepThreshold(address, threshold);
    }
    private static native void _setSleepThreshold(long address, float threshold);

    /**
     * Returns the mass-normalized kinetic energy below which an actor may go to sleep.
     * @return The energy threshold for sleeping.
     * @see #isSleeping
     * @see #wakeUp
     * @see #putToSleep
     * @see #setSleepThreshold
     */
    public float getSleepThreshold() {
        checkNotNull();
        return _getSleepThreshold(address);
    }
    private static native float _getSleepThreshold(long address);

    /**
     * Sets the mass-normalized kinetic energy threshold below which an actor may participate in stabilization.
     * <p>
     * Actors whose kinetic energy divided by their mass is above this threshold will not participate in stabilization.
     * <p>
     * This value has no effect if PxSceneFlag::eENABLE_STABILIZATION was not enabled on the PxSceneDesc.
     * <p>
     * <b>Default:</b> 1e-5f * PxTolerancesScale::speed * PxTolerancesScale::speed
     * @param threshold Energy below which an actor may participate in stabilization. <b>Range:</b> [0,inf)
     * @see #getStabilizationThreshold
     */
    public void setStabilizationThreshold(float threshold) {
        checkNotNull();
        _setStabilizationThreshold(address, threshold);
    }
    private static native void _setStabilizationThreshold(long address, float threshold);

    /**
     * Returns the mass-normalized kinetic energy below which an actor may participate in stabilization.
     * <p>
     * Actors whose kinetic energy divided by their mass is above this threshold will not participate in stabilization. 
     * @return The energy threshold for participating in stabilization.
     * @see #setStabilizationThreshold
     */
    public float getStabilizationThreshold() {
        checkNotNull();
        return _getStabilizationThreshold(address);
    }
    private static native float _getStabilizationThreshold(long address);

    /**
     * Reads the PxRigidDynamic lock flags.
     * <p>
     * See the list of flags #PxRigidDynamicLockFlag
     * @return The values of the PxRigidDynamicLock flags.
     * @see #setRigidDynamicLockFlag
     */
    public PxRigidDynamicLockFlags getRigidDynamicLockFlags() {
        checkNotNull();
        return PxRigidDynamicLockFlags.wrapPointer(_getRigidDynamicLockFlags(address));
    }
    private static native long _getRigidDynamicLockFlags(long address);

    /**
     * Raises or clears a particular rigid dynamic lock flag.
     * <p>
     * See the list of flags #PxRigidDynamicLockFlag
     * <p>
     * <b>Default:</b> no flags are set
     * @param flag  The PxRigidDynamicLockBody flag to raise(set) or clear. See #PxRigidBodyFlag.
     * @param value The new boolean value for the flag.
     * @see #getRigidDynamicLockFlags
     */
    public void setRigidDynamicLockFlag(PxRigidDynamicLockFlagEnum flag, boolean value) {
        checkNotNull();
        _setRigidDynamicLockFlag(address, flag.value, value);
    }
    private static native void _setRigidDynamicLockFlag(long address, int flag, boolean value);

    /**
     * @param flags WebIDL type: {@link PxRigidDynamicLockFlags} [Ref]
     */
    public void setRigidDynamicLockFlags(PxRigidDynamicLockFlags flags) {
        checkNotNull();
        _setRigidDynamicLockFlags(address, flags.getAddress());
    }
    private static native void _setRigidDynamicLockFlags(long address, long flags);

    /**
     * Sets the linear velocity of the actor.
     * <p>
     * Note that if you continuously set the velocity of an actor yourself,
     * forces such as gravity or friction will not be able to manifest themselves, because forces directly
     * influence only the velocity/momentum of an actor.
     * <p>
     * <b>Default:</b> (0.0, 0.0, 0.0)
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping, and the autowake parameter is true (default) or the
     * new velocity is non-zero.
     * <p>
     * <b>Note:</b> It is invalid to use this method if PxActorFlag::eDISABLE_SIMULATION is set.
     * @param linVel New linear velocity of actor. <b>Range:</b> velocity vector
     * smaller than #PxSceneDesc::wakeCounterResetValue it will get increased to the reset value.
     * @see #setAngularVelocity
     */
    public void setLinearVelocity(PxVec3 linVel) {
        checkNotNull();
        _setLinearVelocity(address, linVel.getAddress());
    }
    private static native void _setLinearVelocity(long address, long linVel);

    /**
     * Sets the linear velocity of the actor.
     * <p>
     * Note that if you continuously set the velocity of an actor yourself,
     * forces such as gravity or friction will not be able to manifest themselves, because forces directly
     * influence only the velocity/momentum of an actor.
     * <p>
     * <b>Default:</b> (0.0, 0.0, 0.0)
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping, and the autowake parameter is true (default) or the
     * new velocity is non-zero.
     * <p>
     * <b>Note:</b> It is invalid to use this method if PxActorFlag::eDISABLE_SIMULATION is set.
     * @param linVel New linear velocity of actor. <b>Range:</b> velocity vector
     * @param autowake Whether to wake the object up if it is asleep. If true and the current wake counter value is
     * smaller than #PxSceneDesc::wakeCounterResetValue it will get increased to the reset value.
     * @see #setAngularVelocity
     */
    public void setLinearVelocity(PxVec3 linVel, boolean autowake) {
        checkNotNull();
        _setLinearVelocity(address, linVel.getAddress(), autowake);
    }
    private static native void _setLinearVelocity(long address, long linVel, boolean autowake);

    /**
     * Sets the angular velocity of the actor.
     * <p>
     * Note that if you continuously set the angular velocity of an actor yourself,
     * forces such as friction will not be able to rotate the actor, because forces directly influence only the velocity/momentum.
     * <p>
     * <b>Default:</b> (0.0, 0.0, 0.0)
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping, and the autowake parameter is true (default) or the
     * new velocity is non-zero.
     * <p>
     * <b>Note:</b> It is invalid to use this method if PxActorFlag::eDISABLE_SIMULATION is set.
     * @param angVel New angular velocity of actor. <b>Range:</b> angular velocity vector
     * smaller than #PxSceneDesc::wakeCounterResetValue it will get increased to the reset value.
     * @see #setLinearVelocity
     */
    public void setAngularVelocity(PxVec3 angVel) {
        checkNotNull();
        _setAngularVelocity(address, angVel.getAddress());
    }
    private static native void _setAngularVelocity(long address, long angVel);

    /**
     * Sets the angular velocity of the actor.
     * <p>
     * Note that if you continuously set the angular velocity of an actor yourself,
     * forces such as friction will not be able to rotate the actor, because forces directly influence only the velocity/momentum.
     * <p>
     * <b>Default:</b> (0.0, 0.0, 0.0)
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping, and the autowake parameter is true (default) or the
     * new velocity is non-zero.
     * <p>
     * <b>Note:</b> It is invalid to use this method if PxActorFlag::eDISABLE_SIMULATION is set.
     * @param angVel New angular velocity of actor. <b>Range:</b> angular velocity vector
     * @param autowake Whether to wake the object up if it is asleep. If true and the current wake counter value is
     * smaller than #PxSceneDesc::wakeCounterResetValue it will get increased to the reset value.
     * @see #setLinearVelocity
     */
    public void setAngularVelocity(PxVec3 angVel, boolean autowake) {
        checkNotNull();
        _setAngularVelocity(address, angVel.getAddress(), autowake);
    }
    private static native void _setAngularVelocity(long address, long angVel, boolean autowake);

    /**
     * Sets the wake counter for the actor.
     * <p>
     * The wake counter value determines the minimum amount of time until the body can be put to sleep. Please note
     * that a body will not be put to sleep if the energy is above the specified threshold (see #setSleepThreshold())
     * or if other awake bodies are touching it.
     * <p>
     * <b>Note:</b> Passing in a positive value will wake the actor up automatically.
     * <p>
     * <b>Note:</b> It is invalid to use this method for kinematic actors since the wake counter for kinematics is defined
     * based on whether a target pose has been set (see the comment in #isSleeping()).
     * <p>
     * <b>Note:</b> It is invalid to use this method if PxActorFlag::eDISABLE_SIMULATION is set.
     * <p>
     * <b>Default:</b> 0.4 (which corresponds to 20 frames for a time step of 0.02)
     * @param wakeCounterValue Wake counter value. <b>Range:</b> [0, PX_MAX_F32)
     * @see #isSleeping
     * @see #getWakeCounter
     */
    public void setWakeCounter(float wakeCounterValue) {
        checkNotNull();
        _setWakeCounter(address, wakeCounterValue);
    }
    private static native void _setWakeCounter(long address, float wakeCounterValue);

    /**
     * Returns the wake counter of the actor.
     * <p>
     * <b>Note:</b> It is not allowed to use this method while the simulation is running.
     * @return The wake counter of the actor.
     * @see #isSleeping
     * @see #setWakeCounter
     */
    public float getWakeCounter() {
        checkNotNull();
        return _getWakeCounter(address);
    }
    private static native float _getWakeCounter(long address);

    /**
     * Wakes up the actor if it is sleeping.
     * <p>
     * The actor will get woken up and might cause other touching actors to wake up as well during the next simulation step.
     * <p>
     * <b>Note:</b> This will set the wake counter of the actor to the value specified in #PxSceneDesc::wakeCounterResetValue.
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already or if PxActorFlag::eDISABLE_SIMULATION is set.
     * <p>
     * <b>Note:</b> It is invalid to use this method for kinematic actors since the sleep state for kinematics is defined
     * based on whether a target pose has been set (see the comment in #isSleeping()).
     * @see #isSleeping
     * @see #putToSleep
     */
    public void wakeUp() {
        checkNotNull();
        _wakeUp(address);
    }
    private static native void _wakeUp(long address);

    /**
     * Forces the actor to sleep. 
     * <p>
     * The actor will stay asleep during the next simulation step if not touched by another non-sleeping actor.
     * <p>
     * <b>Note:</b> Any applied force will be cleared and the velocity and the wake counter of the actor will be set to 0.
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already or if PxActorFlag::eDISABLE_SIMULATION is set.
     * <p>
     * <b>Note:</b> It is invalid to use this method for kinematic actors since the sleep state for kinematics is defined
     * based on whether a target pose has been set (see the comment in #isSleeping()).
     * @see #isSleeping
     * @see #wakeUp
     */
    public void putToSleep() {
        checkNotNull();
        _putToSleep(address);
    }
    private static native void _putToSleep(long address);

    /**
     * Sets the solver iteration counts for the body. 
     * <p>
     * The solver iteration count determines how accurately joints and contacts are resolved. 
     * If you are having trouble with jointed bodies oscillating and behaving erratically, then
     * setting a higher position iteration count may improve their stability.
     * <p>
     * If intersecting bodies are being depenetrated too violently, increase the number of velocity 
     * iterations. More velocity iterations will drive the relative exit velocity of the intersecting 
     * objects closer to the correct value given the restitution.
     * <p>
     * <b>Default:</b> 4 position iterations, 1 velocity iteration
     * @param minPositionIters Number of position iterations the solver should perform for this body. <b>Range:</b> [1,255]
     */
    public void setSolverIterationCounts(int minPositionIters) {
        checkNotNull();
        _setSolverIterationCounts(address, minPositionIters);
    }
    private static native void _setSolverIterationCounts(long address, int minPositionIters);

    /**
     * Sets the solver iteration counts for the body. 
     * <p>
     * The solver iteration count determines how accurately joints and contacts are resolved. 
     * If you are having trouble with jointed bodies oscillating and behaving erratically, then
     * setting a higher position iteration count may improve their stability.
     * <p>
     * If intersecting bodies are being depenetrated too violently, increase the number of velocity 
     * iterations. More velocity iterations will drive the relative exit velocity of the intersecting 
     * objects closer to the correct value given the restitution.
     * <p>
     * <b>Default:</b> 4 position iterations, 1 velocity iteration
     * @param minPositionIters Number of position iterations the solver should perform for this body. <b>Range:</b> [1,255]
     * @param minVelocityIters Number of velocity iterations the solver should perform for this body. <b>Range:</b> [0,255]
     */
    public void setSolverIterationCounts(int minPositionIters, int minVelocityIters) {
        checkNotNull();
        _setSolverIterationCounts(address, minPositionIters, minVelocityIters);
    }
    private static native void _setSolverIterationCounts(long address, int minPositionIters, int minVelocityIters);

    /**
     * Retrieves the force threshold for contact reports.
     * <p>
     * The contact report threshold is a force threshold. If the force between 
     * two actors exceeds this threshold for either of the two actors, a contact report 
     * will be generated according to the contact report threshold flags provided by
     * the filter shader/callback.
     * See #PxPairFlag.
     * <p>
     * The threshold used for a collision between a dynamic actor and the static environment is 
     *    the threshold of the dynamic actor, and all contacts with static actors are summed to find 
     *    the total normal force.
     * <p>
     * <b>Default:</b> PX_MAX_F32
     * @return Force threshold for contact reports.
     * @see #setContactReportThreshold
     * @see PxSimulationFilterShader
     */
    public float getContactReportThreshold() {
        checkNotNull();
        return _getContactReportThreshold(address);
    }
    private static native float _getContactReportThreshold(long address);

    /**
     * Sets the force threshold for contact reports.
     * <p>
     * See #getContactReportThreshold().
     * @param threshold Force threshold for contact reports. <b>Range:</b> [0, PX_MAX_F32)
     * @see #getContactReportThreshold
     */
    public void setContactReportThreshold(float threshold) {
        checkNotNull();
        _setContactReportThreshold(address, threshold);
    }
    private static native void _setContactReportThreshold(long address, float threshold);

}
