package physx.vehicle2;

import physx.NativeObject;

/**
 * The clutch connects two plates together. One plate rotates with the speed of the engine. The second plate rotates with a weighted average of all
 * wheels connected to the differential after accounting for the gear ratio. The difference in rotation speeds generates a restoring torque applied to engine and wheels
 * that aims to reduce the difference in rotational speed. The restoring torque is proportional to the clutch strength and the clutch pedal position.
 */
public class PxVehicleClutchParams extends NativeObject {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxVehicleClutchParams wrapPointer(long address) {
        return address != 0L ? new PxVehicleClutchParams(address) : null;
    }
    
    public static PxVehicleClutchParams arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxVehicleClutchParams(long address) {
        super(address);
    }

    // Placed Constructors

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @return Stack allocated object of PxVehicleClutchParams
     */
    public static PxVehicleClutchParams createAt(long address) {
        __placement_new_PxVehicleClutchParams(address);
        PxVehicleClutchParams createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @return Stack allocated object of PxVehicleClutchParams
     */
    public static <T> PxVehicleClutchParams createAt(T allocator, Allocator<T> allocate) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxVehicleClutchParams(address);
        PxVehicleClutchParams createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxVehicleClutchParams(long address);

    // Constructors

    public PxVehicleClutchParams() {
        address = _PxVehicleClutchParams();
    }
    private static native long _PxVehicleClutchParams();

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * The engine and wheel rotation speeds that are coupled through the clutch can be updated by choosing
     * one of two modes: eESTIMATE and eBEST_POSSIBLE.
     * <p>
     * <b>Note:</b> If eESTIMATE is chosen the vehicle sdk will update the wheel and engine rotation speeds
     * with estimated values to the implemented clutch model.
     * <p>
     * <b>Note:</b> If eBEST_POSSIBLE is chosen the vehicle sdk will compute the best possible
     * solution (within floating point tolerance) to the implemented clutch model.
     * This is the recommended mode.
     * <p>
     * <b>Note:</b> The clutch model remains the same if either eESTIMATE or eBEST_POSSIBLE is chosen but the accuracy and
     * computational cost of the solution to the model can be tuned as required.
     */
    public PxVehicleClutchAccuracyModeEnum getAccuracyMode() {
        checkNotNull();
        return PxVehicleClutchAccuracyModeEnum.forValue(_getAccuracyMode(address));
    }
    private static native int _getAccuracyMode(long address);

    /**
     * The engine and wheel rotation speeds that are coupled through the clutch can be updated by choosing
     * one of two modes: eESTIMATE and eBEST_POSSIBLE.
     * <p>
     * <b>Note:</b> If eESTIMATE is chosen the vehicle sdk will update the wheel and engine rotation speeds
     * with estimated values to the implemented clutch model.
     * <p>
     * <b>Note:</b> If eBEST_POSSIBLE is chosen the vehicle sdk will compute the best possible
     * solution (within floating point tolerance) to the implemented clutch model.
     * This is the recommended mode.
     * <p>
     * <b>Note:</b> The clutch model remains the same if either eESTIMATE or eBEST_POSSIBLE is chosen but the accuracy and
     * computational cost of the solution to the model can be tuned as required.
     */
    public void setAccuracyMode(PxVehicleClutchAccuracyModeEnum value) {
        checkNotNull();
        _setAccuracyMode(address, value.value);
    }
    private static native void _setAccuracyMode(long address, int value);

    /**
     * Tune the mathematical accuracy and computational cost of the computed estimate to the wheel and
     * engine rotation speeds if eESTIMATE is chosen.
     * <p>
     * <b>Note:</b> As estimateIterations increases the computational cost of the clutch also increases and the solution
     * approaches the solution that would be computed if eBEST_POSSIBLE was chosen instead.
     * <p>
     * <b>Note:</b> This has no effect if eBEST_POSSIBLE is chosen as the accuracy mode.
     * <p>
     * <b>Note:</b> A value of zero is not allowed if eESTIMATE is chosen as the accuracy mode.
     */
    public int getEstimateIterations() {
        checkNotNull();
        return _getEstimateIterations(address);
    }
    private static native int _getEstimateIterations(long address);

    /**
     * Tune the mathematical accuracy and computational cost of the computed estimate to the wheel and
     * engine rotation speeds if eESTIMATE is chosen.
     * <p>
     * <b>Note:</b> As estimateIterations increases the computational cost of the clutch also increases and the solution
     * approaches the solution that would be computed if eBEST_POSSIBLE was chosen instead.
     * <p>
     * <b>Note:</b> This has no effect if eBEST_POSSIBLE is chosen as the accuracy mode.
     * <p>
     * <b>Note:</b> A value of zero is not allowed if eESTIMATE is chosen as the accuracy mode.
     */
    public void setEstimateIterations(int value) {
        checkNotNull();
        _setEstimateIterations(address, value);
    }
    private static native void _setEstimateIterations(long address, int value);

    // Functions

    /**
     * @param srcFrame WebIDL type: {@link PxVehicleFrame} [Const, Ref]
     * @param trgFrame WebIDL type: {@link PxVehicleFrame} [Const, Ref]
     * @param srcScale WebIDL type: {@link PxVehicleScale} [Const, Ref]
     * @param trgScale WebIDL type: {@link PxVehicleScale} [Const, Ref]
     * @return WebIDL type: {@link PxVehicleClutchParams} [Value]
     */
    public PxVehicleClutchParams transformAndScale(PxVehicleFrame srcFrame, PxVehicleFrame trgFrame, PxVehicleScale srcScale, PxVehicleScale trgScale) {
        checkNotNull();
        return PxVehicleClutchParams.wrapPointer(_transformAndScale(address, srcFrame.getAddress(), trgFrame.getAddress(), srcScale.getAddress(), trgScale.getAddress()));
    }
    private static native long _transformAndScale(long address, long srcFrame, long trgFrame, long srcScale, long trgScale);

    /**
     * @return WebIDL type: boolean
     */
    public boolean isValid() {
        checkNotNull();
        return _isValid(address);
    }
    private static native boolean _isValid(long address);

}
