package physx.common;

import physx.NativeObject;

/**
 * User defined interface class.  Used by the library to emit debug information.
 * <p>
 * <b>Note:</b> The SDK state should not be modified from within any error reporting functions.
 * <p>
 * <b>Threading:</b> The SDK sequences its calls to the output stream using a mutex, so the class need not
 * be implemented in a thread-safe manner if the SDK is the only client.
 */
public class PxErrorCallback extends NativeObject {

    protected PxErrorCallback() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxErrorCallback wrapPointer(long address) {
        return address != 0L ? new PxErrorCallback(address) : null;
    }
    
    public static PxErrorCallback arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxErrorCallback(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Reports an error code.
     * \param code Error code, see #PxErrorCode
     * \param message Message to display.
     * \param file File error occured in.
     * \param line Line number error occured on.
     */
    public void reportError(PxErrorCodeEnum code, String message, String file, int line) {
        checkNotNull();
        _reportError(address, code.value, message, file, line);
    }
    private static native void _reportError(long address, int code, String message, String file, int line);

}
