package physx.geometry;

import physx.NativeObject;
import physx.common.PxBoundedData;

/**
 * A structure describing a triangle mesh.
 */
public class PxSimpleTriangleMesh extends NativeObject {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxSimpleTriangleMesh wrapPointer(long address) {
        return address != 0L ? new PxSimpleTriangleMesh(address) : null;
    }
    
    public static PxSimpleTriangleMesh arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxSimpleTriangleMesh(long address) {
        super(address);
    }

    // Constructors

    /**
     * constructor sets to default.
     */
    public PxSimpleTriangleMesh() {
        address = _PxSimpleTriangleMesh();
    }
    private static native long _PxSimpleTriangleMesh();

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * Pointer to first vertex point.
     */
    public PxBoundedData getPoints() {
        checkNotNull();
        return PxBoundedData.wrapPointer(_getPoints(address));
    }
    private static native long _getPoints(long address);

    /**
     * Pointer to first vertex point.
     */
    public void setPoints(PxBoundedData value) {
        checkNotNull();
        _setPoints(address, value.getAddress());
    }
    private static native void _setPoints(long address, long value);

    /**
     * Pointer to first triangle.
     * <p>
     * Caller may add triangleStrideBytes bytes to the pointer to access the next triangle.
     * <p>
     * These are triplets of 0 based indices:
     * vert0 vert1 vert2
     * vert0 vert1 vert2
     * vert0 vert1 vert2
     * ...
     * <p>
     * where vertex is either a 32 or 16 bit unsigned integer. There are numTriangles*3 indices.
     * <p>
     * This is declared as a void pointer because it is actually either an PxU16 or a PxU32 pointer.
     */
    public PxBoundedData getTriangles() {
        checkNotNull();
        return PxBoundedData.wrapPointer(_getTriangles(address));
    }
    private static native long _getTriangles(long address);

    /**
     * Pointer to first triangle.
     * <p>
     * Caller may add triangleStrideBytes bytes to the pointer to access the next triangle.
     * <p>
     * These are triplets of 0 based indices:
     * vert0 vert1 vert2
     * vert0 vert1 vert2
     * vert0 vert1 vert2
     * ...
     * <p>
     * where vertex is either a 32 or 16 bit unsigned integer. There are numTriangles*3 indices.
     * <p>
     * This is declared as a void pointer because it is actually either an PxU16 or a PxU32 pointer.
     */
    public void setTriangles(PxBoundedData value) {
        checkNotNull();
        _setTriangles(address, value.getAddress());
    }
    private static native void _setTriangles(long address, long value);

    /**
     * Flags bits, combined from values of the enum ::PxMeshFlag
     */
    public PxMeshFlags getFlags() {
        checkNotNull();
        return PxMeshFlags.wrapPointer(_getFlags(address));
    }
    private static native long _getFlags(long address);

    /**
     * Flags bits, combined from values of the enum ::PxMeshFlag
     */
    public void setFlags(PxMeshFlags value) {
        checkNotNull();
        _setFlags(address, value.getAddress());
    }
    private static native void _setFlags(long address, long value);

    // Functions

    /**
     * (re)sets the structure to the default. 
     */
    public void setToDefault() {
        checkNotNull();
        _setToDefault(address);
    }
    private static native void _setToDefault(long address);

    /**
     * returns true if the current settings are valid
     */
    public boolean isValid() {
        checkNotNull();
        return _isValid(address);
    }
    private static native boolean _isValid(long address);

}
