package physx.physics;

/**
 * Broad phase algorithm used in the simulation
 * <p>
 * eSAP is a good generic choice with great performance when many objects are sleeping. Performance
 * can degrade significantly though, when all objects are moving, or when large numbers of objects
 * are added to or removed from the broad phase. This algorithm does not need world bounds to be
 * defined in order to work.
 * <p>
 * eMBP is an alternative broad phase algorithm that does not suffer from the same performance
 * issues as eSAP when all objects are moving or when inserting large numbers of objects. However
 * its generic performance when many objects are sleeping might be inferior to eSAP, and it requires
 * users to define world bounds in order to work.
 * <p>
 * eABP is a revisited implementation of MBP, which automatically manages broad-phase regions.
 * It offers the convenience of eSAP (no need to define world bounds or regions) and the performance
 * of eMBP when a lot of objects are moving. While eSAP can remain faster when most objects are
 * sleeping and eMBP can remain faster when it uses a large number of properly-defined regions,
 * eABP often gives the best performance on average and the best memory usage.
 * <p>
 * ePABP is a parallel implementation of ABP. It can often be the fastest (CPU) broadphase, but it
 * can use more memory than ABP.
 * <p>
 * eGPU is a GPU implementation of the incremental sweep and prune approach. Additionally, it uses a ABP-style
 * initial pair generation approach to avoid large spikes when inserting shapes. It not only has the advantage 
 * of traditional SAP approch which is good for when many objects are sleeping, but due to being fully parallel, 
 * it also is great when lots of shapes are moving or for runtime pair insertion and removal. It can become a 
 * performance bottleneck if there are a very large number of shapes roughly projecting to the same values
 * on a given axis. If the scene has a very large number of shapes in an actor, e.g. a humanoid, it is recommended
 * to use an aggregate to represent multi-shape or multi-body actors to minimize stress placed on the broad phase.
 */
public enum PxBroadPhaseTypeEnum {

    /**
     * 3-axes sweep-and-prune
     */
    eSAP(geteSAP()),
    /**
     * Multi box pruning
     */
    eMBP(geteMBP()),
    /**
     * Automatic box pruning
     */
    eABP(geteABP()),
    /**
     * Parallel automatic box pruning
     */
    ePABP(getePABP()),
    /**
     * GPU broad phase
     */
    eGPU(geteGPU());
    public final int value;
    
    PxBroadPhaseTypeEnum(int value) {
        this.value = value;
    }

    private static native int _geteSAP();
    private static int geteSAP() {
        de.fabmax.physxjni.Loader.load();
        return _geteSAP();
    }

    private static native int _geteMBP();
    private static int geteMBP() {
        de.fabmax.physxjni.Loader.load();
        return _geteMBP();
    }

    private static native int _geteABP();
    private static int geteABP() {
        de.fabmax.physxjni.Loader.load();
        return _geteABP();
    }

    private static native int _getePABP();
    private static int getePABP() {
        de.fabmax.physxjni.Loader.load();
        return _getePABP();
    }

    private static native int _geteGPU();
    private static int geteGPU() {
        de.fabmax.physxjni.Loader.load();
        return _geteGPU();
    }

    public static PxBroadPhaseTypeEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxBroadPhaseTypeEnum: " + value);
    }

}
