package physx.physics;

import physx.NativeObject;
import physx.PlatformChecks;
import physx.common.PxCudaContextManager;
import physx.common.PxFoundation;
import physx.common.PxInsertionCallback;
import physx.common.PxTolerancesScale;
import physx.common.PxTransform;
import physx.geometry.PxGeometry;
import physx.particles.PxPBDMaterial;
import physx.particles.PxPBDParticleSystem;
import physx.particles.PxParticleBuffer;
import physx.particles.PxParticleClothBuffer;

public class PxPhysics extends NativeObject {

    protected PxPhysics() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxPhysics wrapPointer(long address) {
        return address != 0L ? new PxPhysics(address) : null;
    }
    
    public static PxPhysics arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxPhysics(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    public void release() {
        checkNotNull();
        _release(address);
    }
    private static native void _release(long address);

    /**
     * @return WebIDL type: {@link PxFoundation} [Ref]
     */
    public PxFoundation getFoundation() {
        checkNotNull();
        return PxFoundation.wrapPointer(_getFoundation(address));
    }
    private static native long _getFoundation(long address);

    /**
     * @param maxActor            WebIDL type: unsigned long
     * @param maxShape            WebIDL type: unsigned long
     * @param enableSelfCollision WebIDL type: boolean
     * @return WebIDL type: {@link PxAggregate}
     */
    public PxAggregate createAggregate(int maxActor, int maxShape, boolean enableSelfCollision) {
        checkNotNull();
        return PxAggregate.wrapPointer(_createAggregate(address, maxActor, maxShape, enableSelfCollision));
    }
    private static native long _createAggregate(long address, int maxActor, int maxShape, boolean enableSelfCollision);

    /**
     * @return WebIDL type: {@link PxTolerancesScale} [Const, Ref]
     */
    public PxTolerancesScale getTolerancesScale() {
        checkNotNull();
        return PxTolerancesScale.wrapPointer(_getTolerancesScale(address));
    }
    private static native long _getTolerancesScale(long address);

    /**
     * @param sceneDesc WebIDL type: {@link PxSceneDesc} [Const, Ref]
     * @return WebIDL type: {@link PxScene}
     */
    public PxScene createScene(PxSceneDesc sceneDesc) {
        checkNotNull();
        return PxScene.wrapPointer(_createScene(address, sceneDesc.getAddress()));
    }
    private static native long _createScene(long address, long sceneDesc);

    /**
     * @param pose WebIDL type: {@link PxTransform} [Const, Ref]
     * @return WebIDL type: {@link PxRigidStatic}
     */
    public PxRigidStatic createRigidStatic(PxTransform pose) {
        checkNotNull();
        return PxRigidStatic.wrapPointer(_createRigidStatic(address, pose.getAddress()));
    }
    private static native long _createRigidStatic(long address, long pose);

    /**
     * @param pose WebIDL type: {@link PxTransform} [Const, Ref]
     * @return WebIDL type: {@link PxRigidDynamic}
     */
    public PxRigidDynamic createRigidDynamic(PxTransform pose) {
        checkNotNull();
        return PxRigidDynamic.wrapPointer(_createRigidDynamic(address, pose.getAddress()));
    }
    private static native long _createRigidDynamic(long address, long pose);

    /**
     * @param geometry WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param material WebIDL type: {@link PxMaterial} [Const, Ref]
     * @return WebIDL type: {@link PxShape}
     */
    public PxShape createShape(PxGeometry geometry, PxMaterial material) {
        checkNotNull();
        return PxShape.wrapPointer(_createShape(address, geometry.getAddress(), material.getAddress()));
    }
    private static native long _createShape(long address, long geometry, long material);

    /**
     * @param geometry    WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param material    WebIDL type: {@link PxMaterial} [Const, Ref]
     * @param isExclusive WebIDL type: boolean
     * @return WebIDL type: {@link PxShape}
     */
    public PxShape createShape(PxGeometry geometry, PxMaterial material, boolean isExclusive) {
        checkNotNull();
        return PxShape.wrapPointer(_createShape(address, geometry.getAddress(), material.getAddress(), isExclusive));
    }
    private static native long _createShape(long address, long geometry, long material, boolean isExclusive);

    /**
     * @param geometry    WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param material    WebIDL type: {@link PxMaterial} [Const, Ref]
     * @param isExclusive WebIDL type: boolean
     * @param shapeFlags  WebIDL type: {@link PxShapeFlags} [Ref]
     * @return WebIDL type: {@link PxShape}
     */
    public PxShape createShape(PxGeometry geometry, PxMaterial material, boolean isExclusive, PxShapeFlags shapeFlags) {
        checkNotNull();
        return PxShape.wrapPointer(_createShape(address, geometry.getAddress(), material.getAddress(), isExclusive, shapeFlags.getAddress()));
    }
    private static native long _createShape(long address, long geometry, long material, boolean isExclusive, long shapeFlags);

    /**
     * @return WebIDL type: long
     */
    public int getNbShapes() {
        checkNotNull();
        return _getNbShapes(address);
    }
    private static native int _getNbShapes(long address);

    /**
     * @return WebIDL type: {@link PxArticulationReducedCoordinate}
     */
    public PxArticulationReducedCoordinate createArticulationReducedCoordinate() {
        checkNotNull();
        return PxArticulationReducedCoordinate.wrapPointer(_createArticulationReducedCoordinate(address));
    }
    private static native long _createArticulationReducedCoordinate(long address);

    /**
     * @param staticFriction  WebIDL type: float
     * @param dynamicFriction WebIDL type: float
     * @param restitution     WebIDL type: float
     * @return WebIDL type: {@link PxMaterial}
     */
    public PxMaterial createMaterial(float staticFriction, float dynamicFriction, float restitution) {
        checkNotNull();
        return PxMaterial.wrapPointer(_createMaterial(address, staticFriction, dynamicFriction, restitution));
    }
    private static native long _createMaterial(long address, float staticFriction, float dynamicFriction, float restitution);

    /**
     * @return WebIDL type: {@link PxInsertionCallback} [Ref]
     */
    public PxInsertionCallback getPhysicsInsertionCallback() {
        checkNotNull();
        return PxInsertionCallback.wrapPointer(_getPhysicsInsertionCallback(address));
    }
    private static native long _getPhysicsInsertionCallback(long address);

    /**
     * @param cudaContextManager WebIDL type: {@link PxCudaContextManager} [Ref]
     * @return WebIDL type: {@link PxPBDParticleSystem} [Platforms=windows;linux]
     */
    public PxPBDParticleSystem createPBDParticleSystem(PxCudaContextManager cudaContextManager) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxPhysics");
        return PxPBDParticleSystem.wrapPointer(_createPBDParticleSystem(address, cudaContextManager.getAddress()));
    }
    private static native long _createPBDParticleSystem(long address, long cudaContextManager);

    /**
     * @param cudaContextManager WebIDL type: {@link PxCudaContextManager} [Ref]
     * @param maxNeighborhood    WebIDL type: unsigned long
     * @return WebIDL type: {@link PxPBDParticleSystem} [Platforms=windows;linux]
     */
    public PxPBDParticleSystem createPBDParticleSystem(PxCudaContextManager cudaContextManager, int maxNeighborhood) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxPhysics");
        return PxPBDParticleSystem.wrapPointer(_createPBDParticleSystem(address, cudaContextManager.getAddress(), maxNeighborhood));
    }
    private static native long _createPBDParticleSystem(long address, long cudaContextManager, int maxNeighborhood);

    /**
     * @param maxParticles       WebIDL type: unsigned long
     * @param maxVolumes         WebIDL type: unsigned long
     * @param cudaContextManager WebIDL type: {@link PxCudaContextManager}
     * @return WebIDL type: {@link PxParticleBuffer} [Platforms=windows;linux]
     */
    public PxParticleBuffer createParticleBuffer(int maxParticles, int maxVolumes, PxCudaContextManager cudaContextManager) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxPhysics");
        return PxParticleBuffer.wrapPointer(_createParticleBuffer(address, maxParticles, maxVolumes, cudaContextManager.getAddress()));
    }
    private static native long _createParticleBuffer(long address, int maxParticles, int maxVolumes, long cudaContextManager);

    /**
     * @param maxParticles       WebIDL type: unsigned long
     * @param maxNumVolumes      WebIDL type: unsigned long
     * @param maxNumCloths       WebIDL type: unsigned long
     * @param maxNumTriangles    WebIDL type: unsigned long
     * @param maxNumSprings      WebIDL type: unsigned long
     * @param cudaContextManager WebIDL type: {@link PxCudaContextManager}
     * @return WebIDL type: {@link PxParticleClothBuffer} [Platforms=windows;linux]
     */
    public PxParticleClothBuffer createParticleClothBuffer(int maxParticles, int maxNumVolumes, int maxNumCloths, int maxNumTriangles, int maxNumSprings, PxCudaContextManager cudaContextManager) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxPhysics");
        return PxParticleClothBuffer.wrapPointer(_createParticleClothBuffer(address, maxParticles, maxNumVolumes, maxNumCloths, maxNumTriangles, maxNumSprings, cudaContextManager.getAddress()));
    }
    private static native long _createParticleClothBuffer(long address, int maxParticles, int maxNumVolumes, int maxNumCloths, int maxNumTriangles, int maxNumSprings, long cudaContextManager);

    /**
     * @param friction             WebIDL type: float
     * @param damping              WebIDL type: float
     * @param adhesion             WebIDL type: float
     * @param viscosity            WebIDL type: float
     * @param vorticityConfinement WebIDL type: float
     * @param surfaceTension       WebIDL type: float
     * @param cohesion             WebIDL type: float
     * @param lift                 WebIDL type: float
     * @param drag                 WebIDL type: float
     * @return WebIDL type: {@link PxPBDMaterial} [Platforms=windows;linux]
     */
    public PxPBDMaterial createPBDMaterial(float friction, float damping, float adhesion, float viscosity, float vorticityConfinement, float surfaceTension, float cohesion, float lift, float drag) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxPhysics");
        return PxPBDMaterial.wrapPointer(_createPBDMaterial(address, friction, damping, adhesion, viscosity, vorticityConfinement, surfaceTension, cohesion, lift, drag));
    }
    private static native long _createPBDMaterial(long address, float friction, float damping, float adhesion, float viscosity, float vorticityConfinement, float surfaceTension, float cohesion, float lift, float drag);

    /**
     * @param friction             WebIDL type: float
     * @param damping              WebIDL type: float
     * @param adhesion             WebIDL type: float
     * @param viscosity            WebIDL type: float
     * @param vorticityConfinement WebIDL type: float
     * @param surfaceTension       WebIDL type: float
     * @param cohesion             WebIDL type: float
     * @param lift                 WebIDL type: float
     * @param drag                 WebIDL type: float
     * @param cflCoefficient       WebIDL type: float
     * @return WebIDL type: {@link PxPBDMaterial} [Platforms=windows;linux]
     */
    public PxPBDMaterial createPBDMaterial(float friction, float damping, float adhesion, float viscosity, float vorticityConfinement, float surfaceTension, float cohesion, float lift, float drag, float cflCoefficient) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxPhysics");
        return PxPBDMaterial.wrapPointer(_createPBDMaterial(address, friction, damping, adhesion, viscosity, vorticityConfinement, surfaceTension, cohesion, lift, drag, cflCoefficient));
    }
    private static native long _createPBDMaterial(long address, float friction, float damping, float adhesion, float viscosity, float vorticityConfinement, float surfaceTension, float cohesion, float lift, float drag, float cflCoefficient);

    /**
     * @param friction             WebIDL type: float
     * @param damping              WebIDL type: float
     * @param adhesion             WebIDL type: float
     * @param viscosity            WebIDL type: float
     * @param vorticityConfinement WebIDL type: float
     * @param surfaceTension       WebIDL type: float
     * @param cohesion             WebIDL type: float
     * @param lift                 WebIDL type: float
     * @param drag                 WebIDL type: float
     * @param cflCoefficient       WebIDL type: float
     * @param gravityScale         WebIDL type: float
     * @return WebIDL type: {@link PxPBDMaterial} [Platforms=windows;linux]
     */
    public PxPBDMaterial createPBDMaterial(float friction, float damping, float adhesion, float viscosity, float vorticityConfinement, float surfaceTension, float cohesion, float lift, float drag, float cflCoefficient, float gravityScale) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxPhysics");
        return PxPBDMaterial.wrapPointer(_createPBDMaterial(address, friction, damping, adhesion, viscosity, vorticityConfinement, surfaceTension, cohesion, lift, drag, cflCoefficient, gravityScale));
    }
    private static native long _createPBDMaterial(long address, float friction, float damping, float adhesion, float viscosity, float vorticityConfinement, float surfaceTension, float cohesion, float lift, float drag, float cflCoefficient, float gravityScale);

}
