package physx.extensions;

import physx.NativeObject;
import physx.common.PxBase;
import physx.common.PxTransform;
import physx.common.PxVec3;
import physx.physics.PxConstraint;
import physx.physics.PxConstraintFlagEnum;
import physx.physics.PxConstraintFlags;
import physx.physics.PxRigidActor;
import physx.physics.PxScene;

/**
 * a base interface providing common functionality for PhysX joints
 */
public class PxJoint extends PxBase {

    protected PxJoint() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxJoint wrapPointer(long address) {
        return address != 0L ? new PxJoint(address) : null;
    }
    
    public static PxJoint arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxJoint(long address) {
        super(address);
    }

    // Attributes

    /**
     * user can assign this to whatever, usually to create a 1:1 relationship with a user object.
     */
    public NativeObject getUserData() {
        checkNotNull();
        return NativeObject.wrapPointer(_getUserData(address));
    }
    private static native long _getUserData(long address);

    /**
     * user can assign this to whatever, usually to create a 1:1 relationship with a user object.
     */
    public void setUserData(NativeObject value) {
        checkNotNull();
        _setUserData(address, value.getAddress());
    }
    private static native void _setUserData(long address, long value);

    // Functions

    /**
     * Set the actors for this joint. 
     * <p>
     * An actor may be NULL to indicate the world frame. At most one of the actors may be NULL.
     * @param actor0 the first actor.
     * @param actor1 the second actor
     */
    public void setActors(PxRigidActor actor0, PxRigidActor actor1) {
        checkNotNull();
        _setActors(address, actor0.getAddress(), actor1.getAddress());
    }
    private static native void _setActors(long address, long actor0, long actor1);

    /**
     * Set the joint local pose for an actor. 
     * <p>
     * This is the relative pose which locates the joint frame relative to the actor.
     * @param actor 0 for the first actor, 1 for the second actor.
     * @param localPose the local pose for the actor this joint
     * @see #getLocalPose
     */
    public void setLocalPose(PxJointActorIndexEnum actor, PxTransform localPose) {
        checkNotNull();
        _setLocalPose(address, actor.value, localPose.getAddress());
    }
    private static native void _setLocalPose(long address, int actor, long localPose);

    /**
     * get the joint local pose for an actor. 
     * @param actor 0 for the first actor, 1 for the second actor.
     * <p>
     * return the local pose for this joint
     * @see #setLocalPose
     */
    public PxTransform getLocalPose(PxJointActorIndexEnum actor) {
        checkNotNull();
        return PxTransform.wrapPointer(_getLocalPose(address, actor.value));
    }
    private static native long _getLocalPose(long address, int actor);

    /**
     * get the relative pose for this joint
     * <p>
     * This function returns the pose of the joint frame of actor1 relative to actor0
     */
    public PxTransform getRelativeTransform() {
        checkNotNull();
        return PxTransform.wrapPointer(_getRelativeTransform(address));
    }
    private static native long _getRelativeTransform(long address);

    /**
     * get the relative linear velocity of the joint
     * <p>
     * This function returns the linear velocity of the origin of the constraint frame of actor1, relative to the origin of the constraint
     * frame of actor0. The value is returned in the constraint frame of actor0
     */
    public PxVec3 getRelativeLinearVelocity() {
        checkNotNull();
        return PxVec3.wrapPointer(_getRelativeLinearVelocity(address));
    }
    private static native long _getRelativeLinearVelocity(long address);

    /**
     * get the relative angular velocity of the joint
     * <p>
     * This function returns the angular velocity of actor1 relative to actor0. The value is returned in the constraint frame of actor0
     */
    public PxVec3 getRelativeAngularVelocity() {
        checkNotNull();
        return PxVec3.wrapPointer(_getRelativeAngularVelocity(address));
    }
    private static native long _getRelativeAngularVelocity(long address);

    /**
     * set the break force for this joint. 
     * <p>
     * if the constraint force or torque on the joint exceeds the specified values, the joint will break, 
     * at which point it will not constrain the two actors and the flag PxConstraintFlag::eBROKEN will be set. The
     * force and torque are measured in the joint frame of the first actor
     * @param force the maximum force the joint can apply before breaking
     * @param torque the maximum torque the joint can apply before breaking
     */
    public void setBreakForce(float force, float torque) {
        checkNotNull();
        _setBreakForce(address, force, torque);
    }
    private static native void _setBreakForce(long address, float force, float torque);

    /**
     * set the constraint flags for this joint. 
     * @param flags the constraint flags
     */
    public void setConstraintFlags(PxConstraintFlags flags) {
        checkNotNull();
        _setConstraintFlags(address, flags.getAddress());
    }
    private static native void _setConstraintFlags(long address, long flags);

    /**
     * set a constraint flags for this joint to a specified value. 
     * @param flag the constraint flag
     * @param value the value to which to set the flag
     */
    public void setConstraintFlag(PxConstraintFlagEnum flag, boolean value) {
        checkNotNull();
        _setConstraintFlag(address, flag.value, value);
    }
    private static native void _setConstraintFlag(long address, int flag, boolean value);

    /**
     * get the constraint flags for this joint. 
     * @return the constraint flags
     */
    public PxConstraintFlags getConstraintFlags() {
        checkNotNull();
        return PxConstraintFlags.wrapPointer(_getConstraintFlags(address));
    }
    private static native long _getConstraintFlags(long address);

    /**
     * set the inverse mass scale for actor0.
     * @param invMassScale the scale to apply to the inverse mass of actor 0 for resolving this constraint
     * @see #getInvMassScale0
     */
    public void setInvMassScale0(float invMassScale) {
        checkNotNull();
        _setInvMassScale0(address, invMassScale);
    }
    private static native void _setInvMassScale0(long address, float invMassScale);

    /**
     * get the inverse mass scale for actor0.
     * @return inverse mass scale for actor0
     * @see #setInvMassScale0
     */
    public float getInvMassScale0() {
        checkNotNull();
        return _getInvMassScale0(address);
    }
    private static native float _getInvMassScale0(long address);

    /**
     * set the inverse mass scale for actor1.
     * @param invMassScale the scale to apply to the inverse mass of actor 1 for resolving this constraint
     * @see #getInvMassScale1
     */
    public void setInvMassScale1(float invMassScale) {
        checkNotNull();
        _setInvMassScale1(address, invMassScale);
    }
    private static native void _setInvMassScale1(long address, float invMassScale);

    /**
     * get the inverse mass scale for actor1.
     * @return inverse mass scale for actor1
     * @see #setInvMassScale1
     */
    public float getInvMassScale1() {
        checkNotNull();
        return _getInvMassScale1(address);
    }
    private static native float _getInvMassScale1(long address);

    /**
     * Retrieves the PxConstraint corresponding to this joint.
     * <p>
     * This can be used to determine, among other things, the force applied at the joint.
     * @return the constraint
     */
    public PxConstraint getConstraint() {
        checkNotNull();
        return PxConstraint.wrapPointer(_getConstraint(address));
    }
    private static native long _getConstraint(long address);

    /**
     * Sets a name string for the object that can be retrieved with getName().
     * <p>
     * This is for debugging and is not used by the SDK. The string is not copied by the SDK, 
     * only the pointer is stored.
     * @param name String to set the objects name to.
     * @see #getName
     */
    public void setName(String name) {
        checkNotNull();
        _setName(address, name);
    }
    private static native void _setName(long address, String name);

    /**
     * Retrieves the name string set with setName().
     * @return Name string associated with object.
     * @see #setName
     */
    public String getName() {
        checkNotNull();
        return _getName(address);
    }
    private static native String _getName(long address);

    /**
     * Retrieves the scene which this joint belongs to.
     * @return Owner Scene. NULL if not part of a scene.
     * @see physx.physics.PxScene
     */
    public PxScene getScene() {
        checkNotNull();
        return PxScene.wrapPointer(_getScene(address));
    }
    private static native long _getScene(long address);

}
