package physx.extensions;


/**
 * A prismatic joint permits relative translational movement between two bodies along
 * an axis, but no relative rotational movement.
 * <p>
 * the axis on each body is defined as the line containing the origin of the joint frame and
 * extending along the x-axis of that frame
 * <p>
 * \image html prismJoint.png
 * @see PxJoint
 */
public class PxPrismaticJoint extends PxJoint {

    protected PxPrismaticJoint() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxPrismaticJoint wrapPointer(long address) {
        return address != 0L ? new PxPrismaticJoint(address) : null;
    }
    
    public static PxPrismaticJoint arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxPrismaticJoint(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * returns the displacement of the joint along its axis.
     */
    public float getPosition() {
        checkNotNull();
        return _getPosition(address);
    }
    private static native float _getPosition(long address);

    /**
     * returns the velocity of the joint along its axis
     */
    public float getVelocity() {
        checkNotNull();
        return _getVelocity(address);
    }
    private static native float _getVelocity(long address);

    /**
     * sets the joint limit  parameters.
     * <p>
     * The limit range is [-PX_MAX_F32, PX_MAX_F32], but note that the width of the limit (upper-lower) must also be
     * a valid float.
     * @see PxJointLinearLimitPair
     */
    public void setLimit(PxJointLinearLimitPair limit) {
        checkNotNull();
        _setLimit(address, limit.getAddress());
    }
    private static native void _setLimit(long address, long limit);

    /**
     * Set the flags specific to the Prismatic Joint.
     * <p>
     * <b>Default</b> PxPrismaticJointFlags(0)
     * @param flags The joint flags.
     */
    public void setPrismaticJointFlags(PxPrismaticJointFlags flags) {
        checkNotNull();
        _setPrismaticJointFlags(address, flags.getAddress());
    }
    private static native void _setPrismaticJointFlags(long address, long flags);

    /**
     * Set a single flag specific to a Prismatic Joint to true or false.
     * @param flag  The flag to set or clear.
     * @param value The value to which to set the flag
     */
    public void setPrismaticJointFlag(PxPrismaticJointFlagEnum flag, boolean value) {
        checkNotNull();
        _setPrismaticJointFlag(address, flag.value, value);
    }
    private static native void _setPrismaticJointFlag(long address, int flag, boolean value);

    /**
     * Get the flags specific to the Prismatic Joint.
     * @return the joint flags
     */
    public PxPrismaticJointFlags getPrismaticJointFlags() {
        checkNotNull();
        return PxPrismaticJointFlags.wrapPointer(_getPrismaticJointFlags(address));
    }
    private static native long _getPrismaticJointFlags(long address);

}
