package physx.physics;

import physx.common.PxBase;
import physx.common.PxTransform;

/**
 * A joint between two links in an articulation.
 * @see PxArticulationLink
 */
public class PxArticulationJointReducedCoordinate extends PxBase {

    protected PxArticulationJointReducedCoordinate() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxArticulationJointReducedCoordinate wrapPointer(long address) {
        return address != 0L ? new PxArticulationJointReducedCoordinate(address) : null;
    }
    
    public static PxArticulationJointReducedCoordinate arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxArticulationJointReducedCoordinate(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Gets the parent articulation link of this joint.
     * @return The parent link.
     */
    public PxArticulationLink getParentArticulationLink() {
        checkNotNull();
        return PxArticulationLink.wrapPointer(_getParentArticulationLink(address));
    }
    private static native long _getParentArticulationLink(long address);

    /**
     * Sets the joint pose in the parent link actor frame.
     * @param pose The joint pose.
     * <b>Default:</b> The identity transform.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running.
     * @see #getParentPose
     */
    public void setParentPose(PxTransform pose) {
        checkNotNull();
        _setParentPose(address, pose.getAddress());
    }
    private static native void _setParentPose(long address, long pose);

    /**
     * Gets the joint pose in the parent link actor frame.
     * @return The joint pose.
     * @see #setParentPose
     */
    public PxTransform getParentPose() {
        checkNotNull();
        return PxTransform.wrapPointer(_getParentPose(address));
    }
    private static native long _getParentPose(long address);

    /**
     * Gets the child articulation link of this joint.
     * @return The child link.
     */
    public PxArticulationLink getChildArticulationLink() {
        checkNotNull();
        return PxArticulationLink.wrapPointer(_getChildArticulationLink(address));
    }
    private static native long _getChildArticulationLink(long address);

    /**
     * Sets the joint pose in the child link actor frame.
     * @param pose The joint pose.
     * <b>Default:</b> The identity transform.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running.
     * @see #getChildPose
     */
    public void setChildPose(PxTransform pose) {
        checkNotNull();
        _setChildPose(address, pose.getAddress());
    }
    private static native void _setChildPose(long address, long pose);

    /**
     * Gets the joint pose in the child link actor frame.
     * @return The joint pose.
     * @see #setChildPose
     */
    public PxTransform getChildPose() {
        checkNotNull();
        return PxTransform.wrapPointer(_getChildPose(address));
    }
    private static native long _getChildPose(long address);

    /**
     * Sets the joint type (e.g. revolute).
     * @param jointType The joint type to set.
     * <p>
     * <b>Note:</b> Setting the joint type is not allowed while the articulation is in a scene.
     * In order to amend the joint type, remove and then re-add the articulation to the scene.
     * <p>
     * <b>Default:</b> PxArticulationJointType::eUNDEFINED
     * @see #getJointType
     */
    public void setJointType(PxArticulationJointTypeEnum jointType) {
        checkNotNull();
        _setJointType(address, jointType.value);
    }
    private static native void _setJointType(long address, int jointType);

    /**
     * Gets the joint type.
     * @return The joint type.
     * @see #setJointType
     */
    public PxArticulationJointTypeEnum getJointType() {
        checkNotNull();
        return PxArticulationJointTypeEnum.forValue(_getJointType(address));
    }
    private static native int _getJointType(long address);

    /**
     * Sets the joint motion for a given axis.
     * @param axis The target axis.
     * @param motion The motion type to set.
     * <p>
     * <b>Note:</b> Setting the motion of joint axes is not allowed while the articulation is in a scene.
     * In order to set the motion, remove and then re-add the articulation to the scene.
     * <p>
     * <b>Default:</b> PxArticulationMotion::eLOCKED
     * @see #getMotion
     */
    public void setMotion(PxArticulationAxisEnum axis, PxArticulationMotionEnum motion) {
        checkNotNull();
        _setMotion(address, axis.value, motion.value);
    }
    private static native void _setMotion(long address, int axis, int motion);

    /**
     * Returns the joint motion for the given axis.
     * @param axis The target axis.
     * @return The joint motion of the given axis.
     * @see #setMotion
     */
    public PxArticulationMotionEnum getMotion(PxArticulationAxisEnum axis) {
        checkNotNull();
        return PxArticulationMotionEnum.forValue(_getMotion(address, axis.value));
    }
    private static native int _getMotion(long address, int axis);

    /**
     * Sets the joint limits for a given axis.
     * <p>
     * - The motion of the corresponding axis should be set to PxArticulationMotion::eLIMITED in order for the limits to be enforced.
     * - The lower limit should be strictly smaller than the higher limit. If the limits should be equal, use PxArticulationMotion::eLOCKED
     *  and an appropriate offset in the parent/child joint frames.
     * @param axis The target axis.
     * @param limit The joint limits.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running.
     * <p>
     * <b>Note:</b> For PxArticulationJointType::eSPHERICAL, limit.min and limit.max must both be in range [-Pi, Pi].
     * <b>Note:</b> For PxArticulationJointType::eREVOLUTE, limit.min and limit.max must both be in range [-2*Pi, 2*Pi].
     * <b>Note:</b> For PxArticulationJointType::eREVOLUTE_UNWRAPPED, limit.min and limit.max must both be in range [-PX_MAX_REAL, PX_MAX_REAL].
     * <b>Note:</b> For PxArticulationJointType::ePRISMATIC, limit.min and limit.max must both be in range [-PX_MAX_REAL, PX_MAX_REAL].
     * <p>
     * <b>Default:</b> (0,0)
     * @see PxArticulationLimit
     */
    public void setLimitParams(PxArticulationAxisEnum axis, PxArticulationLimit limit) {
        checkNotNull();
        _setLimitParams(address, axis.value, limit.getAddress());
    }
    private static native void _setLimitParams(long address, int axis, long limit);

    /**
     * Returns the joint limits for a given axis.
     * @param axis The target axis.
     * @return The joint limits.
     * @see PxArticulationLimit
     */
    public PxArticulationLimit getLimitParams(PxArticulationAxisEnum axis) {
        checkNotNull();
        return PxArticulationLimit.wrapPointer(_getLimitParams(address, axis.value));
    }
    private static native long _getLimitParams(long address, int axis);

    /**
     * Configures a joint drive for the given axis.
     * <p>
     * See PxArticulationDrive for parameter details; and the manual for further information, and the drives' implicit spring-damper (i.e. PD control) implementation in particular.
     * @param axis The target axis.
     * @param drive The drive parameters
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running.
     * @see PxArticulationDrive
     * <p>
     * <b>Default:</b> PxArticulationDrive(0.0f, 0.0f, 0.0f, PxArticulationDriveType::eNONE)
     */
    public void setDriveParams(PxArticulationAxisEnum axis, PxArticulationDrive drive) {
        checkNotNull();
        _setDriveParams(address, axis.value, drive.getAddress());
    }
    private static native void _setDriveParams(long address, int axis, long drive);

    /**
     * Sets the joint drive position target for the given axis.
     * <p>
     * The target units are linear units (equivalent to scene units) for a translational axis, or rad for a rotational axis.
     * @param axis The target axis.
     * @param target The target position.
     * to #PxSceneDesc::wakeCounterResetValue if the counter value is below the reset value.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running.
     * <p>
     * <b>Note:</b> For spherical joints, target must be in range [-Pi, Pi].
     * <p>
     * <b>Note:</b> The target is specified in the parent frame of the joint. If Gp, Gc are the parent and child actor poses in the world frame and Lp, Lc are the parent and child joint frames expressed in the parent and child actor frames then the joint will drive the parent and child links to poses that obey Gp * Lp * J = Gc * Lc. For joints restricted to angular motion, J has the form PxTranfsorm(PxVec3(PxZero), PxExp(PxVec3(twistTarget, swing1Target, swing2Target))).  For joints restricted to linear motion, J has the form PxTransform(PxVec3(XTarget, YTarget, ZTarget), PxQuat(PxIdentity)).
     * <p>
     * <b>Note:</b> For spherical joints with more than 1 degree of freedom, the joint target angles taken together can collectively represent a rotation of greater than Pi around a vector. When this happens the rotation that matches the joint drive target is not the shortest path rotation.  The joint pose J that is the outcome after driving to the target pose will always be the equivalent of the shortest path rotation.
     * @see #getDriveTarget
     * <p>
     * <b>Default:</b> 0.0
     */
    public void setDriveTarget(PxArticulationAxisEnum axis, float target) {
        checkNotNull();
        _setDriveTarget(address, axis.value, target);
    }
    private static native void _setDriveTarget(long address, int axis, float target);

    /**
     * Sets the joint drive position target for the given axis.
     * <p>
     * The target units are linear units (equivalent to scene units) for a translational axis, or rad for a rotational axis.
     * @param axis The target axis.
     * @param target The target position.
     * @param autowake If true and the articulation is in a scene, the call wakes up the articulation and increases the wake counter
     * to #PxSceneDesc::wakeCounterResetValue if the counter value is below the reset value.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running.
     * <p>
     * <b>Note:</b> For spherical joints, target must be in range [-Pi, Pi].
     * <p>
     * <b>Note:</b> The target is specified in the parent frame of the joint. If Gp, Gc are the parent and child actor poses in the world frame and Lp, Lc are the parent and child joint frames expressed in the parent and child actor frames then the joint will drive the parent and child links to poses that obey Gp * Lp * J = Gc * Lc. For joints restricted to angular motion, J has the form PxTranfsorm(PxVec3(PxZero), PxExp(PxVec3(twistTarget, swing1Target, swing2Target))).  For joints restricted to linear motion, J has the form PxTransform(PxVec3(XTarget, YTarget, ZTarget), PxQuat(PxIdentity)).
     * <p>
     * <b>Note:</b> For spherical joints with more than 1 degree of freedom, the joint target angles taken together can collectively represent a rotation of greater than Pi around a vector. When this happens the rotation that matches the joint drive target is not the shortest path rotation.  The joint pose J that is the outcome after driving to the target pose will always be the equivalent of the shortest path rotation.
     * @see #getDriveTarget
     * <p>
     * <b>Default:</b> 0.0
     */
    public void setDriveTarget(PxArticulationAxisEnum axis, float target, boolean autowake) {
        checkNotNull();
        _setDriveTarget(address, axis.value, target, autowake);
    }
    private static native void _setDriveTarget(long address, int axis, float target, boolean autowake);

    /**
     * Returns the joint drive position target for the given axis.
     * @param axis The target axis.
     * @return The target position.
     * @see #setDriveTarget
     */
    public float getDriveTarget(PxArticulationAxisEnum axis) {
        checkNotNull();
        return _getDriveTarget(address, axis.value);
    }
    private static native float _getDriveTarget(long address, int axis);

    /**
     * Sets the joint drive velocity target for the given axis.
     * <p>
     * The target units are linear units (equivalent to scene units) per second for a translational axis, or radians per second for a rotational axis.
     * @param axis The target axis.
     * @param targetVel The target velocity.
     * to #PxSceneDesc::wakeCounterResetValue if the counter value is below the reset value.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running.
     * @see #getDriveVelocity
     * <p>
     * <b>Default:</b> 0.0
     */
    public void setDriveVelocity(PxArticulationAxisEnum axis, float targetVel) {
        checkNotNull();
        _setDriveVelocity(address, axis.value, targetVel);
    }
    private static native void _setDriveVelocity(long address, int axis, float targetVel);

    /**
     * Sets the joint drive velocity target for the given axis.
     * <p>
     * The target units are linear units (equivalent to scene units) per second for a translational axis, or radians per second for a rotational axis.
     * @param axis The target axis.
     * @param targetVel The target velocity.
     * @param autowake If true and the articulation is in a scene, the call wakes up the articulation and increases the wake counter
     * to #PxSceneDesc::wakeCounterResetValue if the counter value is below the reset value.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running.
     * @see #getDriveVelocity
     * <p>
     * <b>Default:</b> 0.0
     */
    public void setDriveVelocity(PxArticulationAxisEnum axis, float targetVel, boolean autowake) {
        checkNotNull();
        _setDriveVelocity(address, axis.value, targetVel, autowake);
    }
    private static native void _setDriveVelocity(long address, int axis, float targetVel, boolean autowake);

    /**
     * Returns the joint drive velocity target for the given axis.
     * @param axis The target axis.
     * @return The target velocity.
     * @see #setDriveVelocity
     */
    public float getDriveVelocity(PxArticulationAxisEnum axis) {
        checkNotNull();
        return _getDriveVelocity(address, axis.value);
    }
    private static native float _getDriveVelocity(long address, int axis);

    /**
     * Sets the joint armature for the given axis.
     * <p>
     * - The armature is directly added to the joint-space spatial inertia of the corresponding axis.
     * - The armature is in mass units for a prismatic (i.e. linear) joint, and in mass units * (scene linear units)^2 for a rotational joint.
     * @param axis The target axis.
     * @param armature The joint axis armature.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running.
     * @see #getArmature
     * <p>
     * <b>Default:</b> 0.0
     */
    public void setArmature(PxArticulationAxisEnum axis, float armature) {
        checkNotNull();
        _setArmature(address, axis.value, armature);
    }
    private static native void _setArmature(long address, int axis, float armature);

    /**
     * Gets the joint armature for the given axis.
     * @param axis The target axis.
     * @return The armature set on the given axis.
     * @see #setArmature
     */
    public float getArmature(PxArticulationAxisEnum axis) {
        checkNotNull();
        return _getArmature(address, axis.value);
    }
    private static native float _getArmature(long address, int axis);

    /**
     * Sets the joint friction coefficient, which applies to all joint axes.
     * <p>
     * - The joint friction is unitless and relates the magnitude of the spatial force [F_trans, T_trans] transmitted from parent to child link to
     * the maximal friction force F_resist that may be applied by the solver to resist joint motion, per axis; i.e. |F_resist| &lt;= coefficient * (|F_trans| + |T_trans|),
     * where F_resist may refer to a linear force or torque depending on the joint axis.
     * - The simulated friction effect is therefore similar to static and Coulomb friction. In order to simulate dynamic joint friction, use a joint drive with
     * zero stiffness and zero velocity target, and an appropriately dimensioned damping parameter.
     * @param coefficient The joint friction coefficient.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running.
     * @see #getFrictionCoefficient
     * <p>
     * <b>Default:</b> 0.05
     */
    public void setFrictionCoefficient(float coefficient) {
        checkNotNull();
        _setFrictionCoefficient(address, coefficient);
    }
    private static native void _setFrictionCoefficient(long address, float coefficient);

    /**
     * Gets the joint friction coefficient.
     * @return The joint friction coefficient.
     * @see #setFrictionCoefficient
     */
    public float getFrictionCoefficient() {
        checkNotNull();
        return _getFrictionCoefficient(address);
    }
    private static native float _getFrictionCoefficient(long address);

    /**
     * Sets the maximal joint velocity enforced for all axes.
     * <p>
     * - The solver will apply appropriate joint-space impulses in order to enforce the per-axis joint-velocity limit.
     * - The velocity units are linear units (equivalent to scene units) per second for a translational axis, or radians per second for a rotational axis.
     * @param maxJointV The maximal per-axis joint velocity.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running.
     * @see #getMaxJointVelocity
     * <p>
     * <b>Default:</b> 100.0
     */
    public void setMaxJointVelocity(float maxJointV) {
        checkNotNull();
        _setMaxJointVelocity(address, maxJointV);
    }
    private static native void _setMaxJointVelocity(long address, float maxJointV);

    /**
     * Gets the maximal joint velocity enforced for all axes.
     * @return The maximal per-axis joint velocity.
     * @see #setMaxJointVelocity
     */
    public float getMaxJointVelocity() {
        checkNotNull();
        return _getMaxJointVelocity(address);
    }
    private static native float _getMaxJointVelocity(long address);

    /**
     * Sets the joint position for the given axis.
     * <p>
     * - For performance, prefer PxArticulationCache::jointPosition to set joint positions in a batch articulation state update.
     * - Use PxArticulationReducedCoordinate::updateKinematic after all state updates to the articulation via non-cache API such as this method,
     * in order to update link states for the next simulation frame or querying.
     * @param axis The target axis.
     * @param jointPos The joint position in linear units (equivalent to scene units) for a translational axis, or radians for a rotational axis.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running.
     * <p>
     * <b>Note:</b> For PxArticulationJointType::eSPHERICAL, jointPos must be in range [-Pi, Pi].
     * <b>Note:</b> For PxArticulationJointType::eREVOLUTE, jointPos must be in range [-2*Pi, 2*Pi].
     * <b>Note:</b> For PxArticulationJointType::eREVOLUTE_UNWRAPPED, jointPos must be in range [-PX_MAX_REAL, PX_MAX_REAL].
     * <b>Note:</b> For PxArticulationJointType::ePRISMATIC, jointPos must be in range [-PX_MAX_REAL, PX_MAX_REAL].
     * <p>
     * <b>Note:</b> Joint position is specified in the parent frame of the joint. If Gp, Gc are the parent and child actor poses in the world frame and Lp, Lc are the parent and child joint frames expressed in the parent and child actor frames then the parent and child links will be given poses that obey Gp * Lp * J = Gc * Lc with J denoting the joint pose. For joints restricted to angular motion, J has the form PxTranfsorm(PxVec3(PxZero), PxExp(PxVec3(twistPos, swing1Pos, swing2Pos))).  For joints restricted to linear motion, J has the form PxTransform(PxVec3(xPos, yPos, zPos), PxQuat(PxIdentity)).
     * <p>
     * <b>Note:</b> For spherical joints with more than 1 degree of freedom, the input joint positions taken together can collectively represent a rotation of greater than Pi around a vector. When this happens the rotation that matches the joint positions is not the shortest path rotation.  The joint pose J that is the outcome of setting and applying the joint positions will always be the equivalent of the shortest path rotation.
     * <p>
     * <b>Default:</b> 0.0
     */
    public void setJointPosition(PxArticulationAxisEnum axis, float jointPos) {
        checkNotNull();
        _setJointPosition(address, axis.value, jointPos);
    }
    private static native void _setJointPosition(long address, int axis, float jointPos);

    /**
     * Gets the joint position for the given axis, i.e. joint degree of freedom (DOF).
     * <p>
     * For performance, prefer PxArticulationCache::jointPosition to get joint positions in a batch query.
     * @param axis The target axis.
     * @return The joint position in linear units (equivalent to scene units) for a translational axis, or radians for a rotational axis.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running except in a split simulation during #PxScene::collide() and up to #PxScene::advance(),
     * and in PxContactModifyCallback or in contact report callbacks.
     */
    public float getJointPosition(PxArticulationAxisEnum axis) {
        checkNotNull();
        return _getJointPosition(address, axis.value);
    }
    private static native float _getJointPosition(long address, int axis);

    /**
     * Sets the joint velocity for the given axis.
     * <p>
     * - For performance, prefer PxArticulationCache::jointVelocity to set joint velocities in a batch articulation state update.
     * - Use PxArticulationReducedCoordinate::updateKinematic after all state updates to the articulation via non-cache API such as this method,
     * in order to update link states for the next simulation frame or querying.
     * @param axis The target axis.
     * @param jointVel The joint velocity in linear units (equivalent to scene units) per second for a translational axis, or radians per second for a rotational axis.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running.
     * <p>
     * <b>Default:</b> 0.0
     */
    public void setJointVelocity(PxArticulationAxisEnum axis, float jointVel) {
        checkNotNull();
        _setJointVelocity(address, axis.value, jointVel);
    }
    private static native void _setJointVelocity(long address, int axis, float jointVel);

    /**
     * Gets the joint velocity for the given axis.
     * <p>
     * For performance, prefer PxArticulationCache::jointVelocity to get joint velocities in a batch query.
     * @param axis The target axis.
     * @return The joint velocity in linear units (equivalent to scene units) per second for a translational axis, or radians per second for a rotational axis.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running except in a split simulation during #PxScene::collide() and up to #PxScene::advance(),
     * and in PxContactModifyCallback or in contact report callbacks.
     */
    public float getJointVelocity(PxArticulationAxisEnum axis) {
        checkNotNull();
        return _getJointVelocity(address, axis.value);
    }
    private static native float _getJointVelocity(long address, int axis);

}
