package physx.physics;

import physx.NativeObject;
import physx.PlatformChecks;
import physx.common.PxBaseTask;
import physx.common.PxBounds3;
import physx.common.PxCollection;
import physx.common.PxCpuDispatcher;
import physx.common.PxCudaContextManager;
import physx.common.PxRenderBuffer;
import physx.common.PxVec3;
import physx.geometry.PxBVH;
import physx.particles.PxParticleSolverTypeEnum;
import physx.support.PxVisualizationParameterEnum;

/**
 * A scene is a collection of bodies and constraints which can interact.
 * <p>
 * The scene simulates the behavior of these objects over time. Several scenes may exist 
 * at the same time, but each body or constraint is specific to a scene 
 * -- they may not be shared.
 * @see PxSceneDesc
 * @see PxPhysics#createScene
 * @see #release
 */
public class PxScene extends PxSceneSQSystem {

    protected PxScene() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxScene wrapPointer(long address) {
        return address != 0L ? new PxScene(address) : null;
    }
    
    public static PxScene arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxScene(long address) {
        super(address);
    }

    // Attributes

    /**
     * user can assign this to whatever, usually to create a 1:1 relationship with a user object.
     */
    public NativeObject getUserData() {
        checkNotNull();
        return NativeObject.wrapPointer(_getUserData(address));
    }
    private static native long _getUserData(long address);

    /**
     * user can assign this to whatever, usually to create a 1:1 relationship with a user object.
     */
    public void setUserData(NativeObject value) {
        checkNotNull();
        _setUserData(address, value.getAddress());
    }
    private static native void _setUserData(long address, long value);

    // Functions

    /**
     * Adds an actor to this scene.
     * <p>
     * <b>Note:</b> If the actor is already assigned to a scene (see #PxActor::getScene), the call is ignored and an error is issued.
     * <b>Note:</b> If the actor has an invalid constraint, in checked builds the call is ignored and an error is issued.
     * <p>
     * <b>Note:</b> You can not add individual articulation links (see #PxArticulationLink) to the scene. Use #addArticulation() instead.
     * <p>
     * <b>Note:</b> If the actor is a PxRigidActor then each assigned PxConstraint object will get added to the scene automatically if
     * it connects to another actor that is part of the scene already. 
     * <p>
     * <b>Note:</b> When a BVH is provided the actor shapes are grouped together. 
     * The scene query pruning structure inside PhysX SDK will store/update one
     * bound per actor. The scene queries against such an actor will query actor
     * bounds and then make a local space query against the provided BVH, which is in actor's local space.
     * @param actor Actor to add to scene.
     * @return True if success
     * @see physx.geometry.PxBVH
     */
    public boolean addActor(PxActor actor) {
        checkNotNull();
        return _addActor(address, actor.getAddress());
    }
    private static native boolean _addActor(long address, long actor);

    /**
     * Adds an actor to this scene.
     * <p>
     * <b>Note:</b> If the actor is already assigned to a scene (see #PxActor::getScene), the call is ignored and an error is issued.
     * <b>Note:</b> If the actor has an invalid constraint, in checked builds the call is ignored and an error is issued.
     * <p>
     * <b>Note:</b> You can not add individual articulation links (see #PxArticulationLink) to the scene. Use #addArticulation() instead.
     * <p>
     * <b>Note:</b> If the actor is a PxRigidActor then each assigned PxConstraint object will get added to the scene automatically if
     * it connects to another actor that is part of the scene already. 
     * <p>
     * <b>Note:</b> When a BVH is provided the actor shapes are grouped together. 
     * The scene query pruning structure inside PhysX SDK will store/update one
     * bound per actor. The scene queries against such an actor will query actor
     * bounds and then make a local space query against the provided BVH, which is in actor's local space.
     * @param actor Actor to add to scene.
     * @param bvh  BVH for actor shapes.
     * @return True if success
     * @see physx.geometry.PxBVH
     */
    public boolean addActor(PxActor actor, PxBVH bvh) {
        checkNotNull();
        return _addActor(address, actor.getAddress(), bvh.getAddress());
    }
    private static native boolean _addActor(long address, long actor, long bvh);

    /**
     * Removes an actor from this scene.
     * <p>
     * <b>Note:</b> If the actor is not part of this scene (see #PxActor::getScene), the call is ignored and an error is issued.
     * <p>
     * <b>Note:</b> You can not remove individual articulation links (see #PxArticulationLink) from the scene. Use #removeArticulation() instead.
     * <p>
     * <b>Note:</b> If the actor is a PxRigidActor then all assigned PxConstraint objects will get removed from the scene automatically.
     * <p>
     * <b>Note:</b> If the actor is in an aggregate it will be removed from the aggregate.
     * @param actor Actor to remove from scene.
     * @see PxAggregate
     */
    public void removeActor(PxActor actor) {
        checkNotNull();
        _removeActor(address, actor.getAddress());
    }
    private static native void _removeActor(long address, long actor);

    /**
     * Removes an actor from this scene.
     * <p>
     * <b>Note:</b> If the actor is not part of this scene (see #PxActor::getScene), the call is ignored and an error is issued.
     * <p>
     * <b>Note:</b> You can not remove individual articulation links (see #PxArticulationLink) from the scene. Use #removeArticulation() instead.
     * <p>
     * <b>Note:</b> If the actor is a PxRigidActor then all assigned PxConstraint objects will get removed from the scene automatically.
     * <p>
     * <b>Note:</b> If the actor is in an aggregate it will be removed from the aggregate.
     * @param actor Actor to remove from scene.
     * @param wakeOnLostTouch Specifies whether touching objects from the previous frame should get woken up in the next frame. Only applies to PxArticulationReducedCoordinate and PxRigidActor types.
     * @see PxAggregate
     */
    public void removeActor(PxActor actor, boolean wakeOnLostTouch) {
        checkNotNull();
        _removeActor(address, actor.getAddress(), wakeOnLostTouch);
    }
    private static native void _removeActor(long address, long actor, boolean wakeOnLostTouch);

    /**
     * Adds an aggregate to this scene.
     * <p>
     * <b>Note:</b> If the aggregate is already assigned to a scene (see #PxAggregate::getScene), the call is ignored and an error is issued.
     * <b>Note:</b> If the aggregate contains an actor with an invalid constraint, in checked builds the call is ignored and an error is issued.
     * <p>
     * <b>Note:</b> If the aggregate already contains actors, those actors are added to the scene as well.
     * @param aggregate Aggregate to add to scene.
     * @return True if success
     */
    public boolean addAggregate(PxAggregate aggregate) {
        checkNotNull();
        return _addAggregate(address, aggregate.getAddress());
    }
    private static native boolean _addAggregate(long address, long aggregate);

    /**
     * Removes an aggregate from this scene.
     * <p>
     * <b>Note:</b> If the aggregate is not part of this scene (see #PxAggregate::getScene), the call is ignored and an error is issued.
     * <p>
     * <b>Note:</b> If the aggregate contains actors, those actors are removed from the scene as well.
     * @param aggregate Aggregate to remove from scene.
     * @see PxAggregate
     */
    public void removeAggregate(PxAggregate aggregate) {
        checkNotNull();
        _removeAggregate(address, aggregate.getAddress());
    }
    private static native void _removeAggregate(long address, long aggregate);

    /**
     * Removes an aggregate from this scene.
     * <p>
     * <b>Note:</b> If the aggregate is not part of this scene (see #PxAggregate::getScene), the call is ignored and an error is issued.
     * <p>
     * <b>Note:</b> If the aggregate contains actors, those actors are removed from the scene as well.
     * @param aggregate Aggregate to remove from scene.
     * @param wakeOnLostTouch Specifies whether touching objects from the previous frame should get woken up in the next frame. Only applies to PxArticulationReducedCoordinate and PxRigidActor types.
     * @see PxAggregate
     */
    public void removeAggregate(PxAggregate aggregate, boolean wakeOnLostTouch) {
        checkNotNull();
        _removeAggregate(address, aggregate.getAddress(), wakeOnLostTouch);
    }
    private static native void _removeAggregate(long address, long aggregate, boolean wakeOnLostTouch);

    /**
     * Adds objects in the collection to this scene.
     * <p>
     * This function adds the following types of objects to this scene: PxRigidActor (except PxArticulationLink), PxAggregate, PxArticulationReducedCoordinate. 
     * This method is typically used after deserializing the collection in order to populate the scene with deserialized objects.
     * <p>
     * <b>Note:</b> If the collection contains an actor with an invalid constraint, in checked builds the call is ignored and an error is issued.
     * @param collection Objects to add to this scene. See #PxCollection
     * @return True if success
     */
    public boolean addCollection(PxCollection collection) {
        checkNotNull();
        return _addCollection(address, collection.getAddress());
    }
    private static native boolean _addCollection(long address, long collection);

    /**
     * Returns the wake counter reset value.
     * @return Wake counter reset value
     */
    public float getWakeCounterResetValue() {
        checkNotNull();
        return _getWakeCounterResetValue(address);
    }
    private static native float _getWakeCounterResetValue(long address);

    /**
     * Shift the scene origin by the specified vector.
     * <p>
     * The poses of all objects in the scene and the corresponding data structures will get adjusted to reflect the new origin location
     * (the shift vector will get subtracted from all object positions).
     * <p>
     * <b>Note:</b> It is the user's responsibility to keep track of the summed total origin shift and adjust all input/output to/from PhysX accordingly.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running. Calls to this method while the simulation is running will be ignored.
     * <p>
     * <b>Note:</b> Make sure to propagate the origin shift to other dependent modules (for example, the character controller module etc.).
     * <p>
     * <b>Note:</b> This is an expensive operation and we recommend to use it only in the case where distance related precision issues may arise in areas far from the origin.
     * @param shift Translation vector to shift the origin by.
     */
    public void shiftOrigin(PxVec3 shift) {
        checkNotNull();
        _shiftOrigin(address, shift.getAddress());
    }
    private static native void _shiftOrigin(long address, long shift);

    /**
     * Adds an articulation to this scene.
     * <p>
     * <b>Note:</b> If the articulation is already assigned to a scene (see #PxArticulationReducedCoordinate::getScene), the call is ignored and an error is issued.
     * @param articulation The articulation to add to the scene.
     * @return True if success
     * @see PxArticulationReducedCoordinate
     */
    public boolean addArticulation(PxArticulationReducedCoordinate articulation) {
        checkNotNull();
        return _addArticulation(address, articulation.getAddress());
    }
    private static native boolean _addArticulation(long address, long articulation);

    /**
     * Removes an articulation from this scene.
     * <p>
     * <b>Note:</b> If the articulation is not part of this scene (see #PxArticulationReducedCoordinate::getScene), the call is ignored and an error is issued. 
     * <p>
     * <b>Note:</b> If the articulation is in an aggregate it will be removed from the aggregate.
     * @param articulation The articulation to remove from the scene.
     * Only applies to PxArticulationReducedCoordinate and PxRigidActor types.
     * @see PxAggregate
     */
    public void removeArticulation(PxArticulationReducedCoordinate articulation) {
        checkNotNull();
        _removeArticulation(address, articulation.getAddress());
    }
    private static native void _removeArticulation(long address, long articulation);

    /**
     * Removes an articulation from this scene.
     * <p>
     * <b>Note:</b> If the articulation is not part of this scene (see #PxArticulationReducedCoordinate::getScene), the call is ignored and an error is issued. 
     * <p>
     * <b>Note:</b> If the articulation is in an aggregate it will be removed from the aggregate.
     * @param articulation The articulation to remove from the scene.
     * @param wakeOnLostTouch Specifies whether touching objects from the previous frame should get woken up in the next frame.
     * Only applies to PxArticulationReducedCoordinate and PxRigidActor types.
     * @see PxAggregate
     */
    public void removeArticulation(PxArticulationReducedCoordinate articulation, boolean wakeOnLostTouch) {
        checkNotNull();
        _removeArticulation(address, articulation.getAddress(), wakeOnLostTouch);
    }
    private static native void _removeArticulation(long address, long articulation, boolean wakeOnLostTouch);

    /**
     * Retrieve the number of actors of certain types in the scene. For supported types, see PxActorTypeFlags.
     * @param types Combination of actor types.
     * @return the number of actors.
     */
    public int getNbActors(PxActorTypeFlags types) {
        checkNotNull();
        return _getNbActors(address, types.getAddress());
    }
    private static native int _getNbActors(long address, long types);

    /**
     * Retrieve the number of soft bodies in the scene.
     * @return the number of soft bodies.
     */
    public int getNbSoftBodies() {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxScene");
        return _getNbSoftBodies(address);
    }
    private static native int _getNbSoftBodies(long address);

    /**
     * @deprecated  Use getNbPBDParticleSystems() instead.
     * Retrieve the number of particle systems of the requested type in the scene.
     * @param type The particle system type. See PxParticleSolverType. Only one type can be requested per function call.
     * @return the number particle systems.
     * <p>
     * See getPBDParticleSystems(), PxParticleSolverType
     */
    @Deprecated
    public int getNbParticleSystems(PxParticleSolverTypeEnum type) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxScene");
        return _getNbParticleSystems(address, type.value);
    }
    private static native int _getNbParticleSystems(long address, int type);

    /**
     * Returns the number of articulations in the scene.
     * @return the number of articulations in this scene.
     */
    public int getNbArticulations() {
        checkNotNull();
        return _getNbArticulations(address);
    }
    private static native int _getNbArticulations(long address);

    /**
     * Returns the number of constraint shaders in the scene.
     * @return the number of constraint shaders in this scene.
     */
    public int getNbConstraints() {
        checkNotNull();
        return _getNbConstraints(address);
    }
    private static native int _getNbConstraints(long address);

    /**
     * Returns the number of aggregates in the scene.
     * @return the number of aggregates in this scene.
     */
    public int getNbAggregates() {
        checkNotNull();
        return _getNbAggregates(address);
    }
    private static native int _getNbAggregates(long address);

    /**
     * Specifies the dominance behavior of contacts between two actors with two certain dominance groups.
     * <p>
     * It is possible to assign each actor to a dominance groups using #PxActor::setDominanceGroup().
     * <p>
     * With dominance groups one can have all contacts created between actors act in one direction only. This is useful, for example, if you
     * want an object to push debris out of its way and be unaffected,while still responding physically to forces and collisions
     * with non-debris objects.
     * <p>
     * Whenever a contact between two actors (a0, a1) needs to be solved, the groups (g0, g1) of both
     * actors are retrieved. Then the PxDominanceGroupPair setting for this group pair is retrieved with getDominanceGroupPair(g0, g1).
     * <p>
     * In the contact, PxDominanceGroupPair::dominance0 becomes the dominance setting for a0, and 
     * PxDominanceGroupPair::dominance1 becomes the dominance setting for a1. A dominanceN setting of 1.0f, the default, 
     * will permit aN to be pushed or pulled by a(1-N) through the contact. A dominanceN setting of 0.0f, will however 
     * prevent aN to be pushed by a(1-N) via the contact. Thus, a PxDominanceGroupPair of (1.0f, 0.0f) makes 
     * the interaction one-way.
     * <p>
     * The matrix sampled by getDominanceGroupPair(g1, g2) is initialised by default such that:
     * <p>
     * if g1 == g2, then (1.0f, 1.0f) is returned
     * if g1 &lt;  g2, then (0.0f, 1.0f) is returned
     * if g1 &gt;  g2, then (1.0f, 0.0f) is returned
     * <p>
     * In other words, we permit actors in higher groups to be pushed around by actors in lower groups by default.
     * <p>
     * These settings should cover most applications, and in fact not overriding these settings may likely result in higher performance.
     * <p>
     * It is not possible to make the matrix asymetric, or to change the diagonal. In other words: 
     * <p>
     * * it is not possible to change (g1, g2) if (g1==g2) 
     * * if you set 
     * <p>
     * (g1, g2) to X, then (g2, g1) will implicitly and automatically be set to ~X, where:
     * <p>
     * ~(1.0f, 1.0f) is (1.0f, 1.0f)
     * ~(0.0f, 1.0f) is (1.0f, 0.0f)
     * ~(1.0f, 0.0f) is (0.0f, 1.0f)
     * <p>
     * These two restrictions are to make sure that contacts between two actors will always evaluate to the same dominance
     * setting, regardless of the order of the actors.
     * <p>
     * Dominance settings are currently specified as floats 0.0f or 1.0f because in the future we may permit arbitrary 
     * fractional settings to express 'partly-one-way' interactions.
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake actors up automatically.
     * @see #getDominanceGroupPair
     * @see PxDominanceGroupPair
     */
    public void setDominanceGroupPair(byte group1, byte group2, PxDominanceGroupPair dominance) {
        checkNotNull();
        _setDominanceGroupPair(address, group1, group2, dominance.getAddress());
    }
    private static native void _setDominanceGroupPair(long address, byte group1, byte group2, long dominance);

    /**
     * Samples the dominance matrix.
     * @see #setDominanceGroupPair
     * @see PxDominanceGroupPair
     */
    public PxDominanceGroupPair getDominanceGroupPair(byte group1, byte group2) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxScene");
        return PxDominanceGroupPair.wrapPointer(_getDominanceGroupPair(address, group1, group2));
    }
    private static native long _getDominanceGroupPair(long address, byte group1, byte group2);

    /**
     * Return the cpu dispatcher that was set in PxSceneDesc::cpuDispatcher when creating the scene with PxPhysics::createScene
     */
    public PxCpuDispatcher getCpuDispatcher() {
        checkNotNull();
        return PxCpuDispatcher.wrapPointer(_getCpuDispatcher(address));
    }
    private static native long _getCpuDispatcher(long address);

    /**
     * Return the CUDA context manager that was set in PxSceneDesc::cudaContextManager when creating the scene with PxPhysics::createScene
     * <p>
     * <b>Platform specific:</b> Applies to PC GPU only.
     */
    public PxCudaContextManager getCudaContextManager() {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxScene");
        return PxCudaContextManager.wrapPointer(_getCudaContextManager(address));
    }
    private static native long _getCudaContextManager(long address);

    /**
     * Reserves a new client ID.
     * <p>
     * PX_DEFAULT_CLIENT is always available as the default clientID.
     * Additional clients are returned by this function. Clients cannot be released once created. 
     * An error is reported when more than a supported number of clients (currently 128) are created. 
     */
    public byte createClient() {
        checkNotNull();
        return _createClient(address);
    }
    private static native byte _createClient(long address);

    /**
     * Sets a user notify object which receives special simulation events when they occur.
     * <p>
     * <b>Note:</b> Do not set the callback while the simulation is running. Calls to this method while the simulation is running will be ignored.
     * @param callback User notification callback. See #PxSimulationEventCallback.
     * @see PxSimulationEventCallback
     * @see #getSimulationEventCallback
     */
    public void setSimulationEventCallback(PxSimulationEventCallback callback) {
        checkNotNull();
        _setSimulationEventCallback(address, callback.getAddress());
    }
    private static native void _setSimulationEventCallback(long address, long callback);

    /**
     * Retrieves the simulationEventCallback pointer set with setSimulationEventCallback().
     * @return The current user notify pointer. See #PxSimulationEventCallback.
     * @see PxSimulationEventCallback
     * @see #setSimulationEventCallback
     */
    public PxSimulationEventCallback getSimulationEventCallback() {
        checkNotNull();
        return PxSimulationEventCallback.wrapPointer(_getSimulationEventCallback(address));
    }
    private static native long _getSimulationEventCallback(long address);

    /**
     * Sets the shared global filter data which will get passed into the filter shader.
     * <p>
     * <b>Note:</b> It is the user's responsibility to ensure that changing the shared global filter data does not change the filter output value for existing pairs. 
     *       If the filter output for existing pairs does change nonetheless then such a change will not take effect until the pair gets refiltered. 
     *    resetFiltering() can be used to explicitly refilter the pairs of specific objects.
     * <p>
     * <b>Note:</b> The provided data will get copied to internal buffers and this copy will be used for filtering calls.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running. Calls to this method while the simulation is running will be ignored.
     * @param data The shared global filter shader data.
     * @param dataSize Size of the shared global filter shader data (in bytes).
     * @see #getFilterShaderData
     * @see PxSimulationFilterShader
     */
    public void setFilterShaderData(NativeObject data, int dataSize) {
        checkNotNull();
        _setFilterShaderData(address, data.getAddress(), dataSize);
    }
    private static native void _setFilterShaderData(long address, long data, int dataSize);

    /**
     * Gets the shared global filter data in use for this scene.
     * <p>
     * <b>Note:</b> The reference points to a copy of the original filter data specified in #PxSceneDesc.filterShaderData or provided by #setFilterShaderData().
     * @return Shared filter data for filter shader.
     * @see #getFilterShaderDataSize
     * @see #setFilterShaderData
     * @see PxSimulationFilterShader
     */
    public NativeObject getFilterShaderData() {
        checkNotNull();
        return NativeObject.wrapPointer(_getFilterShaderData(address));
    }
    private static native long _getFilterShaderData(long address);

    /**
     * Gets the size of the shared global filter data (#PxSceneDesc.filterShaderData)
     * @return Size of shared filter data [bytes].
     * @see #getFilterShaderData
     * @see PxSimulationFilterShader
     */
    public int getFilterShaderDataSize() {
        checkNotNull();
        return _getFilterShaderDataSize(address);
    }
    private static native int _getFilterShaderDataSize(long address);

    /**
     * Gets the custom collision filter shader in use for this scene.
     * @return Filter shader class that defines the collision pair filtering.
     * @see PxSimulationFilterShader
     */
    public PxSimulationFilterShader getFilterShader() {
        checkNotNull();
        return PxSimulationFilterShader.wrapPointer(_getFilterShader(address));
    }
    private static native long _getFilterShader(long address);

    /**
     * Marks the object to reset interactions and re-run collision filters in the next simulation step.
     * <p>
     * This call forces the object to remove all existing collision interactions, to search anew for existing contact
     * pairs and to run the collision filters again for found collision pairs.
     * <p>
     * <b>Note:</b> The operation is supported for PxRigidActor objects only.
     * <p>
     * <b>Note:</b> All persistent state of existing interactions will be lost and can not be retrieved even if the same collison pair
     * is found again in the next step. This will mean, for example, that you will not get notified about persistent contact
     * for such an interaction (see #PxPairFlag::eNOTIFY_TOUCH_PERSISTS), the contact pair will be interpreted as newly found instead.
     * <p>
     * <b>Note:</b> Lost touch contact reports will be sent for every collision pair which includes this shape, if they have
     * been requested through #PxPairFlag::eNOTIFY_TOUCH_LOST or #PxPairFlag::eNOTIFY_THRESHOLD_FORCE_LOST.
     * <p>
     * <b>Note:</b> This is an expensive operation, don't use it if you don't have to.
     * <p>
     * <b>Note:</b> Can be used to retrieve collision pairs that were killed by the collision filters (see #PxFilterFlag::eKILL)
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already.
     * <p>
     * <b>Note:</b> It is invalid to use this method if PxActorFlag::eDISABLE_SIMULATION is set.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running.
     * <p>
     * <b>Sleeping:</b> Does wake up the actor.
     * @param actor The actor for which to re-evaluate interactions.
     * @return True if success
     * @see PxSimulationFilterShader
     */
    public boolean resetFiltering(PxActor actor) {
        checkNotNull();
        return _resetFiltering(address, actor.getAddress());
    }
    private static native boolean _resetFiltering(long address, long actor);

    /**
     * Gets the pair filtering mode for kinematic-kinematic pairs.
     * @return Filtering mode for kinematic-kinematic pairs.
     * @see PxSceneDesc
     */
    public PxPairFilteringModeEnum getKinematicKinematicFilteringMode() {
        checkNotNull();
        return PxPairFilteringModeEnum.forValue(_getKinematicKinematicFilteringMode(address));
    }
    private static native int _getKinematicKinematicFilteringMode(long address);

    /**
     * Gets the pair filtering mode for static-kinematic pairs.
     * @return Filtering mode for static-kinematic pairs.
     * @see PxSceneDesc
     */
    public PxPairFilteringModeEnum getStaticKinematicFilteringMode() {
        checkNotNull();
        return PxPairFilteringModeEnum.forValue(_getStaticKinematicFilteringMode(address));
    }
    private static native int _getStaticKinematicFilteringMode(long address);

    /**
     *  Advances the simulation by an elapsedTime time.
     * <p>
     * <b>Note:</b> Large elapsedTime values can lead to instabilities. In such cases elapsedTime
     * should be subdivided into smaller time intervals and simulate() should be called
     * multiple times for each interval.
     * <p>
     * Calls to simulate() should pair with calls to fetchResults():
     *  Each fetchResults() invocation corresponds to exactly one simulate()
     *  invocation; calling simulate() twice without an intervening fetchResults()
     *  or fetchResults() twice without an intervening simulate() causes an error
     *  condition.
     * <p>
     *  scene-&gt;simulate();
     *  ...do some processing until physics is computed...
     *  scene-&gt;fetchResults();
     *  ...now results of run may be retrieved.
     * @param elapsedTime Amount of time to advance simulation by. The parameter has to be larger than 0, else the resulting behavior will be undefined. <b>Range:</b> (0, PX_MAX_F32)
     * decremented when the scene is ready to have fetchResults called. So the task will not run until the
     * application also calls removeReference().
     * after fetchResults returns. Must be aligned on a 16-byte boundary
     * @return True if success
     * @see #fetchResults
     * @see #checkResults
     */
    public boolean simulate(float elapsedTime) {
        checkNotNull();
        return _simulate(address, elapsedTime);
    }
    private static native boolean _simulate(long address, float elapsedTime);

    /**
     *  Advances the simulation by an elapsedTime time.
     * <p>
     * <b>Note:</b> Large elapsedTime values can lead to instabilities. In such cases elapsedTime
     * should be subdivided into smaller time intervals and simulate() should be called
     * multiple times for each interval.
     * <p>
     * Calls to simulate() should pair with calls to fetchResults():
     *  Each fetchResults() invocation corresponds to exactly one simulate()
     *  invocation; calling simulate() twice without an intervening fetchResults()
     *  or fetchResults() twice without an intervening simulate() causes an error
     *  condition.
     * <p>
     *  scene-&gt;simulate();
     *  ...do some processing until physics is computed...
     *  scene-&gt;fetchResults();
     *  ...now results of run may be retrieved.
     * @param elapsedTime Amount of time to advance simulation by. The parameter has to be larger than 0, else the resulting behavior will be undefined. <b>Range:</b> (0, PX_MAX_F32)
     * @param completionTask if non-NULL, this task will have its refcount incremented in simulate(), then
     * decremented when the scene is ready to have fetchResults called. So the task will not run until the
     * application also calls removeReference().
     * after fetchResults returns. Must be aligned on a 16-byte boundary
     * @return True if success
     * @see #fetchResults
     * @see #checkResults
     */
    public boolean simulate(float elapsedTime, PxBaseTask completionTask) {
        checkNotNull();
        return _simulate(address, elapsedTime, completionTask.getAddress());
    }
    private static native boolean _simulate(long address, float elapsedTime, long completionTask);

    /**
     *  Advances the simulation by an elapsedTime time.
     * <p>
     * <b>Note:</b> Large elapsedTime values can lead to instabilities. In such cases elapsedTime
     * should be subdivided into smaller time intervals and simulate() should be called
     * multiple times for each interval.
     * <p>
     * Calls to simulate() should pair with calls to fetchResults():
     *  Each fetchResults() invocation corresponds to exactly one simulate()
     *  invocation; calling simulate() twice without an intervening fetchResults()
     *  or fetchResults() twice without an intervening simulate() causes an error
     *  condition.
     * <p>
     *  scene-&gt;simulate();
     *  ...do some processing until physics is computed...
     *  scene-&gt;fetchResults();
     *  ...now results of run may be retrieved.
     * @param elapsedTime Amount of time to advance simulation by. The parameter has to be larger than 0, else the resulting behavior will be undefined. <b>Range:</b> (0, PX_MAX_F32)
     * @param completionTask if non-NULL, this task will have its refcount incremented in simulate(), then
     * decremented when the scene is ready to have fetchResults called. So the task will not run until the
     * application also calls removeReference().
     * @param scratchMemBlock a memory region for physx to use for temporary data during simulation. This block may be reused by the application
     * after fetchResults returns. Must be aligned on a 16-byte boundary
     * @return True if success
     * @see #fetchResults
     * @see #checkResults
     */
    public boolean simulate(float elapsedTime, PxBaseTask completionTask, NativeObject scratchMemBlock) {
        checkNotNull();
        return _simulate(address, elapsedTime, completionTask.getAddress(), scratchMemBlock.getAddress());
    }
    private static native boolean _simulate(long address, float elapsedTime, long completionTask, long scratchMemBlock);

    /**
     *  Advances the simulation by an elapsedTime time.
     * <p>
     * <b>Note:</b> Large elapsedTime values can lead to instabilities. In such cases elapsedTime
     * should be subdivided into smaller time intervals and simulate() should be called
     * multiple times for each interval.
     * <p>
     * Calls to simulate() should pair with calls to fetchResults():
     *  Each fetchResults() invocation corresponds to exactly one simulate()
     *  invocation; calling simulate() twice without an intervening fetchResults()
     *  or fetchResults() twice without an intervening simulate() causes an error
     *  condition.
     * <p>
     *  scene-&gt;simulate();
     *  ...do some processing until physics is computed...
     *  scene-&gt;fetchResults();
     *  ...now results of run may be retrieved.
     * @param elapsedTime Amount of time to advance simulation by. The parameter has to be larger than 0, else the resulting behavior will be undefined. <b>Range:</b> (0, PX_MAX_F32)
     * @param completionTask if non-NULL, this task will have its refcount incremented in simulate(), then
     * decremented when the scene is ready to have fetchResults called. So the task will not run until the
     * application also calls removeReference().
     * @param scratchMemBlock a memory region for physx to use for temporary data during simulation. This block may be reused by the application
     * after fetchResults returns. Must be aligned on a 16-byte boundary
     * @param scratchMemBlockSize the size of the scratch memory block. Must be a multiple of 16K.
     * @return True if success
     * @see #fetchResults
     * @see #checkResults
     */
    public boolean simulate(float elapsedTime, PxBaseTask completionTask, NativeObject scratchMemBlock, int scratchMemBlockSize) {
        checkNotNull();
        return _simulate(address, elapsedTime, completionTask.getAddress(), scratchMemBlock.getAddress(), scratchMemBlockSize);
    }
    private static native boolean _simulate(long address, float elapsedTime, long completionTask, long scratchMemBlock, int scratchMemBlockSize);

    /**
     *  Advances the simulation by an elapsedTime time.
     * <p>
     * <b>Note:</b> Large elapsedTime values can lead to instabilities. In such cases elapsedTime
     * should be subdivided into smaller time intervals and simulate() should be called
     * multiple times for each interval.
     * <p>
     * Calls to simulate() should pair with calls to fetchResults():
     *  Each fetchResults() invocation corresponds to exactly one simulate()
     *  invocation; calling simulate() twice without an intervening fetchResults()
     *  or fetchResults() twice without an intervening simulate() causes an error
     *  condition.
     * <p>
     *  scene-&gt;simulate();
     *  ...do some processing until physics is computed...
     *  scene-&gt;fetchResults();
     *  ...now results of run may be retrieved.
     * @param elapsedTime Amount of time to advance simulation by. The parameter has to be larger than 0, else the resulting behavior will be undefined. <b>Range:</b> (0, PX_MAX_F32)
     * @param completionTask if non-NULL, this task will have its refcount incremented in simulate(), then
     * decremented when the scene is ready to have fetchResults called. So the task will not run until the
     * application also calls removeReference().
     * @param scratchMemBlock a memory region for physx to use for temporary data during simulation. This block may be reused by the application
     * after fetchResults returns. Must be aligned on a 16-byte boundary
     * @param scratchMemBlockSize the size of the scratch memory block. Must be a multiple of 16K.
     * @param controlSimulation if true, the scene controls its PxTaskManager simulation state. Leave
     *    true unless the application is calling the PxTaskManager start/stopSimulation() methods itself.
     * @return True if success
     * @see #fetchResults
     * @see #checkResults
     */
    public boolean simulate(float elapsedTime, PxBaseTask completionTask, NativeObject scratchMemBlock, int scratchMemBlockSize, boolean controlSimulation) {
        checkNotNull();
        return _simulate(address, elapsedTime, completionTask.getAddress(), scratchMemBlock.getAddress(), scratchMemBlockSize, controlSimulation);
    }
    private static native boolean _simulate(long address, float elapsedTime, long completionTask, long scratchMemBlock, int scratchMemBlockSize, boolean controlSimulation);

    /**
     *  Performs dynamics phase of the simulation pipeline.
     * <p>
     * <b>Note:</b> Calls to advance() should follow calls to fetchCollision(). An error message will be issued if this sequence is not followed.
     * <p>
     * decremented when the scene is ready to have fetchResults called. So the task will not run until the
     * application also calls removeReference().
     * @return True if success
     */
    public boolean advance() {
        checkNotNull();
        return _advance(address);
    }
    private static native boolean _advance(long address);

    /**
     *  Performs dynamics phase of the simulation pipeline.
     * <p>
     * <b>Note:</b> Calls to advance() should follow calls to fetchCollision(). An error message will be issued if this sequence is not followed.
     * @param completionTask if non-NULL, this task will have its refcount incremented in advance(), then
     * decremented when the scene is ready to have fetchResults called. So the task will not run until the
     * application also calls removeReference().
     * @return True if success
     */
    public boolean advance(PxBaseTask completionTask) {
        checkNotNull();
        return _advance(address, completionTask.getAddress());
    }
    private static native boolean _advance(long address, long completionTask);

    /**
     * Performs collision detection for the scene over elapsedTime
     * <p>
     * <b>Note:</b> Calls to collide() should be the first method called to simulate a frame.
     * @param elapsedTime Amount of time to advance simulation by. The parameter has to be larger than 0, else the resulting behavior will be undefined. <b>Range:</b> (0, PX_MAX_F32)
     * decremented when the scene is ready to have fetchResults called. So the task will not run until the
     * application also calls removeReference().
     * after fetchResults returns. Must be aligned on a 16-byte boundary
     * @return True if success
     */
    public boolean collide(float elapsedTime) {
        checkNotNull();
        return _collide(address, elapsedTime);
    }
    private static native boolean _collide(long address, float elapsedTime);

    /**
     * Performs collision detection for the scene over elapsedTime
     * <p>
     * <b>Note:</b> Calls to collide() should be the first method called to simulate a frame.
     * @param elapsedTime Amount of time to advance simulation by. The parameter has to be larger than 0, else the resulting behavior will be undefined. <b>Range:</b> (0, PX_MAX_F32)
     * @param completionTask if non-NULL, this task will have its refcount incremented in collide(), then
     * decremented when the scene is ready to have fetchResults called. So the task will not run until the
     * application also calls removeReference().
     * after fetchResults returns. Must be aligned on a 16-byte boundary
     * @return True if success
     */
    public boolean collide(float elapsedTime, PxBaseTask completionTask) {
        checkNotNull();
        return _collide(address, elapsedTime, completionTask.getAddress());
    }
    private static native boolean _collide(long address, float elapsedTime, long completionTask);

    /**
     * Performs collision detection for the scene over elapsedTime
     * <p>
     * <b>Note:</b> Calls to collide() should be the first method called to simulate a frame.
     * @param elapsedTime Amount of time to advance simulation by. The parameter has to be larger than 0, else the resulting behavior will be undefined. <b>Range:</b> (0, PX_MAX_F32)
     * @param completionTask if non-NULL, this task will have its refcount incremented in collide(), then
     * decremented when the scene is ready to have fetchResults called. So the task will not run until the
     * application also calls removeReference().
     * @param scratchMemBlock a memory region for physx to use for temporary data during simulation. This block may be reused by the application
     * after fetchResults returns. Must be aligned on a 16-byte boundary
     * @return True if success
     */
    public boolean collide(float elapsedTime, PxBaseTask completionTask, NativeObject scratchMemBlock) {
        checkNotNull();
        return _collide(address, elapsedTime, completionTask.getAddress(), scratchMemBlock.getAddress());
    }
    private static native boolean _collide(long address, float elapsedTime, long completionTask, long scratchMemBlock);

    /**
     * Performs collision detection for the scene over elapsedTime
     * <p>
     * <b>Note:</b> Calls to collide() should be the first method called to simulate a frame.
     * @param elapsedTime Amount of time to advance simulation by. The parameter has to be larger than 0, else the resulting behavior will be undefined. <b>Range:</b> (0, PX_MAX_F32)
     * @param completionTask if non-NULL, this task will have its refcount incremented in collide(), then
     * decremented when the scene is ready to have fetchResults called. So the task will not run until the
     * application also calls removeReference().
     * @param scratchMemBlock a memory region for physx to use for temporary data during simulation. This block may be reused by the application
     * after fetchResults returns. Must be aligned on a 16-byte boundary
     * @param scratchMemBlockSize the size of the scratch memory block. Must be a multiple of 16K.
     * @return True if success
     */
    public boolean collide(float elapsedTime, PxBaseTask completionTask, NativeObject scratchMemBlock, int scratchMemBlockSize) {
        checkNotNull();
        return _collide(address, elapsedTime, completionTask.getAddress(), scratchMemBlock.getAddress(), scratchMemBlockSize);
    }
    private static native boolean _collide(long address, float elapsedTime, long completionTask, long scratchMemBlock, int scratchMemBlockSize);

    /**
     * Performs collision detection for the scene over elapsedTime
     * <p>
     * <b>Note:</b> Calls to collide() should be the first method called to simulate a frame.
     * @param elapsedTime Amount of time to advance simulation by. The parameter has to be larger than 0, else the resulting behavior will be undefined. <b>Range:</b> (0, PX_MAX_F32)
     * @param completionTask if non-NULL, this task will have its refcount incremented in collide(), then
     * decremented when the scene is ready to have fetchResults called. So the task will not run until the
     * application also calls removeReference().
     * @param scratchMemBlock a memory region for physx to use for temporary data during simulation. This block may be reused by the application
     * after fetchResults returns. Must be aligned on a 16-byte boundary
     * @param scratchMemBlockSize the size of the scratch memory block. Must be a multiple of 16K.
     * @param controlSimulation if true, the scene controls its PxTaskManager simulation state. Leave
     *    true unless the application is calling the PxTaskManager start/stopSimulation() methods itself.
     * @return True if success
     */
    public boolean collide(float elapsedTime, PxBaseTask completionTask, NativeObject scratchMemBlock, int scratchMemBlockSize, boolean controlSimulation) {
        checkNotNull();
        return _collide(address, elapsedTime, completionTask.getAddress(), scratchMemBlock.getAddress(), scratchMemBlockSize, controlSimulation);
    }
    private static native boolean _collide(long address, float elapsedTime, long completionTask, long scratchMemBlock, int scratchMemBlockSize, boolean controlSimulation);

    /**
     * This checks to see if the simulation run has completed.
     * <p>
     * This does not cause the data available for reading to be updated with the results of the simulation, it is simply a status check.
     * The bool will allow it to either return immediately or block waiting for the condition to be met so that it can return true
     * @return True if the results are available.
     * @see #simulate
     * @see #fetchResults
     */
    public boolean checkResults() {
        checkNotNull();
        return _checkResults(address);
    }
    private static native boolean _checkResults(long address);

    /**
     * This checks to see if the simulation run has completed.
     * <p>
     * This does not cause the data available for reading to be updated with the results of the simulation, it is simply a status check.
     * The bool will allow it to either return immediately or block waiting for the condition to be met so that it can return true
     * @param block When set to true will block until the condition is met.
     * @return True if the results are available.
     * @see #simulate
     * @see #fetchResults
     */
    public boolean checkResults(boolean block) {
        checkNotNull();
        return _checkResults(address, block);
    }
    private static native boolean _checkResults(long address, boolean block);

    /**
     * This method must be called after collide() and before advance(). It will wait for the collision phase to finish. If the user makes an illegal simulation call, the SDK will issue an error
     * message.
     */
    public boolean fetchCollision() {
        checkNotNull();
        return _fetchCollision(address);
    }
    private static native boolean _fetchCollision(long address);

    /**
     * This method must be called after collide() and before advance(). It will wait for the collision phase to finish. If the user makes an illegal simulation call, the SDK will issue an error
     * message.
     * @param block When set to true will block until the condition is met, which is collision must finish running.
     */
    public boolean fetchCollision(boolean block) {
        checkNotNull();
        return _fetchCollision(address, block);
    }
    private static native boolean _fetchCollision(long address, boolean block);

    /**
     * This is the big brother to checkResults() it basically does the following:
     * <p>
     * \code
     * if ( checkResults(block) )
     * {
     *  fire appropriate callbacks
     *  swap buffers
     *  return true
     * }
     * else
     *  return false
     * <p>
     * \endcode
     * @return True if the results have been fetched.
     * @see #simulate
     * @see #checkResults
     */
    public boolean fetchResults() {
        checkNotNull();
        return _fetchResults(address);
    }
    private static native boolean _fetchResults(long address);

    /**
     * This is the big brother to checkResults() it basically does the following:
     * <p>
     * \code
     * if ( checkResults(block) )
     * {
     *  fire appropriate callbacks
     *  swap buffers
     *  return true
     * }
     * else
     *  return false
     * <p>
     * \endcode
     * @param block When set to true will block until results are available.
     * @return True if the results have been fetched.
     * @see #simulate
     * @see #checkResults
     */
    public boolean fetchResults(boolean block) {
        checkNotNull();
        return _fetchResults(address, block);
    }
    private static native boolean _fetchResults(long address, boolean block);

    /**
     * This call processes all event callbacks in parallel. It takes a continuation task, which will be executed once all callbacks have been processed.
     * <p>
     * This is a utility function to make it easier to process callbacks in parallel using the PhysX task system. It can only be used in conjunction with 
     * fetchResultsStart(...) and fetchResultsFinish(...)
     * @param continuation The task that will be executed once all callbacks have been processed.
     */
    public void processCallbacks(PxBaseTask continuation) {
        checkNotNull();
        _processCallbacks(address, continuation.getAddress());
    }
    private static native void _processCallbacks(long address, long continuation);

    /**
     * This call performs the synchronization of particle system data copies.
     */
    public void fetchResultsParticleSystem() {
        checkNotNull();
        _fetchResultsParticleSystem(address);
    }
    private static native void _fetchResultsParticleSystem(long address);

    /**
     * Clear internal buffers and free memory.
     * <p>
     * This method can be used to clear buffers and free internal memory without having to destroy the scene. Can be useful if
     * the physics data gets streamed in and a checkpoint with a clean state should be created.
     * <p>
     * <b>Note:</b> It is not allowed to call this method while the simulation is running. The call will fail.
     */
    public void flushSimulation() {
        checkNotNull();
        _flushSimulation(address);
    }
    private static native void _flushSimulation(long address);

    /**
     * Clear internal buffers and free memory.
     * <p>
     * This method can be used to clear buffers and free internal memory without having to destroy the scene. Can be useful if
     * the physics data gets streamed in and a checkpoint with a clean state should be created.
     * <p>
     * <b>Note:</b> It is not allowed to call this method while the simulation is running. The call will fail.
     * @param sendPendingReports When set to true pending reports will be sent out before the buffers get cleaned up (for instance lost touch contact/trigger reports due to deleted objects).
     */
    public void flushSimulation(boolean sendPendingReports) {
        checkNotNull();
        _flushSimulation(address, sendPendingReports);
    }
    private static native void _flushSimulation(long address, boolean sendPendingReports);

    /**
     * Sets a constant gravity for the entire scene.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running.
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the actor up automatically.
     * @param vec A new gravity vector(e.g. PxVec3(0.0f,-9.8f,0.0f) ) <b>Range:</b> force vector
     * @see #getGravity
     */
    public void setGravity(PxVec3 vec) {
        checkNotNull();
        _setGravity(address, vec.getAddress());
    }
    private static native void _setGravity(long address, long vec);

    /**
     * Retrieves the current gravity setting.
     * @return The current gravity for the scene.
     * @see #setGravity
     */
    public PxVec3 getGravity() {
        checkNotNull();
        return PxVec3.wrapPointer(_getGravity(address));
    }
    private static native long _getGravity(long address);

    /**
     * Set the bounce threshold velocity.  Collision speeds below this threshold will not cause a bounce.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running.
     * @see #getBounceThresholdVelocity
     */
    public void setBounceThresholdVelocity(float t) {
        checkNotNull();
        _setBounceThresholdVelocity(address, t);
    }
    private static native void _setBounceThresholdVelocity(long address, float t);

    /**
     * Return the bounce threshold velocity.
     * @see #setBounceThresholdVelocity
     */
    public float getBounceThresholdVelocity() {
        checkNotNull();
        return _getBounceThresholdVelocity(address);
    }
    private static native float _getBounceThresholdVelocity(long address);

    /**
     * Sets the maximum number of CCD passes
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running.
     * @param ccdMaxPasses Maximum number of CCD passes
     * @see #getCCDMaxPasses
     */
    public void setCCDMaxPasses(int ccdMaxPasses) {
        checkNotNull();
        _setCCDMaxPasses(address, ccdMaxPasses);
    }
    private static native void _setCCDMaxPasses(long address, int ccdMaxPasses);

    /**
     * Gets the maximum number of CCD passes.
     * @return The maximum number of CCD passes.
     * @see #setCCDMaxPasses
     */
    public int getCCDMaxPasses() {
        checkNotNull();
        return _getCCDMaxPasses(address);
    }
    private static native int _getCCDMaxPasses(long address);

    /**
     * Set the maximum CCD separation.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running.
     * @see #getCCDMaxSeparation
     */
    public void setCCDMaxSeparation(float t) {
        checkNotNull();
        _setCCDMaxSeparation(address, t);
    }
    private static native void _setCCDMaxSeparation(long address, float t);

    /**
     * Gets the maximum CCD separation.
     * @return The maximum CCD separation.
     * @see #setCCDMaxSeparation
     */
    public float getCCDMaxSeparation() {
        checkNotNull();
        return _getCCDMaxSeparation(address);
    }
    private static native float _getCCDMaxSeparation(long address);

    /**
     * Set the CCD threshold.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running.
     * @see #getCCDThreshold
     */
    public void setCCDThreshold(float t) {
        checkNotNull();
        _setCCDThreshold(address, t);
    }
    private static native void _setCCDThreshold(long address, float t);

    /**
     * Gets the CCD threshold.
     * @return The CCD threshold.
     * @see #setCCDThreshold
     */
    public float getCCDThreshold() {
        checkNotNull();
        return _getCCDThreshold(address);
    }
    private static native float _getCCDThreshold(long address);

    /**
     * Set the max bias coefficient.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running.
     * @see #getMaxBiasCoefficient
     */
    public void setMaxBiasCoefficient(float t) {
        checkNotNull();
        _setMaxBiasCoefficient(address, t);
    }
    private static native void _setMaxBiasCoefficient(long address, float t);

    /**
     * Gets the max bias coefficient.
     * @return The max bias coefficient.
     * @see #setMaxBiasCoefficient
     */
    public float getMaxBiasCoefficient() {
        checkNotNull();
        return _getMaxBiasCoefficient(address);
    }
    private static native float _getMaxBiasCoefficient(long address);

    /**
     * Set the friction offset threshold.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running.
     * @see #getFrictionOffsetThreshold
     */
    public void setFrictionOffsetThreshold(float t) {
        checkNotNull();
        _setFrictionOffsetThreshold(address, t);
    }
    private static native void _setFrictionOffsetThreshold(long address, float t);

    /**
     * Gets the friction offset threshold.
     * @see #setFrictionOffsetThreshold
     */
    public float getFrictionOffsetThreshold() {
        checkNotNull();
        return _getFrictionOffsetThreshold(address);
    }
    private static native float _getFrictionOffsetThreshold(long address);

    /**
     * Set the friction correlation distance.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running.
     * @see #getFrictionCorrelationDistance
     */
    public void setFrictionCorrelationDistance(float t) {
        checkNotNull();
        _setFrictionCorrelationDistance(address, t);
    }
    private static native void _setFrictionCorrelationDistance(long address, float t);

    /**
     * Gets the friction correlation distance.
     * @see #setFrictionCorrelationDistance
     */
    public float getFrictionCorrelationDistance() {
        checkNotNull();
        return _getFrictionCorrelationDistance(address);
    }
    private static native float _getFrictionCorrelationDistance(long address);

    /**
     * Return the friction model.
     */
    public PxFrictionTypeEnum getFrictionType() {
        checkNotNull();
        return PxFrictionTypeEnum.forValue(_getFrictionType(address));
    }
    private static native int _getFrictionType(long address);

    /**
     * Return the solver model.
     */
    public PxSolverTypeEnum getSolverType() {
        checkNotNull();
        return PxSolverTypeEnum.forValue(_getSolverType(address));
    }
    private static native int _getSolverType(long address);

    /**
     * Retrieves the render buffer.
     * <p>
     * This will contain the results of any active visualization for this scene.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running. Calls to this method while the simulation is running will result in undefined behaviour.
     * @return The render buffer.
     * @see physx.common.PxRenderBuffer
     */
    public PxRenderBuffer getRenderBuffer() {
        checkNotNull();
        return PxRenderBuffer.wrapPointer(_getRenderBuffer(address));
    }
    private static native long _getRenderBuffer(long address);

    /**
     * Function that lets you set debug visualization parameters.
     * <p>
     * Returns false if the value passed is out of range for usage specified by the enum.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running.
     * @param param Parameter to set. See #PxVisualizationParameter
     * @param value The value to set, see #PxVisualizationParameter for allowable values. Setting to zero disables visualization for the specified property, setting to a positive value usually enables visualization and defines the scale factor.
     * @return False if the parameter is out of range.
     * @see #getVisualizationParameter
     * @see #getRenderBuffer
     */
    public boolean setVisualizationParameter(PxVisualizationParameterEnum param, float value) {
        checkNotNull();
        return _setVisualizationParameter(address, param.value, value);
    }
    private static native boolean _setVisualizationParameter(long address, int param, float value);

    /**
     * Function that lets you query debug visualization parameters.
     * @param paramEnum The Parameter to retrieve.
     * @return The value of the parameter.
     * @see #setVisualizationParameter
     */
    public float getVisualizationParameter(PxVisualizationParameterEnum paramEnum) {
        checkNotNull();
        return _getVisualizationParameter(address, paramEnum.value);
    }
    private static native float _getVisualizationParameter(long address, int paramEnum);

    /**
     * Defines a box in world space to which visualization geometry will be (conservatively) culled. Use a non-empty culling box to enable the feature, and an empty culling box to disable it.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running.
     * @param box the box to which the geometry will be culled. Empty box to disable the feature.
     * @see #setVisualizationParameter
     * @see #getVisualizationCullingBox
     * @see #getRenderBuffer
     */
    public void setVisualizationCullingBox(PxBounds3 box) {
        checkNotNull();
        _setVisualizationCullingBox(address, box.getAddress());
    }
    private static native void _setVisualizationCullingBox(long address, long box);

    /**
     * Retrieves the visualization culling box.
     * @return the box to which the geometry will be culled.
     * @see #setVisualizationParameter
     * @see #setVisualizationCullingBox
     */
    public PxBounds3 getVisualizationCullingBox() {
        checkNotNull();
        return PxBounds3.wrapPointer(_getVisualizationCullingBox(address));
    }
    private static native long _getVisualizationCullingBox(long address);

    /**
     * Call this method to retrieve statistics for the current simulation step.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running. Calls to this method while the simulation is running will be ignored.
     * @param stats Used to retrieve statistics for the current simulation step.
     * @see PxSimulationStatistics
     */
    public void getSimulationStatistics(PxSimulationStatistics stats) {
        checkNotNull();
        _getSimulationStatistics(address, stats.getAddress());
    }
    private static native void _getSimulationStatistics(long address, long stats);

    /**
     * Returns broad-phase type.
     * @return Broad-phase type
     */
    public PxBroadPhaseTypeEnum getBroadPhaseType() {
        checkNotNull();
        return PxBroadPhaseTypeEnum.forValue(_getBroadPhaseType(address));
    }
    private static native int _getBroadPhaseType(long address);

    /**
     * Gets broad-phase caps.
     * @param caps Broad-phase caps
     * @return True if success
     */
    public boolean getBroadPhaseCaps(PxBroadPhaseCaps caps) {
        checkNotNull();
        return _getBroadPhaseCaps(address, caps.getAddress());
    }
    private static native boolean _getBroadPhaseCaps(long address, long caps);

    /**
     * Returns number of regions currently registered in the broad-phase.
     * @return Number of regions
     */
    public int getNbBroadPhaseRegions() {
        checkNotNull();
        return _getNbBroadPhaseRegions(address);
    }
    private static native int _getNbBroadPhaseRegions(long address);

    /**
     * Gets broad-phase regions.
     * @param userBuffer Returned broad-phase regions
     * @param bufferSize Size of userBuffer
     * @return Number of written out regions
     */
    public int getBroadPhaseRegions(PxBroadPhaseRegionInfo userBuffer, int bufferSize) {
        checkNotNull();
        return _getBroadPhaseRegions(address, userBuffer.getAddress(), bufferSize);
    }
    private static native int _getBroadPhaseRegions(long address, long userBuffer, int bufferSize);

    /**
     * Gets broad-phase regions.
     * @param userBuffer Returned broad-phase regions
     * @param bufferSize Size of userBuffer
     * @param startIndex Index of first desired region, in [0 ; getNbRegions()[
     * @return Number of written out regions
     */
    public int getBroadPhaseRegions(PxBroadPhaseRegionInfo userBuffer, int bufferSize, int startIndex) {
        checkNotNull();
        return _getBroadPhaseRegions(address, userBuffer.getAddress(), bufferSize, startIndex);
    }
    private static native int _getBroadPhaseRegions(long address, long userBuffer, int bufferSize, int startIndex);

    /**
     * Adds a new broad-phase region.
     * <p>
     * The bounds for the new region must be non-empty, otherwise an error occurs and the call is ignored.
     * <p>
     * Note that by default, objects already existing in the SDK that might touch this region will not be automatically
     * added to the region. In other words the newly created region will be empty, and will only be populated with new
     * objects when they are added to the simulation, or with already existing objects when they are updated.
     * <p>
     * It is nonetheless possible to override this default behavior and let the SDK populate the new region automatically
     * with already existing objects overlapping the incoming region. This has a cost though, and it should only be used
     * when the game can not guarantee that all objects within the new region will be added to the simulation after the
     * region itself.
     * <p>
     * Objects automatically move from one region to another during their lifetime. The system keeps tracks of what
     * regions a given object is in. It is legal for an object to be in an arbitrary number of regions. However if an
     * object leaves all regions, or is created outside of all regions, several things happen:
     *  - collisions get disabled for this object
     *  - if a PxBroadPhaseCallback object is provided, an "out-of-bounds" event is generated via that callback
     *  - if a PxBroadPhaseCallback object is not provided, a warning/error message is sent to the error stream
     * <p>
     * If an object goes out-of-bounds and user deletes it during the same frame, neither the out-of-bounds event nor the
     * error message is generated.
     * @param region   User-provided region data
     * @return Handle for newly created region, or 0xffffffff in case of failure.
     * @see PxBroadPhaseRegion
     */
    public int addBroadPhaseRegion(PxBroadPhaseRegion region) {
        checkNotNull();
        return _addBroadPhaseRegion(address, region.getAddress());
    }
    private static native int _addBroadPhaseRegion(long address, long region);

    /**
     * Adds a new broad-phase region.
     * <p>
     * The bounds for the new region must be non-empty, otherwise an error occurs and the call is ignored.
     * <p>
     * Note that by default, objects already existing in the SDK that might touch this region will not be automatically
     * added to the region. In other words the newly created region will be empty, and will only be populated with new
     * objects when they are added to the simulation, or with already existing objects when they are updated.
     * <p>
     * It is nonetheless possible to override this default behavior and let the SDK populate the new region automatically
     * with already existing objects overlapping the incoming region. This has a cost though, and it should only be used
     * when the game can not guarantee that all objects within the new region will be added to the simulation after the
     * region itself.
     * <p>
     * Objects automatically move from one region to another during their lifetime. The system keeps tracks of what
     * regions a given object is in. It is legal for an object to be in an arbitrary number of regions. However if an
     * object leaves all regions, or is created outside of all regions, several things happen:
     *  - collisions get disabled for this object
     *  - if a PxBroadPhaseCallback object is provided, an "out-of-bounds" event is generated via that callback
     *  - if a PxBroadPhaseCallback object is not provided, a warning/error message is sent to the error stream
     * <p>
     * If an object goes out-of-bounds and user deletes it during the same frame, neither the out-of-bounds event nor the
     * error message is generated.
     * @param region   User-provided region data
     * @param populateRegion Automatically populate new region with already existing objects overlapping it
     * @return Handle for newly created region, or 0xffffffff in case of failure.
     * @see PxBroadPhaseRegion
     */
    public int addBroadPhaseRegion(PxBroadPhaseRegion region, boolean populateRegion) {
        checkNotNull();
        return _addBroadPhaseRegion(address, region.getAddress(), populateRegion);
    }
    private static native int _addBroadPhaseRegion(long address, long region, boolean populateRegion);

    /**
     * Removes a new broad-phase region.
     * <p>
     * If the region still contains objects, and if those objects do not overlap any region any more, they are not
     * automatically removed from the simulation. Instead, the PxBroadPhaseCallback::onObjectOutOfBounds notification
     * is used for each object. Users are responsible for removing the objects from the simulation if this is the
     * desired behavior.
     * <p>
     * If the handle is invalid, or if a valid handle is removed twice, an error message is sent to the error stream.
     * @param handle Region's handle, as returned by PxScene::addBroadPhaseRegion.
     * @return True if success
     */
    public boolean removeBroadPhaseRegion(int handle) {
        checkNotNull();
        return _removeBroadPhaseRegion(address, handle);
    }
    private static native boolean _removeBroadPhaseRegion(long address, int handle);

    /**
     * Lock the scene for reading from the calling thread.
     * <p>
     * When the PxSceneFlag::eREQUIRE_RW_LOCK flag is enabled lockRead() must be 
     * called before any read calls are made on the scene.
     * <p>
     * Multiple threads may read at the same time, no threads may read while a thread is writing.
     * If a call to lockRead() is made while another thread is holding a write lock 
     * then the calling thread will be blocked until the writing thread calls unlockWrite().
     * <p>
     * <b>Note:</b> Lock upgrading is *not* supported, that means it is an error to
     * call lockRead() followed by lockWrite().
     * <p>
     * <b>Note:</b> Recursive locking is supported but each lockRead() call must be paired with an unlockRead().
     */
    public void lockRead() {
        checkNotNull();
        _lockRead(address);
    }
    private static native void _lockRead(long address);

    /**
     * Lock the scene for reading from the calling thread.
     * <p>
     * When the PxSceneFlag::eREQUIRE_RW_LOCK flag is enabled lockRead() must be 
     * called before any read calls are made on the scene.
     * <p>
     * Multiple threads may read at the same time, no threads may read while a thread is writing.
     * If a call to lockRead() is made while another thread is holding a write lock 
     * then the calling thread will be blocked until the writing thread calls unlockWrite().
     * <p>
     * <b>Note:</b> Lock upgrading is *not* supported, that means it is an error to
     * call lockRead() followed by lockWrite().
     * <p>
     * <b>Note:</b> Recursive locking is supported but each lockRead() call must be paired with an unlockRead().
     * <p>
     * \param file String representing the calling file, for debug purposes
     */
    public void lockRead(String file) {
        checkNotNull();
        _lockRead(address, file);
    }
    private static native void _lockRead(long address, String file);

    /**
     * Lock the scene for reading from the calling thread.
     * <p>
     * When the PxSceneFlag::eREQUIRE_RW_LOCK flag is enabled lockRead() must be 
     * called before any read calls are made on the scene.
     * <p>
     * Multiple threads may read at the same time, no threads may read while a thread is writing.
     * If a call to lockRead() is made while another thread is holding a write lock 
     * then the calling thread will be blocked until the writing thread calls unlockWrite().
     * <p>
     * <b>Note:</b> Lock upgrading is *not* supported, that means it is an error to
     * call lockRead() followed by lockWrite().
     * <p>
     * <b>Note:</b> Recursive locking is supported but each lockRead() call must be paired with an unlockRead().
     * <p>
     * \param file String representing the calling file, for debug purposes
     * \param line The source file line number, for debug purposes
     */
    public void lockRead(String file, int line) {
        checkNotNull();
        _lockRead(address, file, line);
    }
    private static native void _lockRead(long address, String file, int line);

    /**
     * Unlock the scene from reading.
     * <p>
     * <b>Note:</b> Each unlockRead() must be paired with a lockRead() from the same thread.
     */
    public void unlockRead() {
        checkNotNull();
        _unlockRead(address);
    }
    private static native void _unlockRead(long address);

    /**
     * Lock the scene for writing from this thread.
     * <p>
     * When the PxSceneFlag::eREQUIRE_RW_LOCK flag is enabled lockWrite() must be 
     * called before any write calls are made on the scene.
     * <p>
     * Only one thread may write at a time and no threads may read while a thread is writing.
     * If a call to lockWrite() is made and there are other threads reading then the 
     * calling thread will be blocked until the readers complete.
     * <p>
     * Writers have priority. If a thread is blocked waiting to write then subsequent calls to 
     * lockRead() from other threads will be blocked until the writer completes.
     * <p>
     * <b>Note:</b> If multiple threads are waiting to write then the thread that is first
     * granted access depends on OS scheduling.
     * <p>
     * <b>Note:</b> Recursive locking is supported but each lockWrite() call must be paired 
     * with an unlockWrite(). 
     * <p>
     * <b>Note:</b> If a thread has already locked the scene for writing then it may call
     * lockRead().
     */
    public void lockWrite() {
        checkNotNull();
        _lockWrite(address);
    }
    private static native void _lockWrite(long address);

    /**
     * Lock the scene for writing from this thread.
     * <p>
     * When the PxSceneFlag::eREQUIRE_RW_LOCK flag is enabled lockWrite() must be 
     * called before any write calls are made on the scene.
     * <p>
     * Only one thread may write at a time and no threads may read while a thread is writing.
     * If a call to lockWrite() is made and there are other threads reading then the 
     * calling thread will be blocked until the readers complete.
     * <p>
     * Writers have priority. If a thread is blocked waiting to write then subsequent calls to 
     * lockRead() from other threads will be blocked until the writer completes.
     * <p>
     * <b>Note:</b> If multiple threads are waiting to write then the thread that is first
     * granted access depends on OS scheduling.
     * <p>
     * <b>Note:</b> Recursive locking is supported but each lockWrite() call must be paired 
     * with an unlockWrite(). 
     * <p>
     * <b>Note:</b> If a thread has already locked the scene for writing then it may call
     * lockRead().
     * <p>
     * \param file String representing the calling file, for debug purposes
     */
    public void lockWrite(String file) {
        checkNotNull();
        _lockWrite(address, file);
    }
    private static native void _lockWrite(long address, String file);

    /**
     * Lock the scene for writing from this thread.
     * <p>
     * When the PxSceneFlag::eREQUIRE_RW_LOCK flag is enabled lockWrite() must be 
     * called before any write calls are made on the scene.
     * <p>
     * Only one thread may write at a time and no threads may read while a thread is writing.
     * If a call to lockWrite() is made and there are other threads reading then the 
     * calling thread will be blocked until the readers complete.
     * <p>
     * Writers have priority. If a thread is blocked waiting to write then subsequent calls to 
     * lockRead() from other threads will be blocked until the writer completes.
     * <p>
     * <b>Note:</b> If multiple threads are waiting to write then the thread that is first
     * granted access depends on OS scheduling.
     * <p>
     * <b>Note:</b> Recursive locking is supported but each lockWrite() call must be paired 
     * with an unlockWrite(). 
     * <p>
     * <b>Note:</b> If a thread has already locked the scene for writing then it may call
     * lockRead().
     * <p>
     * \param file String representing the calling file, for debug purposes
     * \param line The source file line number, for debug purposes
     */
    public void lockWrite(String file, int line) {
        checkNotNull();
        _lockWrite(address, file, line);
    }
    private static native void _lockWrite(long address, String file, int line);

    /**
     * Unlock the scene from writing.
     * <p>
     * <b>Note:</b> Each unlockWrite() must be paired with a lockWrite() from the same thread.
     */
    public void unlockWrite() {
        checkNotNull();
        _unlockWrite(address);
    }
    private static native void _unlockWrite(long address);

    /**
     * set the cache blocks that can be used during simulate(). 
     * <p>
     * Each frame the simulation requires memory to store contact, friction, and contact cache data. This memory is used in blocks of 16K.
     * Each frame the blocks used by the previous frame are freed, and may be retrieved by the application using PxScene::flushSimulation()
     * <p>
     * This call will force allocation of cache blocks if the numBlocks parameter is greater than the currently allocated number
     * of blocks, and less than the max16KContactDataBlocks parameter specified at scene creation time.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running.
     * @param numBlocks The number of blocks to allocate. 
     * @see #flushSimulation
     * @see #getNbContactDataBlocksUsed
     * @see #getMaxNbContactDataBlocksUsed
     */
    public void setNbContactDataBlocks(int numBlocks) {
        checkNotNull();
        _setNbContactDataBlocks(address, numBlocks);
    }
    private static native void _setNbContactDataBlocks(long address, int numBlocks);

    /**
     * get the number of cache blocks currently used by the scene 
     * <p>
     * This function may not be called while the scene is simulating
     * @return the number of cache blocks currently used by the scene
     * @see #flushSimulation
     * @see #setNbContactDataBlocks
     * @see #getMaxNbContactDataBlocksUsed
     */
    public int getNbContactDataBlocksUsed() {
        checkNotNull();
        return _getNbContactDataBlocksUsed(address);
    }
    private static native int _getNbContactDataBlocksUsed(long address);

    /**
     * get the maximum number of cache blocks used by the scene 
     * <p>
     * This function may not be called while the scene is simulating
     * @return the maximum number of cache blocks everused by the scene
     * @see #flushSimulation
     * @see #setNbContactDataBlocks
     * @see #getNbContactDataBlocksUsed
     */
    public int getMaxNbContactDataBlocksUsed() {
        checkNotNull();
        return _getMaxNbContactDataBlocksUsed(address);
    }
    private static native int _getMaxNbContactDataBlocksUsed(long address);

    /**
     * Return the value of PxSceneDesc::contactReportStreamBufferSize that was set when creating the scene with PxPhysics::createScene
     */
    public int getContactReportStreamBufferSize() {
        checkNotNull();
        return _getContactReportStreamBufferSize(address);
    }
    private static native int _getContactReportStreamBufferSize(long address);

    /**
     * Sets the number of actors required to spawn a separate rigid body solver thread.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running.
     * @param solverBatchSize Number of actors required to spawn a separate rigid body solver thread.
     * @see #getSolverBatchSize
     */
    public void setSolverBatchSize(int solverBatchSize) {
        checkNotNull();
        _setSolverBatchSize(address, solverBatchSize);
    }
    private static native void _setSolverBatchSize(long address, int solverBatchSize);

    /**
     * Retrieves the number of actors required to spawn a separate rigid body solver thread.
     * @return Current number of actors required to spawn a separate rigid body solver thread.
     * @see #setSolverBatchSize
     */
    public int getSolverBatchSize() {
        checkNotNull();
        return _getSolverBatchSize(address);
    }
    private static native int _getSolverBatchSize(long address);

    /**
     * Sets the number of articulations required to spawn a separate rigid body solver thread.
     * <p>
     * <b>Note:</b> Do not use this method while the simulation is running.
     * @param solverBatchSize Number of articulations required to spawn a separate rigid body solver thread.
     * @see #getSolverArticulationBatchSize
     */
    public void setSolverArticulationBatchSize(int solverBatchSize) {
        checkNotNull();
        _setSolverArticulationBatchSize(address, solverBatchSize);
    }
    private static native void _setSolverArticulationBatchSize(long address, int solverBatchSize);

    /**
     * Retrieves the number of articulations required to spawn a separate rigid body solver thread.
     * @return Current number of articulations required to spawn a separate rigid body solver thread.
     * @see #setSolverArticulationBatchSize
     */
    public int getSolverArticulationBatchSize() {
        checkNotNull();
        return _getSolverArticulationBatchSize(address);
    }
    private static native int _getSolverArticulationBatchSize(long address);

    /**
     * Deletes the scene.
     * <p>
     * Removes any actors and constraint shaders from this scene
     * (if the user hasn't already done so).
     * <p>
     * Be sure to not keep a reference to this object after calling release.
     * Avoid release calls while the scene is simulating (in between simulate() and fetchResults() calls).
     * @see PxPhysics#createScene
     */
    public void release() {
        checkNotNull();
        _release(address);
    }
    private static native void _release(long address);

    /**
     * Sets a scene flag. You can only set one flag at a time.
     * <p>
     * <b>Note:</b> Not all flags are mutable and changing some will result in an error. Please check #PxSceneFlag to see which flags can be changed.
     */
    public void setFlag(PxSceneFlagEnum flag, boolean value) {
        checkNotNull();
        _setFlag(address, flag.value, value);
    }
    private static native void _setFlag(long address, int flag, boolean value);

    /**
     * Get the scene flags.
     * @return The scene flags. See #PxSceneFlag
     */
    public PxSceneFlags getFlags() {
        checkNotNull();
        return PxSceneFlags.wrapPointer(_getFlags(address));
    }
    private static native long _getFlags(long address);

    /**
     * Set new scene limits. 
     * <p>
     * <b>Note:</b> Increase the maximum capacity of various data structures in the scene. The new capacities will be 
     * at least as large as required to deal with the objects currently in the scene. Further, these values 
     * are for preallocation and do not represent hard limits.
     * @param limits Scene limits.
     * @see PxSceneLimits
     */
    public void setLimits(PxSceneLimits limits) {
        checkNotNull();
        _setLimits(address, limits.getAddress());
    }
    private static native void _setLimits(long address, long limits);

    /**
     * Get current scene limits.
     * @return Current scene limits.
     * @see PxSceneLimits
     */
    public PxSceneLimits getLimits() {
        checkNotNull();
        return PxSceneLimits.wrapPointer(_getLimits(address));
    }
    private static native long _getLimits(long address);

    /**
     * Call this method to retrieve the Physics SDK.
     * @return The physics SDK this scene is associated with.
     * @see PxPhysics
     */
    public PxPhysics getPhysics() {
        checkNotNull();
        return PxPhysics.wrapPointer(_getPhysics(address));
    }
    private static native long _getPhysics(long address);

    /**
     * Retrieves the scene's internal timestamp, increased each time a simulation step is completed.
     * @return scene timestamp
     */
    public int getTimestamp() {
        checkNotNull();
        return _getTimestamp(address);
    }
    private static native int _getTimestamp(long address);

}
