package physx.physics;

import physx.NativeObject;

/**
 * Class used to retrieve limits(e.g. maximum number of bodies) for a scene. The limits
 * are used as a hint to the size of the scene, not as a hard limit (i.e. it will be possible
 * to create more objects than specified in the scene limits).
 * <p>
 * 0 indicates no limit. Using limits allows the SDK to preallocate various arrays, leading to
 * less re-allocations and faster code at runtime.
 */
public class PxSceneLimits extends NativeObject {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxSceneLimits wrapPointer(long address) {
        return address != 0L ? new PxSceneLimits(address) : null;
    }
    
    public static PxSceneLimits arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxSceneLimits(long address) {
        super(address);
    }

    // Placed Constructors

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @return Stack allocated object of PxSceneLimits
     */
    public static PxSceneLimits createAt(long address) {
        __placement_new_PxSceneLimits(address);
        PxSceneLimits createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @return Stack allocated object of PxSceneLimits
     */
    public static <T> PxSceneLimits createAt(T allocator, Allocator<T> allocate) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxSceneLimits(address);
        PxSceneLimits createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxSceneLimits(long address);

    // Constructors

    /**
     * constructor sets to default 
     */
    public PxSceneLimits() {
        address = _PxSceneLimits();
    }
    private static native long _PxSceneLimits();

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * Expected maximum number of actors
     */
    public int getMaxNbActors() {
        checkNotNull();
        return _getMaxNbActors(address);
    }
    private static native int _getMaxNbActors(long address);

    /**
     * Expected maximum number of actors
     */
    public void setMaxNbActors(int value) {
        checkNotNull();
        _setMaxNbActors(address, value);
    }
    private static native void _setMaxNbActors(long address, int value);

    /**
     * Expected maximum number of dynamic rigid bodies
     */
    public int getMaxNbBodies() {
        checkNotNull();
        return _getMaxNbBodies(address);
    }
    private static native int _getMaxNbBodies(long address);

    /**
     * Expected maximum number of dynamic rigid bodies
     */
    public void setMaxNbBodies(int value) {
        checkNotNull();
        _setMaxNbBodies(address, value);
    }
    private static native void _setMaxNbBodies(long address, int value);

    /**
     * Expected maximum number of static shapes
     */
    public int getMaxNbStaticShapes() {
        checkNotNull();
        return _getMaxNbStaticShapes(address);
    }
    private static native int _getMaxNbStaticShapes(long address);

    /**
     * Expected maximum number of static shapes
     */
    public void setMaxNbStaticShapes(int value) {
        checkNotNull();
        _setMaxNbStaticShapes(address, value);
    }
    private static native void _setMaxNbStaticShapes(long address, int value);

    /**
     * Expected maximum number of dynamic shapes
     */
    public int getMaxNbDynamicShapes() {
        checkNotNull();
        return _getMaxNbDynamicShapes(address);
    }
    private static native int _getMaxNbDynamicShapes(long address);

    /**
     * Expected maximum number of dynamic shapes
     */
    public void setMaxNbDynamicShapes(int value) {
        checkNotNull();
        _setMaxNbDynamicShapes(address, value);
    }
    private static native void _setMaxNbDynamicShapes(long address, int value);

    /**
     * Expected maximum number of aggregates
     */
    public int getMaxNbAggregates() {
        checkNotNull();
        return _getMaxNbAggregates(address);
    }
    private static native int _getMaxNbAggregates(long address);

    /**
     * Expected maximum number of aggregates
     */
    public void setMaxNbAggregates(int value) {
        checkNotNull();
        _setMaxNbAggregates(address, value);
    }
    private static native void _setMaxNbAggregates(long address, int value);

    /**
     * Expected maximum number of constraint shaders
     */
    public int getMaxNbConstraints() {
        checkNotNull();
        return _getMaxNbConstraints(address);
    }
    private static native int _getMaxNbConstraints(long address);

    /**
     * Expected maximum number of constraint shaders
     */
    public void setMaxNbConstraints(int value) {
        checkNotNull();
        _setMaxNbConstraints(address, value);
    }
    private static native void _setMaxNbConstraints(long address, int value);

    /**
     * Expected maximum number of broad-phase regions
     */
    public int getMaxNbRegions() {
        checkNotNull();
        return _getMaxNbRegions(address);
    }
    private static native int _getMaxNbRegions(long address);

    /**
     * Expected maximum number of broad-phase regions
     */
    public void setMaxNbRegions(int value) {
        checkNotNull();
        _setMaxNbRegions(address, value);
    }
    private static native void _setMaxNbRegions(long address, int value);

    /**
     * Expected maximum number of broad-phase overlaps
     */
    public int getMaxNbBroadPhaseOverlaps() {
        checkNotNull();
        return _getMaxNbBroadPhaseOverlaps(address);
    }
    private static native int _getMaxNbBroadPhaseOverlaps(long address);

    /**
     * Expected maximum number of broad-phase overlaps
     */
    public void setMaxNbBroadPhaseOverlaps(int value) {
        checkNotNull();
        _setMaxNbBroadPhaseOverlaps(address, value);
    }
    private static native void _setMaxNbBroadPhaseOverlaps(long address, int value);

    // Functions

    /**
     * (re)sets the structure to the default
     */
    public void setToDefault() {
        checkNotNull();
        _setToDefault(address);
    }
    private static native void _setToDefault(long address);

    /**
     * Returns true if the descriptor is valid.
     * @return true if the current settings are valid.
     */
    public boolean isValid() {
        checkNotNull();
        return _isValid(address);
    }
    private static native boolean _isValid(long address);

}
