package physx.physics;

import physx.NativeObject;
import physx.common.PxBounds3;
import physx.common.PxTransform;
import physx.common.PxVec3;
import physx.geometry.PxGeometry;

/**
 * utility functions for use with PxShape
 * @see PxShape
 */
public class PxShapeExt extends NativeObject {

    protected PxShapeExt() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxShapeExt wrapPointer(long address) {
        return address != 0L ? new PxShapeExt(address) : null;
    }
    
    public static PxShapeExt arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxShapeExt(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Retrieves the world space pose of the shape.
     * @param shape The shape for which to get the global pose.
     * @param actor The actor to which the shape is attached
     * @return Global pose of shape.
     */
    public static PxTransform getGlobalPose(PxShape shape, PxRigidActor actor) {
        return PxTransform.wrapPointer(_getGlobalPose(shape.getAddress(), actor.getAddress()));
    }
    private static native long _getGlobalPose(long shape, long actor);

    /**
     * Raycast test against the shape.
     * @param shape the shape
     * @param actor the actor to which the shape is attached
     * @param rayOrigin The origin of the ray to test the geometry object against
     * @param rayDir The direction of the ray to test the geometry object against
     * @param maxDist Maximum ray length
     * @param hitFlags Specify which properties per hit should be computed and written to result hit array. Combination of #PxHitFlag flags
     * @param maxHits max number of returned hits = size of 'rayHits' buffer
     * @param rayHits Raycast hits information
     * @return Number of hits between the ray and the shape
     * @see PxRaycastHit
     * @see physx.common.PxTransform
     */
    public static int raycast(PxShape shape, PxRigidActor actor, PxVec3 rayOrigin, PxVec3 rayDir, float maxDist, PxHitFlags hitFlags, int maxHits, PxRaycastHit rayHits) {
        return _raycast(shape.getAddress(), actor.getAddress(), rayOrigin.getAddress(), rayDir.getAddress(), maxDist, hitFlags.getAddress(), maxHits, rayHits.getAddress());
    }
    private static native int _raycast(long shape, long actor, long rayOrigin, long rayDir, float maxDist, long hitFlags, int maxHits, long rayHits);

    /**
     * Test overlap between the shape and a geometry object
     * @param shape the shape
     * @param actor the actor to which the shape is attached
     * @param otherGeom The other geometry object to test overlap with
     * @param otherGeomPose Pose of the other geometry object
     * @return True if the shape overlaps the geometry object
     * @see physx.geometry.PxGeometry
     * @see physx.common.PxTransform
     */
    public static boolean overlap(PxShape shape, PxRigidActor actor, PxGeometry otherGeom, PxTransform otherGeomPose) {
        return _overlap(shape.getAddress(), actor.getAddress(), otherGeom.getAddress(), otherGeomPose.getAddress());
    }
    private static native boolean _overlap(long shape, long actor, long otherGeom, long otherGeomPose);

    /**
     * Sweep a geometry object against the shape.
     * <p>
     * Currently only box, sphere, capsule and convex mesh shapes are supported, i.e. the swept geometry object must be one of those types.
     * @param shape the shape
     * @param actor the actor to which the shape is attached
     * @param unitDir Normalized direction along which the geometry object should be swept.
     * @param distance Sweep distance. Needs to be larger than 0.
     * @param otherGeom The geometry object to sweep against the shape
     * @param otherGeomPose Pose of the geometry object
     * @param sweepHit The sweep hit information. Only valid if this method returns true.
     * @param hitFlags Specify which properties per hit should be computed and written to result hit array. Combination of #PxHitFlag flags
     * @return True if the swept geometry object hits the shape
     * @see physx.geometry.PxGeometry
     * @see physx.common.PxTransform
     * @see PxSweepHit
     */
    public static boolean sweep(PxShape shape, PxRigidActor actor, PxVec3 unitDir, float distance, PxGeometry otherGeom, PxTransform otherGeomPose, PxSweepHit sweepHit, PxHitFlags hitFlags) {
        return _sweep(shape.getAddress(), actor.getAddress(), unitDir.getAddress(), distance, otherGeom.getAddress(), otherGeomPose.getAddress(), sweepHit.getAddress(), hitFlags.getAddress());
    }
    private static native boolean _sweep(long shape, long actor, long unitDir, float distance, long otherGeom, long otherGeomPose, long sweepHit, long hitFlags);

    /**
     * Retrieves the axis aligned bounding box enclosing the shape.
     * @return The shape's bounding box.
     * @param shape the shape
     * @param actor the actor to which the shape is attached
     * @see physx.common.PxBounds3
     */
    public static PxBounds3 getWorldBounds(PxShape shape, PxRigidActor actor) {
        return PxBounds3.wrapPointer(_getWorldBounds(shape.getAddress(), actor.getAddress()));
    }
    private static native long _getWorldBounds(long shape, long actor);

    /**
     * Retrieves the axis aligned bounding box enclosing the shape.
     * @return The shape's bounding box.
     * @param shape the shape
     * @param actor the actor to which the shape is attached
     * @param inflation  Scale factor for computed world bounds. Box extents are multiplied by this value.
     * @see physx.common.PxBounds3
     */
    public static PxBounds3 getWorldBounds(PxShape shape, PxRigidActor actor, float inflation) {
        return PxBounds3.wrapPointer(_getWorldBounds(shape.getAddress(), actor.getAddress(), inflation));
    }
    private static native long _getWorldBounds(long shape, long actor, float inflation);

}
