package physx.common;

import physx.NativeObject;

/**
 * Class representing 3D range or axis aligned bounding box.
 * <p>
 * Stored as minimum and maximum extent corners. Alternate representation
 * would be center and dimensions.
 * May be empty or nonempty. For nonempty bounds, minimum &lt;= maximum has to hold for all axes.
 * Empty bounds have to be represented as minimum = PX_MAX_BOUNDS_EXTENTS and maximum = -PX_MAX_BOUNDS_EXTENTS for all
 * axes.
 * All other representations are invalid and the behavior is undefined.
 */
public class PxBounds3 extends NativeObject {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxBounds3 wrapPointer(long address) {
        return address != 0L ? new PxBounds3(address) : null;
    }
    
    public static PxBounds3 arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxBounds3(long address) {
        super(address);
    }

    // Placed Constructors

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @return Stack allocated object of PxBounds3
     */
    public static PxBounds3 createAt(long address) {
        __placement_new_PxBounds3(address);
        PxBounds3 createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @return Stack allocated object of PxBounds3
     */
    public static <T> PxBounds3 createAt(T allocator, Allocator<T> allocate) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxBounds3(address);
        PxBounds3 createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxBounds3(long address);

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @param minimum WebIDL type: {@link PxVec3} [Const, Ref]
     * @param maximum WebIDL type: {@link PxVec3} [Const, Ref]
     * @return Stack allocated object of PxBounds3
     */
    public static PxBounds3 createAt(long address, PxVec3 minimum, PxVec3 maximum) {
        __placement_new_PxBounds3(address, minimum.getAddress(), maximum.getAddress());
        PxBounds3 createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @param minimum   WebIDL type: {@link PxVec3} [Const, Ref]
     * @param maximum   WebIDL type: {@link PxVec3} [Const, Ref]
     * @return Stack allocated object of PxBounds3
     */
    public static <T> PxBounds3 createAt(T allocator, Allocator<T> allocate, PxVec3 minimum, PxVec3 maximum) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxBounds3(address, minimum.getAddress(), maximum.getAddress());
        PxBounds3 createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxBounds3(long address, long minimum, long maximum);

    // Constructors

    /**
     * Default constructor, not performing any initialization for performance reason.
     * \remark Use empty() function below to construct empty bounds.
     */
    public PxBounds3() {
        address = _PxBounds3();
    }
    private static native long _PxBounds3();

    /**
     * Construct from two bounding points
     */
    public PxBounds3(PxVec3 minimum, PxVec3 maximum) {
        address = _PxBounds3(minimum.getAddress(), maximum.getAddress());
    }
    private static native long _PxBounds3(long minimum, long maximum);

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * @return WebIDL type: {@link PxVec3} [Value]
     */
    public PxVec3 getMinimum() {
        checkNotNull();
        return PxVec3.wrapPointer(_getMinimum(address));
    }
    private static native long _getMinimum(long address);

    /**
     * @param value WebIDL type: {@link PxVec3} [Value]
     */
    public void setMinimum(PxVec3 value) {
        checkNotNull();
        _setMinimum(address, value.getAddress());
    }
    private static native void _setMinimum(long address, long value);

    /**
     */
    public PxVec3 getMaximum() {
        checkNotNull();
        return PxVec3.wrapPointer(_getMaximum(address));
    }
    private static native long _getMaximum(long address);

    /**
     */
    public void setMaximum(PxVec3 value) {
        checkNotNull();
        _setMaximum(address, value.getAddress());
    }
    private static native void _setMaximum(long address, long value);

    // Functions

    /**
     * Sets empty to true
     */
    public void setEmpty() {
        checkNotNull();
        _setEmpty(address);
    }
    private static native void _setEmpty(long address);

    /**
     * Sets the bounds to maximum size [-PX_MAX_BOUNDS_EXTENTS, PX_MAX_BOUNDS_EXTENTS].
     */
    public void setMaximal() {
        checkNotNull();
        _setMaximal(address);
    }
    private static native void _setMaximal(long address);

    /**
     * expands the volume to include v
     * \param v Point to expand to.
     */
    public void include(PxVec3 v) {
        checkNotNull();
        _include(address, v.getAddress());
    }
    private static native void _include(long address, long v);

    /**
     * @return WebIDL type: boolean
     */
    public boolean isEmpty() {
        checkNotNull();
        return _isEmpty(address);
    }
    private static native boolean _isEmpty(long address);

    /**
     * indicates whether the intersection of this and b is empty or not.
     * \param b Bounds to test for intersection.
     */
    public boolean intersects(PxBounds3 b) {
        checkNotNull();
        return _intersects(address, b.getAddress());
    }
    private static native boolean _intersects(long address, long b);

    /**
     * computes the 1D-intersection between two AABBs, on a given axis.
     * \param axis the axis (0, 1, 2)
     */
    public boolean intersects1D(PxBounds3 b, int axis) {
        checkNotNull();
        return _intersects1D(address, b.getAddress(), axis);
    }
    private static native boolean _intersects1D(long address, long b, int axis);

    /**
     * indicates if these bounds contain v.
     * \param v Point to test against bounds.
     */
    public boolean contains(PxVec3 v) {
        checkNotNull();
        return _contains(address, v.getAddress());
    }
    private static native boolean _contains(long address, long v);

    /**
     * checks a box is inside another box.
     * \param box  the other AABB
     */
    public boolean isInside(PxBounds3 box) {
        checkNotNull();
        return _isInside(address, box.getAddress());
    }
    private static native boolean _isInside(long address, long box);

    /**
     * returns the center of this axis aligned box.
     */
    public PxVec3 getCenter() {
        checkNotNull();
        return PxVec3.wrapPointer(_getCenter(address));
    }
    private static native long _getCenter(long address);

    /**
     * returns the dimensions (width/height/depth) of this axis aligned box.
     */
    public PxVec3 getDimensions() {
        checkNotNull();
        return PxVec3.wrapPointer(_getDimensions(address));
    }
    private static native long _getDimensions(long address);

    /**
     * returns the extents, which are half of the width/height/depth.
     */
    public PxVec3 getExtents() {
        checkNotNull();
        return PxVec3.wrapPointer(_getExtents(address));
    }
    private static native long _getExtents(long address);

    /**
     * scales the AABB.
     * <p>
     * This version is safe to call for empty bounds.
     * <p>
     * \param scale Factor to scale AABB by.
     */
    public void scaleSafe(float scale) {
        checkNotNull();
        _scaleSafe(address, scale);
    }
    private static native void _scaleSafe(long address, float scale);

    /**
     * scales the AABB.
     * <p>
     * Calling this method for empty bounds leads to undefined behavior. Use #scaleSafe() instead.
     * <p>
     * \param scale Factor to scale AABB by.
     */
    public void scaleFast(float scale) {
        checkNotNull();
        _scaleFast(address, scale);
    }
    private static native void _scaleFast(long address, float scale);

    /**
     * fattens the AABB in all 3 dimensions by the given distance.
     * <p>
     * This version is safe to call for empty bounds.
     */
    public void fattenSafe(float distance) {
        checkNotNull();
        _fattenSafe(address, distance);
    }
    private static native void _fattenSafe(long address, float distance);

    /**
     * fattens the AABB in all 3 dimensions by the given distance.
     * <p>
     * Calling this method for empty bounds leads to undefined behavior. Use #fattenSafe() instead.
     */
    public void fattenFast(float distance) {
        checkNotNull();
        _fattenFast(address, distance);
    }
    private static native void _fattenFast(long address, float distance);

    /**
     * checks that the AABB values are not NaN
     */
    public boolean isFinite() {
        checkNotNull();
        return _isFinite(address);
    }
    private static native boolean _isFinite(long address);

    /**
     * checks that the AABB values describe a valid configuration.
     */
    public boolean isValid() {
        checkNotNull();
        return _isValid(address);
    }
    private static native boolean _isValid(long address);

}
