package physx.geometry;

import physx.common.PxBounds3;
import physx.common.PxRefCounted;
import physx.common.PxVec3;
import physx.support.PxU8ConstPtr;

/**
 * A convex mesh.
 * <p>
 * Internally represented as a list of convex polygons. The number
 * of polygons is limited to 256.
 * <p>
 * To avoid duplicating data when you have several instances of a particular
 * mesh positioned differently, you do not use this class to represent a
 * convex object directly. Instead, you create an instance of this mesh via
 * the PxConvexMeshGeometry and PxShape classes.
 * <p>
 * <h3>Creation</h3>
 * <p>
 * To create an instance of this class call PxPhysics::createConvexMesh(),
 * and PxConvexMesh::release() to delete it. This is only possible
 * once you have released all of its #PxShape instances.
 * <p>
 * <h3>Visualizations:</h3>
 * \li #PxVisualizationParameter::eCOLLISION_AABBS
 * \li #PxVisualizationParameter::eCOLLISION_SHAPES
 * \li #PxVisualizationParameter::eCOLLISION_AXES
 * \li #PxVisualizationParameter::eCOLLISION_FNORMALS
 * \li #PxVisualizationParameter::eCOLLISION_EDGES
 * @see physx.cooking.PxConvexMeshDesc
 */
public class PxConvexMesh extends PxRefCounted {

    protected PxConvexMesh() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxConvexMesh wrapPointer(long address) {
        return address != 0L ? new PxConvexMesh(address) : null;
    }
    
    public static PxConvexMesh arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxConvexMesh(long address) {
        super(address);
    }

    // Functions

    /**
     * Returns the number of vertices.
     * @return Number of vertices.
     * @see #getVertices
     */
    public int getNbVertices() {
        checkNotNull();
        return _getNbVertices(address);
    }
    private static native int _getNbVertices(long address);

    /**
     * Returns the vertices.
     * @return Array of vertices.
     * @see #getNbVertices
     */
    public PxVec3 getVertices() {
        checkNotNull();
        return PxVec3.wrapPointer(_getVertices(address));
    }
    private static native long _getVertices(long address);

    /**
     * Returns the index buffer.
     * @return Index buffer.
     * @see #getNbPolygons
     * @see #getPolygonData
     */
    public PxU8ConstPtr getIndexBuffer() {
        checkNotNull();
        return PxU8ConstPtr.wrapPointer(_getIndexBuffer(address));
    }
    private static native long _getIndexBuffer(long address);

    /**
     * Returns the number of polygons.
     * @return Number of polygons.
     * @see #getIndexBuffer
     * @see #getPolygonData
     */
    public int getNbPolygons() {
        checkNotNull();
        return _getNbPolygons(address);
    }
    private static native int _getNbPolygons(long address);

    /**
     * Returns the polygon data.
     * @param index Polygon index in [0 ; getNbPolygons()[.
     * @param data Polygon data.
     * @return True if success.
     * @see #getIndexBuffer
     * @see #getNbPolygons
     */
    public boolean getPolygonData(int index, PxHullPolygon data) {
        checkNotNull();
        return _getPolygonData(address, index, data.getAddress());
    }
    private static native boolean _getPolygonData(long address, int index, long data);

    /**
     * Returns the local-space (vertex space) AABB from the convex mesh.
     * @return local-space bounds
     */
    public PxBounds3 getLocalBounds() {
        checkNotNull();
        return PxBounds3.wrapPointer(_getLocalBounds(address));
    }
    private static native long _getLocalBounds(long address);

    /**
     * This method decides whether a convex mesh is gpu compatible. If the total number of vertices are more than 64 or any number of vertices in a polygon is more than 32, or
     * convex hull data was not cooked with GPU data enabled during cooking or was loaded from a serialized collection, the convex hull is incompatible with GPU collision detection. Otherwise
     * it is compatible.
     * @return True if the convex hull is gpu compatible
     */
    public boolean isGpuCompatible() {
        checkNotNull();
        return _isGpuCompatible(address);
    }
    private static native boolean _isGpuCompatible(long address);

}
