package physx.particles;

import physx.NativeObject;
import physx.PlatformChecks;
import physx.support.PxU32Ptr;

/**
 * Container class for a GPU particle system. Used to communicate particle system parameters and simulation state 
 * between the internal SDK simulation and the particle system callbacks.
 * <p>
 * See #PxPBDParticleSystem, #PxParticleSystemCallback.
 */
public class PxGpuParticleSystem extends NativeObject {

    static {
        PlatformChecks.requirePlatform(3, "physx.particles.PxGpuParticleSystem");
    }

    protected PxGpuParticleSystem() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxGpuParticleSystem wrapPointer(long address) {
        return address != 0L ? new PxGpuParticleSystem(address) : null;
    }
    
    public static PxGpuParticleSystem arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxGpuParticleSystem(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * GPU pointer to unsorted particle phase array. See #PxParticlePhaseFlag.
     */
    public PxU32Ptr getMUnsortedPhaseArray() {
        checkNotNull();
        return PxU32Ptr.wrapPointer(_getMUnsortedPhaseArray(address));
    }
    private static native long _getMUnsortedPhaseArray(long address);

    /**
     * GPU pointer to unsorted particle phase array. See #PxParticlePhaseFlag.
     */
    public void setMUnsortedPhaseArray(PxU32Ptr value) {
        checkNotNull();
        _setMUnsortedPhaseArray(address, value.getAddress());
    }
    private static native void _setMUnsortedPhaseArray(long address, long value);

    /**
     * GPU pointer to sorted particle phase array
     */
    public PxU32Ptr getMSortedPhaseArray() {
        checkNotNull();
        return PxU32Ptr.wrapPointer(_getMSortedPhaseArray(address));
    }
    private static native long _getMSortedPhaseArray(long address);

    /**
     * GPU pointer to sorted particle phase array
     */
    public void setMSortedPhaseArray(PxU32Ptr value) {
        checkNotNull();
        _setMSortedPhaseArray(address, value.getAddress());
    }
    private static native void _setMSortedPhaseArray(long address, long value);

    /**
     * GPU pointer to the mapping from unsortedParticle ID to sorted particle ID
     */
    public PxU32Ptr getMUnsortedToSortedMapping() {
        checkNotNull();
        return PxU32Ptr.wrapPointer(_getMUnsortedToSortedMapping(address));
    }
    private static native long _getMUnsortedToSortedMapping(long address);

    /**
     * GPU pointer to the mapping from unsortedParticle ID to sorted particle ID
     */
    public void setMUnsortedToSortedMapping(PxU32Ptr value) {
        checkNotNull();
        _setMUnsortedToSortedMapping(address, value.getAddress());
    }
    private static native void _setMUnsortedToSortedMapping(long address, long value);

    /**
     * GPU pointer to the mapping from sorted particle ID to unsorted particle ID
     */
    public PxU32Ptr getMSortedToUnsortedMapping() {
        checkNotNull();
        return PxU32Ptr.wrapPointer(_getMSortedToUnsortedMapping(address));
    }
    private static native long _getMSortedToUnsortedMapping(long address);

    /**
     * GPU pointer to the mapping from sorted particle ID to unsorted particle ID
     */
    public void setMSortedToUnsortedMapping(PxU32Ptr value) {
        checkNotNull();
        _setMSortedToUnsortedMapping(address, value.getAddress());
    }
    private static native void _setMSortedToUnsortedMapping(long address, long value);

    /**
     * Per-particle neighborhood count
     */
    public PxU32Ptr getMParticleSelfCollisionCount() {
        checkNotNull();
        return PxU32Ptr.wrapPointer(_getMParticleSelfCollisionCount(address));
    }
    private static native long _getMParticleSelfCollisionCount(long address);

    /**
     * Per-particle neighborhood count
     */
    public void setMParticleSelfCollisionCount(PxU32Ptr value) {
        checkNotNull();
        _setMParticleSelfCollisionCount(address, value.getAddress());
    }
    private static native void _setMParticleSelfCollisionCount(long address, long value);

    /**
     * Set of sorted particle indices per neighbor
     */
    public PxU32Ptr getMCollisionIndex() {
        checkNotNull();
        return PxU32Ptr.wrapPointer(_getMCollisionIndex(address));
    }
    private static native long _getMCollisionIndex(long address);

    /**
     * Set of sorted particle indices per neighbor
     */
    public void setMCollisionIndex(PxU32Ptr value) {
        checkNotNull();
        _setMCollisionIndex(address, value.getAddress());
    }
    private static native void _setMCollisionIndex(long address, long value);

    // Functions

    /**
     * Returns the number of cells of the background simulation grid.
     * @return PxU32 the number of cells. 
     */
    public int getNumCells() {
        checkNotNull();
        return _getNumCells(address);
    }
    private static native int _getNumCells(long address);

}
