package physx.particles;

import physx.PlatformChecks;
import physx.common.PxVec4;

/**
 * A particle buffer used to simulate diffuse particles.
 * <p>
 * See #PxPhysics::createParticleAndDiffuseBuffer.
 */
public class PxParticleAndDiffuseBuffer extends PxParticleBuffer {

    static {
        PlatformChecks.requirePlatform(3, "physx.particles.PxParticleAndDiffuseBuffer");
    }

    protected PxParticleAndDiffuseBuffer() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxParticleAndDiffuseBuffer wrapPointer(long address) {
        return address != 0L ? new PxParticleAndDiffuseBuffer(address) : null;
    }
    
    public static PxParticleAndDiffuseBuffer arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxParticleAndDiffuseBuffer(long address) {
        super(address);
    }

    // Functions

    /**
     * Get a device buffer of positions and remaining lifetimes for the diffuse particles.
     * @return A device buffer containing positions and lifetimes of diffuse particles packed as PxVec4(pos.x, pos.y, pos.z, lifetime).
     */
    public PxVec4 getDiffusePositionLifeTime() {
        checkNotNull();
        return PxVec4.wrapPointer(_getDiffusePositionLifeTime(address));
    }
    private static native long _getDiffusePositionLifeTime(long address);

    /**
     * Get number of currently active diffuse particles.
     * @return The number of currently active diffuse particles.
     */
    public int getNbActiveDiffuseParticles() {
        checkNotNull();
        return _getNbActiveDiffuseParticles(address);
    }
    private static native int _getNbActiveDiffuseParticles(long address);

    /**
     * Set the maximum possible number of diffuse particles for this buffer.
     * @param maxActiveDiffuseParticles the maximum number of active diffuse particles.
     * <p>
     * <b>Note:</b> Must be in the range [0, PxParticleAndDiffuseBuffer::getMaxDiffuseParticles()]
     */
    public void setMaxActiveDiffuseParticles(int maxActiveDiffuseParticles) {
        checkNotNull();
        _setMaxActiveDiffuseParticles(address, maxActiveDiffuseParticles);
    }
    private static native void _setMaxActiveDiffuseParticles(long address, int maxActiveDiffuseParticles);

    /**
     * Get maximum possible number of diffuse particles.
     * @return The maximum possible number diffuse particles.
     */
    public int getMaxDiffuseParticles() {
        checkNotNull();
        return _getMaxDiffuseParticles(address);
    }
    private static native int _getMaxDiffuseParticles(long address);

    /**
     * Set the parameters for diffuse particle simulation.
     * @param params The diffuse particle parameters.
     * <p>
     * See #PxDiffuseParticleParams
     */
    public void setDiffuseParticleParams(PxDiffuseParticleParams params) {
        checkNotNull();
        _setDiffuseParticleParams(address, params.getAddress());
    }
    private static native void _setDiffuseParticleParams(long address, long params);

    /**
     * Get the parameters currently used for diffuse particle simulation.
     * @return A PxDiffuseParticleParams structure.
     */
    public PxDiffuseParticleParams getDiffuseParticleParams() {
        checkNotNull();
        return PxDiffuseParticleParams.wrapPointer(_getDiffuseParticleParams(address));
    }
    private static native long _getDiffuseParticleParams(long address);

}
