package physx.physics;

import physx.common.PxBaseTask;

/**
 * Traditional SQ system for PxScene.
 * <p>
 * Methods defined here are only available through the traditional PxScene API.
 * Thus PxSceneSQSystem effectively captures the scene-query related part of the PxScene API.
 * @see PxScene
 * @see PxSceneQuerySystemBase
 */
public class PxSceneSQSystem extends PxSceneQuerySystemBase {

    protected PxSceneSQSystem() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxSceneSQSystem wrapPointer(long address) {
        return address != 0L ? new PxSceneSQSystem(address) : null;
    }
    
    public static PxSceneSQSystem arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxSceneSQSystem(long address) {
        super(address);
    }

    // Functions

    /**
     * Sets scene query update mode 
     * @param updateMode Scene query update mode.
     */
    public void setSceneQueryUpdateMode(PxSceneQueryUpdateModeEnum updateMode) {
        checkNotNull();
        _setSceneQueryUpdateMode(address, updateMode.value);
    }
    private static native void _setSceneQueryUpdateMode(long address, int updateMode);

    /**
     * Gets scene query update mode 
     * @return Current scene query update mode.
     */
    public PxSceneQueryUpdateModeEnum getSceneQueryUpdateMode() {
        checkNotNull();
        return PxSceneQueryUpdateModeEnum.forValue(_getSceneQueryUpdateMode(address));
    }
    private static native int _getSceneQueryUpdateMode(long address);

    /**
     * Retrieves the scene's internal scene query timestamp, increased each time a change to the
     * static scene query structure is performed.
     * @return scene query static timestamp
     */
    public int getSceneQueryStaticTimestamp() {
        checkNotNull();
        return _getSceneQueryStaticTimestamp(address);
    }
    private static native int _getSceneQueryStaticTimestamp(long address);

    /**
     * Flushes any changes to the scene query representation.
     */
    public void flushQueryUpdates() {
        checkNotNull();
        _flushQueryUpdates(address);
    }
    private static native void _flushQueryUpdates(long address);

    /**
     * Forces dynamic trees to be immediately rebuilt.
     * @param rebuildStaticStructure True to rebuild the dynamic tree containing static objects
     * @param rebuildDynamicStructure True to rebuild the dynamic tree containing dynamic objects
     */
    public void forceDynamicTreeRebuild(boolean rebuildStaticStructure, boolean rebuildDynamicStructure) {
        checkNotNull();
        _forceDynamicTreeRebuild(address, rebuildStaticStructure, rebuildDynamicStructure);
    }
    private static native void _forceDynamicTreeRebuild(long address, boolean rebuildStaticStructure, boolean rebuildDynamicStructure);

    /**
     * Return the value of PxSceneQueryDesc::staticStructure that was set when creating the scene with PxPhysics::createScene
     */
    public PxPruningStructureTypeEnum getStaticStructure() {
        checkNotNull();
        return PxPruningStructureTypeEnum.forValue(_getStaticStructure(address));
    }
    private static native int _getStaticStructure(long address);

    /**
     * Return the value of PxSceneQueryDesc::dynamicStructure that was set when creating the scene with PxPhysics::createScene
     */
    public PxPruningStructureTypeEnum getDynamicStructure() {
        checkNotNull();
        return PxPruningStructureTypeEnum.forValue(_getDynamicStructure(address));
    }
    private static native int _getDynamicStructure(long address);

    /**
     * Executes scene queries update tasks.
     * <p>
     * This function will refit dirty shapes within the pruner and will execute a task to build a new AABB tree, which is
     * build on a different thread. The new AABB tree is built based on the dynamic tree rebuild hint rate. Once
     * the new tree is ready it will be commited in next fetchQueries call, which must be called after.
     * <p>
     * This function is equivalent to the following PxSceneQuerySystem calls:
     * Synchronous calls:
     *  - PxSceneQuerySystemBase::flushUpdates()
     *  - handle0 = PxSceneQuerySystem::prepareSceneQueryBuildStep(PX_SCENE_PRUNER_STATIC)
     *  - handle1 = PxSceneQuerySystem::prepareSceneQueryBuildStep(PX_SCENE_PRUNER_DYNAMIC)
     * Asynchronous calls:
     *  - PxSceneQuerySystem::sceneQueryBuildStep(handle0);
     *  - PxSceneQuerySystem::sceneQueryBuildStep(handle1);
     * <p>
     * This function is part of the PxSceneSQSystem interface because it uses the PxScene task system under the hood. But
     * it calls PxSceneQuerySystem functions, which are independent from this system and could be called in a similar
     * fashion by a separate, possibly user-defined task manager.
     * <p>
     * <b>Note:</b> If PxSceneQueryUpdateMode::eBUILD_DISABLED_COMMIT_DISABLED is used, it is required to update the scene queries
     * using this function.
     * <p>
     * decremented when the scene is ready to have fetchQueries called. So the task will not run until the
     * application also calls removeReference().
     * true unless the application is calling the PxTaskManager start/stopSimulation() methods itself.
     */
    public void sceneQueriesUpdate() {
        checkNotNull();
        _sceneQueriesUpdate(address);
    }
    private static native void _sceneQueriesUpdate(long address);

    /**
     * Executes scene queries update tasks.
     * <p>
     * This function will refit dirty shapes within the pruner and will execute a task to build a new AABB tree, which is
     * build on a different thread. The new AABB tree is built based on the dynamic tree rebuild hint rate. Once
     * the new tree is ready it will be commited in next fetchQueries call, which must be called after.
     * <p>
     * This function is equivalent to the following PxSceneQuerySystem calls:
     * Synchronous calls:
     *  - PxSceneQuerySystemBase::flushUpdates()
     *  - handle0 = PxSceneQuerySystem::prepareSceneQueryBuildStep(PX_SCENE_PRUNER_STATIC)
     *  - handle1 = PxSceneQuerySystem::prepareSceneQueryBuildStep(PX_SCENE_PRUNER_DYNAMIC)
     * Asynchronous calls:
     *  - PxSceneQuerySystem::sceneQueryBuildStep(handle0);
     *  - PxSceneQuerySystem::sceneQueryBuildStep(handle1);
     * <p>
     * This function is part of the PxSceneSQSystem interface because it uses the PxScene task system under the hood. But
     * it calls PxSceneQuerySystem functions, which are independent from this system and could be called in a similar
     * fashion by a separate, possibly user-defined task manager.
     * <p>
     * <b>Note:</b> If PxSceneQueryUpdateMode::eBUILD_DISABLED_COMMIT_DISABLED is used, it is required to update the scene queries
     * using this function.
     * @param completionTask if non-NULL, this task will have its refcount incremented in sceneQueryUpdate(), then
     * decremented when the scene is ready to have fetchQueries called. So the task will not run until the
     * application also calls removeReference().
     * true unless the application is calling the PxTaskManager start/stopSimulation() methods itself.
     */
    public void sceneQueriesUpdate(PxBaseTask completionTask) {
        checkNotNull();
        _sceneQueriesUpdate(address, completionTask.getAddress());
    }
    private static native void _sceneQueriesUpdate(long address, long completionTask);

    /**
     * Executes scene queries update tasks.
     * <p>
     * This function will refit dirty shapes within the pruner and will execute a task to build a new AABB tree, which is
     * build on a different thread. The new AABB tree is built based on the dynamic tree rebuild hint rate. Once
     * the new tree is ready it will be commited in next fetchQueries call, which must be called after.
     * <p>
     * This function is equivalent to the following PxSceneQuerySystem calls:
     * Synchronous calls:
     *  - PxSceneQuerySystemBase::flushUpdates()
     *  - handle0 = PxSceneQuerySystem::prepareSceneQueryBuildStep(PX_SCENE_PRUNER_STATIC)
     *  - handle1 = PxSceneQuerySystem::prepareSceneQueryBuildStep(PX_SCENE_PRUNER_DYNAMIC)
     * Asynchronous calls:
     *  - PxSceneQuerySystem::sceneQueryBuildStep(handle0);
     *  - PxSceneQuerySystem::sceneQueryBuildStep(handle1);
     * <p>
     * This function is part of the PxSceneSQSystem interface because it uses the PxScene task system under the hood. But
     * it calls PxSceneQuerySystem functions, which are independent from this system and could be called in a similar
     * fashion by a separate, possibly user-defined task manager.
     * <p>
     * <b>Note:</b> If PxSceneQueryUpdateMode::eBUILD_DISABLED_COMMIT_DISABLED is used, it is required to update the scene queries
     * using this function.
     * @param completionTask if non-NULL, this task will have its refcount incremented in sceneQueryUpdate(), then
     * decremented when the scene is ready to have fetchQueries called. So the task will not run until the
     * application also calls removeReference().
     * @param controlSimulation if true, the scene controls its PxTaskManager simulation state. Leave
     * true unless the application is calling the PxTaskManager start/stopSimulation() methods itself.
     */
    public void sceneQueriesUpdate(PxBaseTask completionTask, boolean controlSimulation) {
        checkNotNull();
        _sceneQueriesUpdate(address, completionTask.getAddress(), controlSimulation);
    }
    private static native void _sceneQueriesUpdate(long address, long completionTask, boolean controlSimulation);

    /**
     * This checks to see if the scene queries update has completed.
     * <p>
     * This does not cause the data available for reading to be updated with the results of the scene queries update, it is simply a status check.
     * The bool will allow it to either return immediately or block waiting for the condition to be met so that it can return true
     * @return True if the results are available.
     * @see #sceneQueriesUpdate
     */
    public boolean checkQueries() {
        checkNotNull();
        return _checkQueries(address);
    }
    private static native boolean _checkQueries(long address);

    /**
     * This checks to see if the scene queries update has completed.
     * <p>
     * This does not cause the data available for reading to be updated with the results of the scene queries update, it is simply a status check.
     * The bool will allow it to either return immediately or block waiting for the condition to be met so that it can return true
     * @param block When set to true will block until the condition is met.
     * @return True if the results are available.
     * @see #sceneQueriesUpdate
     */
    public boolean checkQueries(boolean block) {
        checkNotNull();
        return _checkQueries(address, block);
    }
    private static native boolean _checkQueries(long address, boolean block);

    /**
     * This method must be called after sceneQueriesUpdate. It will wait for the scene queries update to finish. If the user makes an illegal scene queries update call, 
     * the SDK will issue an error message.
     * <p>
     * If a new AABB tree build finished, then during fetchQueries the current tree within the pruning structure is swapped with the new tree. 
     */
    public boolean fetchQueries() {
        checkNotNull();
        return _fetchQueries(address);
    }
    private static native boolean _fetchQueries(long address);

    /**
     * This method must be called after sceneQueriesUpdate. It will wait for the scene queries update to finish. If the user makes an illegal scene queries update call, 
     * the SDK will issue an error message.
     * <p>
     * If a new AABB tree build finished, then during fetchQueries the current tree within the pruning structure is swapped with the new tree. 
     * @param block When set to true will block until the condition is met, which is tree built task must finish running.
     */
    public boolean fetchQueries(boolean block) {
        checkNotNull();
        return _fetchQueries(address, block);
    }
    private static native boolean _fetchQueries(long address, boolean block);

}
