package physx.character;

import physx.NativeObject;
import physx.common.PxVec3;
import physx.physics.PxMaterial;

/**
 * Descriptor class for a character controller.
 * @see PxBoxController
 * @see PxCapsuleController
 */
public class PxControllerDesc extends NativeObject {

    protected PxControllerDesc() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxControllerDesc wrapPointer(long address) {
        return address != 0L ? new PxControllerDesc(address) : null;
    }
    
    public static PxControllerDesc arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxControllerDesc(long address) {
        super(address);
    }

    // Attributes

    /**
     * The position of the character
     * <p>
     * <b>Note:</b> The character's initial position must be such that it does not overlap the static geometry.
     * <p>
     * <b>Default:</b> Zero
     */
    public PxExtendedVec3 getPosition() {
        checkNotNull();
        return PxExtendedVec3.wrapPointer(_getPosition(address));
    }
    private static native long _getPosition(long address);

    /**
     * The position of the character
     * <p>
     * <b>Note:</b> The character's initial position must be such that it does not overlap the static geometry.
     * <p>
     * <b>Default:</b> Zero
     */
    public void setPosition(PxExtendedVec3 value) {
        checkNotNull();
        _setPosition(address, value.getAddress());
    }
    private static native void _setPosition(long address, long value);

    /**
     * Specifies the 'up' direction
     * <p>
     * In order to provide stepping functionality the SDK must be informed about the up direction.
     * <p>
     * <b>Default:</b> (0, 1, 0)
     */
    public PxVec3 getUpDirection() {
        checkNotNull();
        return PxVec3.wrapPointer(_getUpDirection(address));
    }
    private static native long _getUpDirection(long address);

    /**
     * Specifies the 'up' direction
     * <p>
     * In order to provide stepping functionality the SDK must be informed about the up direction.
     * <p>
     * <b>Default:</b> (0, 1, 0)
     */
    public void setUpDirection(PxVec3 value) {
        checkNotNull();
        _setUpDirection(address, value.getAddress());
    }
    private static native void _setUpDirection(long address, long value);

    /**
     * The maximum slope which the character can walk up.
     * <p>
     * In general it is desirable to limit where the character can walk, in particular it is unrealistic
     * for the character to be able to climb arbitary slopes.
     * <p>
     * The limit is expressed as the cosine of desired limit angle. A value of 0 disables this feature.
     * <p>
     * \warning It is currently enabled for static actors only (not for dynamic/kinematic actors), and not supported for spheres or capsules.
     * <p>
     * <b>Default:</b> 0.707
     */
    public float getSlopeLimit() {
        checkNotNull();
        return _getSlopeLimit(address);
    }
    private static native float _getSlopeLimit(long address);

    /**
     * The maximum slope which the character can walk up.
     * <p>
     * In general it is desirable to limit where the character can walk, in particular it is unrealistic
     * for the character to be able to climb arbitary slopes.
     * <p>
     * The limit is expressed as the cosine of desired limit angle. A value of 0 disables this feature.
     * <p>
     * \warning It is currently enabled for static actors only (not for dynamic/kinematic actors), and not supported for spheres or capsules.
     * <p>
     * <b>Default:</b> 0.707
     */
    public void setSlopeLimit(float value) {
        checkNotNull();
        _setSlopeLimit(address, value);
    }
    private static native void _setSlopeLimit(long address, float value);

    /**
     * Height of invisible walls created around non-walkable triangles
     * <p>
     * The library can automatically create invisible walls around non-walkable triangles defined
     * by the 'slopeLimit' parameter. This defines the height of those walls. If it is 0.0, then
     * no extra triangles are created.
     * <p>
     * <b>Default:</b> 0.0
     */
    public float getInvisibleWallHeight() {
        checkNotNull();
        return _getInvisibleWallHeight(address);
    }
    private static native float _getInvisibleWallHeight(long address);

    /**
     * Height of invisible walls created around non-walkable triangles
     * <p>
     * The library can automatically create invisible walls around non-walkable triangles defined
     * by the 'slopeLimit' parameter. This defines the height of those walls. If it is 0.0, then
     * no extra triangles are created.
     * <p>
     * <b>Default:</b> 0.0
     */
    public void setInvisibleWallHeight(float value) {
        checkNotNull();
        _setInvisibleWallHeight(address, value);
    }
    private static native void _setInvisibleWallHeight(long address, float value);

    /**
     * Maximum height a jumping character can reach
     * <p>
     * This is only used if invisible walls are created ('invisibleWallHeight' is non zero).
     * <p>
     * When a character jumps, the non-walkable triangles he might fly over are not found
     * by the collision queries (since the character's bounding volume does not touch them).
     * Thus those non-walkable triangles do not create invisible walls, and it is possible
     * for a jumping character to land on a non-walkable triangle, while he wouldn't have
     * reached that place by just walking.
     * <p>
     * The 'maxJumpHeight' variable is used to extend the size of the collision volume
     * downward. This way, all the non-walkable triangles are properly found by the collision
     * queries and it becomes impossible to 'jump over' invisible walls.
     * <p>
     * If the character in your game can not jump, it is safe to use 0.0 here. Otherwise it
     * is best to keep this value as small as possible, since a larger collision volume
     * means more triangles to process.
     * <p>
     * <b>Default:</b> 0.0
     */
    public float getMaxJumpHeight() {
        checkNotNull();
        return _getMaxJumpHeight(address);
    }
    private static native float _getMaxJumpHeight(long address);

    /**
     * Maximum height a jumping character can reach
     * <p>
     * This is only used if invisible walls are created ('invisibleWallHeight' is non zero).
     * <p>
     * When a character jumps, the non-walkable triangles he might fly over are not found
     * by the collision queries (since the character's bounding volume does not touch them).
     * Thus those non-walkable triangles do not create invisible walls, and it is possible
     * for a jumping character to land on a non-walkable triangle, while he wouldn't have
     * reached that place by just walking.
     * <p>
     * The 'maxJumpHeight' variable is used to extend the size of the collision volume
     * downward. This way, all the non-walkable triangles are properly found by the collision
     * queries and it becomes impossible to 'jump over' invisible walls.
     * <p>
     * If the character in your game can not jump, it is safe to use 0.0 here. Otherwise it
     * is best to keep this value as small as possible, since a larger collision volume
     * means more triangles to process.
     * <p>
     * <b>Default:</b> 0.0
     */
    public void setMaxJumpHeight(float value) {
        checkNotNull();
        _setMaxJumpHeight(address, value);
    }
    private static native void _setMaxJumpHeight(long address, float value);

    /**
     * The contact offset used by the controller.
     * <p>
     * Specifies a skin around the object within which contacts will be generated.
     * Use it to avoid numerical precision issues.
     * <p>
     * This is dependant on the scale of the users world, but should be a small, positive 
     * non zero value.
     * <p>
     * <b>Default:</b> 0.1
     */
    public float getContactOffset() {
        checkNotNull();
        return _getContactOffset(address);
    }
    private static native float _getContactOffset(long address);

    /**
     * The contact offset used by the controller.
     * <p>
     * Specifies a skin around the object within which contacts will be generated.
     * Use it to avoid numerical precision issues.
     * <p>
     * This is dependant on the scale of the users world, but should be a small, positive 
     * non zero value.
     * <p>
     * <b>Default:</b> 0.1
     */
    public void setContactOffset(float value) {
        checkNotNull();
        _setContactOffset(address, value);
    }
    private static native void _setContactOffset(long address, float value);

    /**
     * Defines the maximum height of an obstacle which the character can climb.
     * <p>
     * A small value will mean that the character gets stuck and cannot walk up stairs etc, 
     * a value which is too large will mean that the character can climb over unrealistically 
     * high obstacles.
     * <p>
     * <b>Default:</b> 0.5
     */
    public float getStepOffset() {
        checkNotNull();
        return _getStepOffset(address);
    }
    private static native float _getStepOffset(long address);

    /**
     * Defines the maximum height of an obstacle which the character can climb.
     * <p>
     * A small value will mean that the character gets stuck and cannot walk up stairs etc, 
     * a value which is too large will mean that the character can climb over unrealistically 
     * high obstacles.
     * <p>
     * <b>Default:</b> 0.5
     */
    public void setStepOffset(float value) {
        checkNotNull();
        _setStepOffset(address, value);
    }
    private static native void _setStepOffset(long address, float value);

    /**
     * Density of underlying kinematic actor
     * <p>
     * The CCT creates a PhysX's kinematic actor under the hood. This controls its density.
     * <p>
     * <b>Default:</b> 10.0
     */
    public float getDensity() {
        checkNotNull();
        return _getDensity(address);
    }
    private static native float _getDensity(long address);

    /**
     * Density of underlying kinematic actor
     * <p>
     * The CCT creates a PhysX's kinematic actor under the hood. This controls its density.
     * <p>
     * <b>Default:</b> 10.0
     */
    public void setDensity(float value) {
        checkNotNull();
        _setDensity(address, value);
    }
    private static native void _setDensity(long address, float value);

    /**
     * Scale coefficient for underlying kinematic actor
     * <p>
     * The CCT creates a PhysX's kinematic actor under the hood. This controls its scale factor.
     * This should be a number a bit smaller than 1.0.
     * <p>
     * This scale factor affects how the character interacts with dynamic rigid bodies around it (e.g. pushing them, etc).
     * <p>
     * With a scale factor &lt; 1, the underlying kinematic actor will not touch surrounding rigid bodies - they will
     * only interact with the character controller's shapes (capsules or boxes), and users will have full control
     * over the interactions (i.e. they will have to push the objects with explicit forces themselves).
     * <p>
     * With a scale factor &gt;=1, the underlying kinematic actor will touch and push surrounding rigid bodies based
     * on PhysX's computations, as if there would be no character controller involved. This works fine except
     * when you push objects into a wall. PhysX has no control over kinematic actors (since they are kinematic)
     * so they would freely push dynamic objects into walls, and make them tunnel / explode / behave badly.
     * <p>
     * With a smaller kinematic actor however, the character controller's swept shape touches dynamic rigid bodies
     * first, and can apply forces to them to move them away (or not, depending on what the gameplay needs).
     * Meanwhile the character controller's swept shape itself is stopped by these dynamic bodies.
     * <p>
     * Setting the scale factor to 1 could still work, but it is unreliable. Depending on FPU accuracy you could
     * end up with either the CCT's volume or the underlying kinematic actor touching the dynamic bodies first,
     * and this could change from one moment to the next.
     * <p>
     * <b>Default:</b> 0.8
     */
    public float getScaleCoeff() {
        checkNotNull();
        return _getScaleCoeff(address);
    }
    private static native float _getScaleCoeff(long address);

    /**
     * Scale coefficient for underlying kinematic actor
     * <p>
     * The CCT creates a PhysX's kinematic actor under the hood. This controls its scale factor.
     * This should be a number a bit smaller than 1.0.
     * <p>
     * This scale factor affects how the character interacts with dynamic rigid bodies around it (e.g. pushing them, etc).
     * <p>
     * With a scale factor &lt; 1, the underlying kinematic actor will not touch surrounding rigid bodies - they will
     * only interact with the character controller's shapes (capsules or boxes), and users will have full control
     * over the interactions (i.e. they will have to push the objects with explicit forces themselves).
     * <p>
     * With a scale factor &gt;=1, the underlying kinematic actor will touch and push surrounding rigid bodies based
     * on PhysX's computations, as if there would be no character controller involved. This works fine except
     * when you push objects into a wall. PhysX has no control over kinematic actors (since they are kinematic)
     * so they would freely push dynamic objects into walls, and make them tunnel / explode / behave badly.
     * <p>
     * With a smaller kinematic actor however, the character controller's swept shape touches dynamic rigid bodies
     * first, and can apply forces to them to move them away (or not, depending on what the gameplay needs).
     * Meanwhile the character controller's swept shape itself is stopped by these dynamic bodies.
     * <p>
     * Setting the scale factor to 1 could still work, but it is unreliable. Depending on FPU accuracy you could
     * end up with either the CCT's volume or the underlying kinematic actor touching the dynamic bodies first,
     * and this could change from one moment to the next.
     * <p>
     * <b>Default:</b> 0.8
     */
    public void setScaleCoeff(float value) {
        checkNotNull();
        _setScaleCoeff(address, value);
    }
    private static native void _setScaleCoeff(long address, float value);

    /**
     * Cached volume growth
     * <p>
     * Amount of space around the controller we cache to improve performance. This is a scale factor
     * that should be higher than 1.0f but not too big, ideally lower than 2.0f.
     * <p>
     * <b>Default:</b> 1.5
     */
    public float getVolumeGrowth() {
        checkNotNull();
        return _getVolumeGrowth(address);
    }
    private static native float _getVolumeGrowth(long address);

    /**
     * Cached volume growth
     * <p>
     * Amount of space around the controller we cache to improve performance. This is a scale factor
     * that should be higher than 1.0f but not too big, ideally lower than 2.0f.
     * <p>
     * <b>Default:</b> 1.5
     */
    public void setVolumeGrowth(float value) {
        checkNotNull();
        _setVolumeGrowth(address, value);
    }
    private static native void _setVolumeGrowth(long address, float value);

    /**
     * Specifies a user report callback.
     * <p>
     * This report callback is called when the character collides with shapes and other characters.
     * <p>
     * Setting this to NULL disables the callback.
     * <p>
     * <b>Default:</b> NULL
     * @see PxUserControllerHitReport
     */
    public PxUserControllerHitReport getReportCallback() {
        checkNotNull();
        return PxUserControllerHitReport.wrapPointer(_getReportCallback(address));
    }
    private static native long _getReportCallback(long address);

    /**
     * Specifies a user report callback.
     * <p>
     * This report callback is called when the character collides with shapes and other characters.
     * <p>
     * Setting this to NULL disables the callback.
     * <p>
     * <b>Default:</b> NULL
     * @see PxUserControllerHitReport
     */
    public void setReportCallback(PxUserControllerHitReport value) {
        checkNotNull();
        _setReportCallback(address, value.getAddress());
    }
    private static native void _setReportCallback(long address, long value);

    /**
     * Specifies a user behavior callback.
     * <p>
     * This behavior callback is called to customize the controller's behavior w.r.t. touched shapes.
     * <p>
     * Setting this to NULL disables the callback.
     * <p>
     * <b>Default:</b> NULL
     * @see PxControllerBehaviorCallback
     */
    public PxControllerBehaviorCallback getBehaviorCallback() {
        checkNotNull();
        return PxControllerBehaviorCallback.wrapPointer(_getBehaviorCallback(address));
    }
    private static native long _getBehaviorCallback(long address);

    /**
     * Specifies a user behavior callback.
     * <p>
     * This behavior callback is called to customize the controller's behavior w.r.t. touched shapes.
     * <p>
     * Setting this to NULL disables the callback.
     * <p>
     * <b>Default:</b> NULL
     * @see PxControllerBehaviorCallback
     */
    public void setBehaviorCallback(PxControllerBehaviorCallback value) {
        checkNotNull();
        _setBehaviorCallback(address, value.getAddress());
    }
    private static native void _setBehaviorCallback(long address, long value);

    /**
     * The non-walkable mode controls if a character controller slides or not on a non-walkable part.
     * <p>
     * This is only used when slopeLimit is non zero.
     * <p>
     * <b>Default:</b> PxControllerNonWalkableMode::ePREVENT_CLIMBING
     */
    public PxControllerNonWalkableModeEnum getNonWalkableMode() {
        checkNotNull();
        return PxControllerNonWalkableModeEnum.forValue(_getNonWalkableMode(address));
    }
    private static native int _getNonWalkableMode(long address);

    /**
     * The non-walkable mode controls if a character controller slides or not on a non-walkable part.
     * <p>
     * This is only used when slopeLimit is non zero.
     * <p>
     * <b>Default:</b> PxControllerNonWalkableMode::ePREVENT_CLIMBING
     */
    public void setNonWalkableMode(PxControllerNonWalkableModeEnum value) {
        checkNotNull();
        _setNonWalkableMode(address, value.value);
    }
    private static native void _setNonWalkableMode(long address, int value);

    /**
     * The material for the actor associated with the controller.
     * <p>
     * The controller internally creates a rigid body actor. This parameter specifies the material of the actor.
     * <p>
     * <b>Default:</b> NULL
     * @see physx.physics.PxMaterial
     */
    public PxMaterial getMaterial() {
        checkNotNull();
        return PxMaterial.wrapPointer(_getMaterial(address));
    }
    private static native long _getMaterial(long address);

    /**
     * The material for the actor associated with the controller.
     * <p>
     * The controller internally creates a rigid body actor. This parameter specifies the material of the actor.
     * <p>
     * <b>Default:</b> NULL
     * @see physx.physics.PxMaterial
     */
    public void setMaterial(PxMaterial value) {
        checkNotNull();
        _setMaterial(address, value.getAddress());
    }
    private static native void _setMaterial(long address, long value);

    /**
     * Use a deletion listener to get informed about released objects and clear internal caches if needed.
     * <p>
     * If a character controller registers a deletion listener, it will get informed about released objects. That allows the
     * controller to invalidate cached data that connects to a released object. If a deletion listener is not
     * registered, PxController::invalidateCache has to be called manually after objects have been released.
     * <p>
     * <b>Default:</b> true
     */
    public boolean getRegisterDeletionListener() {
        checkNotNull();
        return _getRegisterDeletionListener(address);
    }
    private static native boolean _getRegisterDeletionListener(long address);

    /**
     * Use a deletion listener to get informed about released objects and clear internal caches if needed.
     * <p>
     * If a character controller registers a deletion listener, it will get informed about released objects. That allows the
     * controller to invalidate cached data that connects to a released object. If a deletion listener is not
     * registered, PxController::invalidateCache has to be called manually after objects have been released.
     * <p>
     * <b>Default:</b> true
     */
    public void setRegisterDeletionListener(boolean value) {
        checkNotNull();
        _setRegisterDeletionListener(address, value);
    }
    private static native void _setRegisterDeletionListener(long address, boolean value);

    /**
     * User specified data associated with the controller.
     * <p>
     * <b>Default:</b> NULL
     */
    public NativeObject getUserData() {
        checkNotNull();
        return NativeObject.wrapPointer(_getUserData(address));
    }
    private static native long _getUserData(long address);

    /**
     * User specified data associated with the controller.
     * <p>
     * <b>Default:</b> NULL
     */
    public void setUserData(NativeObject value) {
        checkNotNull();
        _setUserData(address, value.getAddress());
    }
    private static native void _setUserData(long address, long value);

    // Functions

    /**
     * returns true if the current settings are valid
     * @return True if the descriptor is valid.
     */
    public boolean isValid() {
        checkNotNull();
        return _isValid(address);
    }
    private static native boolean _isValid(long address);

    /**
     * Returns the character controller type
     * @return The controllers type.
     * @see PxCapsuleControllerDesc
     * @see PxBoxControllerDesc
     */
    public PxControllerShapeTypeEnum getType() {
        checkNotNull();
        return PxControllerShapeTypeEnum.forValue(_getType(address));
    }
    private static native int _getType(long address);

}
