package physx.common;

import physx.NativeObject;

/**
 * Base class for objects that can be members of a PxCollection.
 * <p>
 * All PxBase sub-classes can be serialized.
 * @see PxCollection
 */
public class PxBase extends NativeObject {

    protected PxBase() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxBase wrapPointer(long address) {
        return address != 0L ? new PxBase(address) : null;
    }
    
    public static PxBase arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxBase(long address) {
        super(address);
    }

    // Functions

    /**
     * Releases the PxBase instance, please check documentation of release in derived class.
     */
    public void release() {
        checkNotNull();
        _release(address);
    }
    private static native void _release(long address);

    /**
     * Returns string name of dynamic type.
     * @return Class name of most derived type of this object.
     */
    public String getConcreteTypeName() {
        checkNotNull();
        return _getConcreteTypeName(address);
    }
    private static native String _getConcreteTypeName(long address);

    /**
     * Returns concrete type of object.
     * @return PxConcreteType::Enum of serialized object
     */
    public int getConcreteType() {
        checkNotNull();
        return _getConcreteType(address);
    }
    private static native int _getConcreteType(long address);

    /**
     * Set PxBaseFlag 
     * @param flag The flag to be set
     * @param value The flags new value
     */
    public void setBaseFlag(PxBaseFlagEnum flag, boolean value) {
        checkNotNull();
        _setBaseFlag(address, flag.value, value);
    }
    private static native void _setBaseFlag(long address, int flag, boolean value);

    /**
     * Set PxBaseFlags 
     * @param inFlags The flags to be set
     * @see PxBaseFlags
     */
    public void setBaseFlags(PxBaseFlags inFlags) {
        checkNotNull();
        _setBaseFlags(address, inFlags.getAddress());
    }
    private static native void _setBaseFlags(long address, long inFlags);

    /**
     * Returns PxBaseFlags 
     * @return PxBaseFlags
     * @see PxBaseFlags
     */
    public PxBaseFlags getBaseFlags() {
        checkNotNull();
        return PxBaseFlags.wrapPointer(_getBaseFlags(address));
    }
    private static native long _getBaseFlags(long address);

    /**
     * Whether the object is subordinate.
     * <p>
     * A class is subordinate, if it can only be instantiated in the context of another class.
     * @return Whether the class is subordinate
     */
    public boolean isReleasable() {
        checkNotNull();
        return _isReleasable(address);
    }
    private static native boolean _isReleasable(long address);

}
