package physx.extensions;

import physx.NativeObject;
import physx.geometry.PxGeometry;
import physx.physics.PxMaterial;
import physx.physics.PxRigidActor;
import physx.physics.PxShape;
import physx.physics.PxShapeFlags;

/**
 * utility functions for use with PxRigidActor and subclasses
 * @see physx.physics.PxRigidActor
 * @see physx.physics.PxRigidStatic
 * @see physx.physics.PxRigidBody
 * @see physx.physics.PxRigidDynamic
 * @see physx.physics.PxArticulationLink
 */
public class PxRigidActorExt extends NativeObject {

    protected PxRigidActorExt() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxRigidActorExt wrapPointer(long address) {
        return address != 0L ? new PxRigidActorExt(address) : null;
    }
    
    public static PxRigidActorExt arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxRigidActorExt(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Creates a new shape with default properties and a single material adds it to the list of shapes of this actor.
     * <p>
     * This is equivalent to the following
     * <p>
     * PxShape* shape(...) = PxGetPhysics().createShape(...); // reference count is 1
     * actor-&gt;attachShape(shape);        // increments reference count
     * shape-&gt;release();          // releases user reference, leaving reference count at 1
     * <p>
     * As a consequence, detachShape() will result in the release of the last reference, and the shape will be deleted.
     * <p>
     * <b>Note:</b> The default shape flags to be set are: eVISUALIZATION, eSIMULATION_SHAPE, eSCENE_QUERY_SHAPE (see #PxShapeFlag).
     * Triangle mesh, heightfield or plane geometry shapes configured as eSIMULATION_SHAPE are not supported for 
     * non-kinematic PxRigidDynamic instances.
     * <p>
     * <b>Note:</b> Creating compounds with a very large number of shapes may adversely affect performance and stability.
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the actor up automatically.
     * @param actor the actor to which to attach the shape
     * @param geometry the geometry of the shape
     * @param material the material for the shape
     * @return The newly created shape.
     * @see physx.physics.PxShape
     */
    public static PxShape createExclusiveShape(PxRigidActor actor, PxGeometry geometry, PxMaterial material) {
        return PxShape.wrapPointer(_createExclusiveShape(actor.getAddress(), geometry.getAddress(), material.getAddress()));
    }
    private static native long _createExclusiveShape(long actor, long geometry, long material);

    /**
     * Creates a new shape with default properties and a single material adds it to the list of shapes of this actor.
     * <p>
     * This is equivalent to the following
     * <p>
     * PxShape* shape(...) = PxGetPhysics().createShape(...); // reference count is 1
     * actor-&gt;attachShape(shape);        // increments reference count
     * shape-&gt;release();          // releases user reference, leaving reference count at 1
     * <p>
     * As a consequence, detachShape() will result in the release of the last reference, and the shape will be deleted.
     * <p>
     * <b>Note:</b> The default shape flags to be set are: eVISUALIZATION, eSIMULATION_SHAPE, eSCENE_QUERY_SHAPE (see #PxShapeFlag).
     * Triangle mesh, heightfield or plane geometry shapes configured as eSIMULATION_SHAPE are not supported for 
     * non-kinematic PxRigidDynamic instances.
     * <p>
     * <b>Note:</b> Creating compounds with a very large number of shapes may adversely affect performance and stability.
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the actor up automatically.
     * @param actor the actor to which to attach the shape
     * @param geometry the geometry of the shape
     * @param material the material for the shape
     * @return The newly created shape.
     * @see physx.physics.PxShape
     */
    public static PxShape createExclusiveShape(PxRigidActor actor, PxGeometry geometry, PxMaterial material, PxShapeFlags flags) {
        return PxShape.wrapPointer(_createExclusiveShape(actor.getAddress(), geometry.getAddress(), material.getAddress(), flags.getAddress()));
    }
    private static native long _createExclusiveShape(long actor, long geometry, long material, long flags);

}
