package physx.geometry;

import physx.NativeObject;
import physx.common.PxVec3;
import physx.common.PxVec4;
import physx.support.PxArray_PxU32;
import physx.support.PxArray_PxVec3;
import physx.support.PxArray_PxVec4;

/**
 * utility functions for use with PxTetrahedronMesh and subclasses
 */
public class PxTetrahedronMeshExt extends NativeObject {

    protected PxTetrahedronMeshExt() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxTetrahedronMeshExt wrapPointer(long address) {
        return address != 0L ? new PxTetrahedronMeshExt(address) : null;
    }
    
    public static PxTetrahedronMeshExt arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxTetrahedronMeshExt(long address) {
        super(address);
    }

    // Functions

    /**
     * Returns the index of the tetrahedron that contains a point
     * @param mesh The tetmesh
     * @param point The point to find the enclosing tetrahedron for
     * @param bary The barycentric coordinates of the point inside the enclosing tetrahedron
     * @param tolerance Tolerance value used classify points as inside if they lie exactly a tetrahedron's surface
     * @return The index of the tetrahedon containing the point, -1 if not tetrahedron contains the opoint
     */
    public static int findTetrahedronContainingPoint(PxTetrahedronMesh mesh, PxVec3 point, PxVec4 bary, float tolerance) {
        return _findTetrahedronContainingPoint(mesh.getAddress(), point.getAddress(), bary.getAddress(), tolerance);
    }
    private static native int _findTetrahedronContainingPoint(long mesh, long point, long bary, float tolerance);

    /**
     * Returns the index of the tetrahedron closest to a point
     * @param mesh The tetmesh
     * @param point The point to find the closest tetrahedron for
     * @param bary The barycentric coordinates of the point in the tetrahedron
     * @return The index of the tetrahedon closest to the point
     */
    public static int findTetrahedronClosestToPoint(PxTetrahedronMesh mesh, PxVec3 point, PxVec4 bary) {
        return _findTetrahedronClosestToPoint(mesh.getAddress(), point.getAddress(), bary.getAddress());
    }
    private static native int _findTetrahedronClosestToPoint(long mesh, long point, long bary);

    /**
     * Associates points with closest tetrahedra from input tetrahedral mesh. If the tetmesh does not have any tetrahedra
     * or points, a warning will be generated and the result arrays will be empty, even if there are query points passed into the method.
     * @param tetMeshVertices The tetrahedral mesh vertices
     * @param tetMeshIndices The tetraheral mesh indices
     * @param pointsToEmbed  The points for which the embedding should be created
     *       @param barycentricCoordinates  The output barycentric coordinates for each input point relative to its closest tetrahedron
     *       @param tetLinks The output indices of the closest tetrahedron for each input point
     */
    public static void createPointsToTetrahedronMap(PxArray_PxVec3 tetMeshVertices, PxArray_PxU32 tetMeshIndices, PxArray_PxVec3 pointsToEmbed, PxArray_PxVec4 barycentricCoordinates, PxArray_PxU32 tetLinks) {
        _createPointsToTetrahedronMap(tetMeshVertices.getAddress(), tetMeshIndices.getAddress(), pointsToEmbed.getAddress(), barycentricCoordinates.getAddress(), tetLinks.getAddress());
    }
    private static native void _createPointsToTetrahedronMap(long tetMeshVertices, long tetMeshIndices, long pointsToEmbed, long barycentricCoordinates, long tetLinks);

    /**
     * Extracts the surface triangles of a tetmesh
     * <p>
     * The extracted triangle's vertex indices point to the vertex buffer of the tetmesh.
     * @param mesh The mesh from which the surface shall be computed
     * @param surfaceTriangles The resulting surface triangles
     */
    public static void extractTetMeshSurface(PxTetrahedronMesh mesh, PxArray_PxU32 surfaceTriangles) {
        _extractTetMeshSurface(mesh.getAddress(), surfaceTriangles.getAddress());
    }
    private static native void _extractTetMeshSurface(long mesh, long surfaceTriangles);

    /**
     * Extracts the surface triangles of a tetmesh
     * <p>
     * The extracted triangle's vertex indices point to the vertex buffer of the tetmesh.
     * @param mesh The mesh from which the surface shall be computed
     * @param surfaceTriangles The resulting surface triangles
     * @param surfaceTriangleToTet Optional array to get the index of a tetrahedron that is adjacent to the surface triangle with the corresponding index
     */
    public static void extractTetMeshSurface(PxTetrahedronMesh mesh, PxArray_PxU32 surfaceTriangles, PxArray_PxU32 surfaceTriangleToTet) {
        _extractTetMeshSurface(mesh.getAddress(), surfaceTriangles.getAddress(), surfaceTriangleToTet.getAddress());
    }
    private static native void _extractTetMeshSurface(long mesh, long surfaceTriangles, long surfaceTriangleToTet);

    /**
     * Extracts the surface triangles of a tetmesh
     * <p>
     * The extracted triangle's vertex indices point to the vertex buffer of the tetmesh.
     * @param mesh The mesh from which the surface shall be computed
     * @param surfaceTriangles The resulting surface triangles
     * @param surfaceTriangleToTet Optional array to get the index of a tetrahedron that is adjacent to the surface triangle with the corresponding index
     * @param flipTriangleOrientation Reverses the orientation of the ouput triangles
     */
    public static void extractTetMeshSurface(PxTetrahedronMesh mesh, PxArray_PxU32 surfaceTriangles, PxArray_PxU32 surfaceTriangleToTet, boolean flipTriangleOrientation) {
        _extractTetMeshSurface(mesh.getAddress(), surfaceTriangles.getAddress(), surfaceTriangleToTet.getAddress(), flipTriangleOrientation);
    }
    private static native void _extractTetMeshSurface(long mesh, long surfaceTriangles, long surfaceTriangleToTet, boolean flipTriangleOrientation);

}
