package physx.geometry;

import physx.NativeObject;
import physx.common.PxVec3;

/**
 * Triangle class.
 */
public class PxTriangle extends NativeObject {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxTriangle wrapPointer(long address) {
        return address != 0L ? new PxTriangle(address) : null;
    }
    
    public static PxTriangle arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxTriangle(long address) {
        super(address);
    }

    // Constructors

    /**
     * Constructor
     */
    public PxTriangle() {
        address = _PxTriangle();
    }
    private static native long _PxTriangle();

    /**
     * Constructor
     * @param p0 Point 0
     * @param p1 Point 1
     * @param p2 Point 2
     */
    public PxTriangle(PxVec3 p0, PxVec3 p1, PxVec3 p2) {
        address = _PxTriangle(p0.getAddress(), p1.getAddress(), p2.getAddress());
    }
    private static native long _PxTriangle(long p0, long p1, long p2);

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Compute the normal of the Triangle.
     */
    public void normal(PxVec3 normal) {
        checkNotNull();
        _normal(address, normal.getAddress());
    }
    private static native void _normal(long address, long normal);

    /**
     * Compute the unnormalized normal of the triangle.
     */
    public void denormalizedNormal(PxVec3 normal) {
        checkNotNull();
        _denormalizedNormal(address, normal.getAddress());
    }
    private static native void _denormalizedNormal(long address, long normal);

    /**
     * Compute the area of the triangle.
     * @return Area of the triangle.
     */
    public float area() {
        checkNotNull();
        return _area(address);
    }
    private static native float _area(long address);

    /**
     * @return Computes a point on the triangle from u and v barycentric coordinates.
     */
    public PxVec3 pointFromUV(float u, float v) {
        checkNotNull();
        return PxVec3.wrapPointer(_pointFromUV(address, u, v));
    }
    private static native long _pointFromUV(long address, float u, float v);

}
