package physx.physics;

import physx.NativeObject;
import physx.support.PxRealPtr;

/**
 * Data structure used to read and write internal articulation data.
 * <p>
 * PxArticulationReducedCoordinate::copyInternalStateToCache
 */
public class PxArticulationCache extends NativeObject {

    protected PxArticulationCache() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxArticulationCache wrapPointer(long address) {
        return address != 0L ? new PxArticulationCache(address) : null;
    }
    
    public static PxArticulationCache arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxArticulationCache(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * External forces acting on the articulation links for inverse dynamics computation.
     * <p>
     * - N = getNbLinks().
     * - Indexing follows the low-level link indices, see PxArticulationLink::getLinkIndex.
     * - The forces are with respect to the center of mass of the link.
     * - This field cannot be used to apply forces to links during the next PxScene::simulate() call. Use PxRigidBody::addForce and related functions instead.
     */
    public PxSpatialForce getExternalForces() {
        checkNotNull();
        return PxSpatialForce.wrapPointer(_getExternalForces(address));
    }
    private static native long _getExternalForces(long address);

    /**
     * External forces acting on the articulation links for inverse dynamics computation.
     * <p>
     * - N = getNbLinks().
     * - Indexing follows the low-level link indices, see PxArticulationLink::getLinkIndex.
     * - The forces are with respect to the center of mass of the link.
     * - This field cannot be used to apply forces to links during the next PxScene::simulate() call. Use PxRigidBody::addForce and related functions instead.
     */
    public void setExternalForces(PxSpatialForce value) {
        checkNotNull();
        _setExternalForces(address, value.getAddress());
    }
    private static native void _setExternalForces(long address, long value);

    /**
     * Dense Jacobian data.
     * <p>
     * - N = nbRows * nbCols = (6 * getNbLinks()) * (6 + getDofs()) -&gt; size includes possible floating-base DOFs regardless of PxArticulationFlag::eFIX_BASE flag.
     * - The links, i.e. rows are in order of the low-level link indices (minus one if PxArticulationFlag::eFIX_BASE is true), see PxArticulationLink::getLinkIndex.
     * The corresponding spatial velocities are stacked [vx; vy; vz; wx; wy; wz], where vx and wx refer to the linear and rotational velocity in world X.
     * - The DOFs, i.e. column indices correspond to the low-level DOF indices, see PxArticulationCache::jointVelocity.
     */
    public PxRealPtr getDenseJacobian() {
        checkNotNull();
        return PxRealPtr.wrapPointer(_getDenseJacobian(address));
    }
    private static native long _getDenseJacobian(long address);

    /**
     * Dense Jacobian data.
     * <p>
     * - N = nbRows * nbCols = (6 * getNbLinks()) * (6 + getDofs()) -&gt; size includes possible floating-base DOFs regardless of PxArticulationFlag::eFIX_BASE flag.
     * - The links, i.e. rows are in order of the low-level link indices (minus one if PxArticulationFlag::eFIX_BASE is true), see PxArticulationLink::getLinkIndex.
     * The corresponding spatial velocities are stacked [vx; vy; vz; wx; wy; wz], where vx and wx refer to the linear and rotational velocity in world X.
     * - The DOFs, i.e. column indices correspond to the low-level DOF indices, see PxArticulationCache::jointVelocity.
     */
    public void setDenseJacobian(PxRealPtr value) {
        checkNotNull();
        _setDenseJacobian(address, value.getAddress());
    }
    private static native void _setDenseJacobian(long address, long value);

    /**
     * The generalized mass matrix that maps joint accelerations to joint forces.
     * <p>
     * - N = getDofs() * getDofs().
     * - The indexing follows the internal DOF index order, see PxArticulationCache::jointVelocity.
     */
    public PxRealPtr getMassMatrix() {
        checkNotNull();
        return PxRealPtr.wrapPointer(_getMassMatrix(address));
    }
    private static native long _getMassMatrix(long address);

    /**
     * The generalized mass matrix that maps joint accelerations to joint forces.
     * <p>
     * - N = getDofs() * getDofs().
     * - The indexing follows the internal DOF index order, see PxArticulationCache::jointVelocity.
     */
    public void setMassMatrix(PxRealPtr value) {
        checkNotNull();
        _setMassMatrix(address, value.getAddress());
    }
    private static native void _setMassMatrix(long address, long value);

    /**
     * The articulation joint DOF velocities.
     * <p>
     * - N = getDofs().
     * - Read/write using PxArticulationCacheFlag::eVELOCITY.
     * - The indexing follows the internal DOF index order. Therefore, the application should calculate the DOF data indices by summing the joint DOFs in the order of
     * the links' low-level indices (see the manual Section "Cache Indexing" for a snippet for this calculation):
     * \verbatim Low-level link index:   | link 0 | link 1 | link 2 | link 3 | ... | &lt;- PxArticulationLink::getLinkIndex()       \endverbatim
     * \verbatim Link inbound joint DOF: | 0      | 1      | 2      | 1      | ... | &lt;- PxArticulationLink::getInboundJointDof() \endverbatim
     * \verbatim Low-level DOF index:    | -      | 0      | 1, 2   | 3      | ... |                                             \endverbatim
     * The root link always has low-level index 0 and always has zero inbound joint DOFs. The link DOF indexing follows the order in PxArticulationAxis::Enum.
     * For example, assume that low-level link 2 has an inbound spherical joint with two DOFs: eSWING1 and eSWING2. The corresponding low-level joint DOF indices
     * are therefore 1 for eSWING1 and 2 for eSWING2.
     */
    public PxRealPtr getJointVelocity() {
        checkNotNull();
        return PxRealPtr.wrapPointer(_getJointVelocity(address));
    }
    private static native long _getJointVelocity(long address);

    /**
     * The articulation joint DOF velocities.
     * <p>
     * - N = getDofs().
     * - Read/write using PxArticulationCacheFlag::eVELOCITY.
     * - The indexing follows the internal DOF index order. Therefore, the application should calculate the DOF data indices by summing the joint DOFs in the order of
     * the links' low-level indices (see the manual Section "Cache Indexing" for a snippet for this calculation):
     * \verbatim Low-level link index:   | link 0 | link 1 | link 2 | link 3 | ... | &lt;- PxArticulationLink::getLinkIndex()       \endverbatim
     * \verbatim Link inbound joint DOF: | 0      | 1      | 2      | 1      | ... | &lt;- PxArticulationLink::getInboundJointDof() \endverbatim
     * \verbatim Low-level DOF index:    | -      | 0      | 1, 2   | 3      | ... |                                             \endverbatim
     * The root link always has low-level index 0 and always has zero inbound joint DOFs. The link DOF indexing follows the order in PxArticulationAxis::Enum.
     * For example, assume that low-level link 2 has an inbound spherical joint with two DOFs: eSWING1 and eSWING2. The corresponding low-level joint DOF indices
     * are therefore 1 for eSWING1 and 2 for eSWING2.
     */
    public void setJointVelocity(PxRealPtr value) {
        checkNotNull();
        _setJointVelocity(address, value.getAddress());
    }
    private static native void _setJointVelocity(long address, long value);

    /**
     * The articulation joint DOF accelerations.
     * <p>
     * - N = getDofs().
     * - Read using PxArticulationCacheFlag::eACCELERATION.
     * - The indexing follows the internal DOF index order, see PxArticulationCache::jointVelocity.
     * - Delta joint DOF velocities can be computed from acceleration * dt.
     */
    public PxRealPtr getJointAcceleration() {
        checkNotNull();
        return PxRealPtr.wrapPointer(_getJointAcceleration(address));
    }
    private static native long _getJointAcceleration(long address);

    /**
     * The articulation joint DOF accelerations.
     * <p>
     * - N = getDofs().
     * - Read using PxArticulationCacheFlag::eACCELERATION.
     * - The indexing follows the internal DOF index order, see PxArticulationCache::jointVelocity.
     * - Delta joint DOF velocities can be computed from acceleration * dt.
     */
    public void setJointAcceleration(PxRealPtr value) {
        checkNotNull();
        _setJointAcceleration(address, value.getAddress());
    }
    private static native void _setJointAcceleration(long address, long value);

    /**
     * The articulation joint DOF positions.
     * <p>
     * - N = getDofs().
     * - Read/write using PxArticulationCacheFlag::ePOSITION.
     * - The indexing follows the internal DOF index order, see PxArticulationCache::jointVelocity.
     *       - For spherical joints, the joint position for each axis on the joint must be in range [-Pi, Pi].
     */
    public PxRealPtr getJointPosition() {
        checkNotNull();
        return PxRealPtr.wrapPointer(_getJointPosition(address));
    }
    private static native long _getJointPosition(long address);

    /**
     * The articulation joint DOF positions.
     * <p>
     * - N = getDofs().
     * - Read/write using PxArticulationCacheFlag::ePOSITION.
     * - The indexing follows the internal DOF index order, see PxArticulationCache::jointVelocity.
     *       - For spherical joints, the joint position for each axis on the joint must be in range [-Pi, Pi].
     */
    public void setJointPosition(PxRealPtr value) {
        checkNotNull();
        _setJointPosition(address, value.getAddress());
    }
    private static native void _setJointPosition(long address, long value);

    /**
     * The articulation joint DOF forces.
     * <p>
     * - N = getDofs().
     * - Read/Write using PxArticulationCacheFlag::eFORCE.
     * - The indexing follows the internal DOF index order, see PxArticulationCache::jointVelocity.
     * - Applied joint forces persist and are applied each frame until changed.
     */
    public PxRealPtr getJointForce() {
        checkNotNull();
        return PxRealPtr.wrapPointer(_getJointForce(address));
    }
    private static native long _getJointForce(long address);

    /**
     * The articulation joint DOF forces.
     * <p>
     * - N = getDofs().
     * - Read/Write using PxArticulationCacheFlag::eFORCE.
     * - The indexing follows the internal DOF index order, see PxArticulationCache::jointVelocity.
     * - Applied joint forces persist and are applied each frame until changed.
     */
    public void setJointForce(PxRealPtr value) {
        checkNotNull();
        _setJointForce(address, value.getAddress());
    }
    private static native void _setJointForce(long address, long value);

    /**
     * Link spatial velocity.
     * <p>
     * - N = getNbLinks().
     * - Read using PxArticulationCacheFlag::eLINK_VELOCITY.
     * - The indexing follows the internal link indexing, see PxArticulationLink::getLinkIndex.
     * - The velocity is with respect to the link's center of mass but represented in world space.
     */
    public PxSpatialVelocity getLinkVelocity() {
        checkNotNull();
        return PxSpatialVelocity.wrapPointer(_getLinkVelocity(address));
    }
    private static native long _getLinkVelocity(long address);

    /**
     * Link spatial velocity.
     * <p>
     * - N = getNbLinks().
     * - Read using PxArticulationCacheFlag::eLINK_VELOCITY.
     * - The indexing follows the internal link indexing, see PxArticulationLink::getLinkIndex.
     * - The velocity is with respect to the link's center of mass but represented in world space.
     */
    public void setLinkVelocity(PxSpatialVelocity value) {
        checkNotNull();
        _setLinkVelocity(address, value.getAddress());
    }
    private static native void _setLinkVelocity(long address, long value);

    /**
     * Link classical acceleration.
     * <p>
     * - N = getNbLinks().
     * - Read using PxArticulationCacheFlag::eLINK_ACCELERATION.
     * - The indexing follows the internal link indexing, see PxArticulationLink::getLinkIndex.
     * - The acceleration is with respect to the link's center of mass.
     */
    public PxSpatialVelocity getLinkAcceleration() {
        checkNotNull();
        return PxSpatialVelocity.wrapPointer(_getLinkAcceleration(address));
    }
    private static native long _getLinkAcceleration(long address);

    /**
     * Link classical acceleration.
     * <p>
     * - N = getNbLinks().
     * - Read using PxArticulationCacheFlag::eLINK_ACCELERATION.
     * - The indexing follows the internal link indexing, see PxArticulationLink::getLinkIndex.
     * - The acceleration is with respect to the link's center of mass.
     */
    public void setLinkAcceleration(PxSpatialVelocity value) {
        checkNotNull();
        _setLinkAcceleration(address, value.getAddress());
    }
    private static native void _setLinkAcceleration(long address, long value);

    /**
     * Link incoming joint force, i.e. the total force transmitted from the parent link to this link.
     * <p>
     * - N = getNbLinks().
     * - Read using PxArticulationCacheFlag::eLINK_INCOMING_JOINT_FORCE.
     * - The indexing follows the internal link indexing, see PxArticulationLink::getLinkIndex.
     * - The force is reported in the child joint frame of the link's incoming joint.
     * <p>
     * <b>Note:</b> The root link reports a zero spatial force.
     */
    public PxSpatialForce getLinkIncomingJointForce() {
        checkNotNull();
        return PxSpatialForce.wrapPointer(_getLinkIncomingJointForce(address));
    }
    private static native long _getLinkIncomingJointForce(long address);

    /**
     * Link incoming joint force, i.e. the total force transmitted from the parent link to this link.
     * <p>
     * - N = getNbLinks().
     * - Read using PxArticulationCacheFlag::eLINK_INCOMING_JOINT_FORCE.
     * - The indexing follows the internal link indexing, see PxArticulationLink::getLinkIndex.
     * - The force is reported in the child joint frame of the link's incoming joint.
     * <p>
     * <b>Note:</b> The root link reports a zero spatial force.
     */
    public void setLinkIncomingJointForce(PxSpatialForce value) {
        checkNotNull();
        _setLinkIncomingJointForce(address, value.getAddress());
    }
    private static native void _setLinkIncomingJointForce(long address, long value);

    /**
     * Root link transform, velocities, and accelerations.
     * <p>
     * - N = 1.
     * - Read/write using PxArticulationCacheFlag::eROOT_TRANSFORM and PxArticulationCacheFlag::eROOT_VELOCITIES (accelerations are read-only).
     * @see PxArticulationRootLinkData
     */
    public PxArticulationRootLinkData getRootLinkData() {
        checkNotNull();
        return PxArticulationRootLinkData.wrapPointer(_getRootLinkData(address));
    }
    private static native long _getRootLinkData(long address);

    /**
     * Root link transform, velocities, and accelerations.
     * <p>
     * - N = 1.
     * - Read/write using PxArticulationCacheFlag::eROOT_TRANSFORM and PxArticulationCacheFlag::eROOT_VELOCITIES (accelerations are read-only).
     * @see PxArticulationRootLinkData
     */
    public void setRootLinkData(PxArticulationRootLinkData value) {
        checkNotNull();
        _setRootLinkData(address, value.getAddress());
    }
    private static native void _setRootLinkData(long address, long value);

    /**
     * @deprecated  The API related to loop joints will be removed in a future version once a replacement is made available.
     * <p>
     * Constraint coefficient matrix.
     * <p>
     * - N = getCoefficentMatrixSize().
     * - The user needs to allocate memory and set this member to the allocated memory.
     */
    @Deprecated
    public PxRealPtr getCoefficientMatrix() {
        checkNotNull();
        return PxRealPtr.wrapPointer(_getCoefficientMatrix(address));
    }
    private static native long _getCoefficientMatrix(long address);

    /**
     * @deprecated  The API related to loop joints will be removed in a future version once a replacement is made available.
     * <p>
     * Constraint coefficient matrix.
     * <p>
     * - N = getCoefficentMatrixSize().
     * - The user needs to allocate memory and set this member to the allocated memory.
     */
    @Deprecated
    public void setCoefficientMatrix(PxRealPtr value) {
        checkNotNull();
        _setCoefficientMatrix(address, value.getAddress());
    }
    private static native void _setCoefficientMatrix(long address, long value);

    /**
     * @deprecated  The API related to loop joints will be removed in a future version once a replacement is made available.
     * <p>
     * Constraint lambda values (impulses applied by the respective constraints).
     * <p>
     * - N = getNbLoopJoints().
     * - The user needs to allocate memory and set this member to the allocated memory.
     */
    @Deprecated
    public PxRealPtr getLambda() {
        checkNotNull();
        return PxRealPtr.wrapPointer(_getLambda(address));
    }
    private static native long _getLambda(long address);

    /**
     * @deprecated  The API related to loop joints will be removed in a future version once a replacement is made available.
     * <p>
     * Constraint lambda values (impulses applied by the respective constraints).
     * <p>
     * - N = getNbLoopJoints().
     * - The user needs to allocate memory and set this member to the allocated memory.
     */
    @Deprecated
    public void setLambda(PxRealPtr value) {
        checkNotNull();
        _setLambda(address, value.getAddress());
    }
    private static native void _setLambda(long address, long value);

    /**
     * The scratch memory is used for internal calculations.
     */
    public NativeObject getScratchMemory() {
        checkNotNull();
        return NativeObject.wrapPointer(_getScratchMemory(address));
    }
    private static native long _getScratchMemory(long address);

    /**
     * The scratch memory is used for internal calculations.
     */
    public void setScratchMemory(NativeObject value) {
        checkNotNull();
        _setScratchMemory(address, value.getAddress());
    }
    private static native void _setScratchMemory(long address, long value);

    /**
     * The scratch allocator is used for internal calculations.
     */
    public NativeObject getScratchAllocator() {
        checkNotNull();
        return NativeObject.wrapPointer(_getScratchAllocator(address));
    }
    private static native long _getScratchAllocator(long address);

    /**
     * The scratch allocator is used for internal calculations.
     */
    public void setScratchAllocator(NativeObject value) {
        checkNotNull();
        _setScratchAllocator(address, value.getAddress());
    }
    private static native void _setScratchAllocator(long address, long value);

    /**
     * The cache version used internally to check compatibility with the articulation, i.e. detect if the articulation configuration changed after the cache was created.
     */
    public int getVersion() {
        checkNotNull();
        return _getVersion(address);
    }
    private static native int _getVersion(long address);

    /**
     * The cache version used internally to check compatibility with the articulation, i.e. detect if the articulation configuration changed after the cache was created.
     */
    public void setVersion(int value) {
        checkNotNull();
        _setVersion(address, value);
    }
    private static native void _setVersion(long address, int value);

    // Functions

    /**
     * Releases an articulation cache.
     * <p>
     *   PxArticulationReducedCoordinate::copyInternalStateToCache
     */
    public void release() {
        checkNotNull();
        _release(address);
    }
    private static native void _release(long address);

}
