package physx.physics;

import physx.NativeObject;
import physx.common.PxBounds3;

/**
 * "Region of interest" for the broad-phase.
 * <p>
 * This is currently only used for the PxBroadPhaseType::eMBP broad-phase, which requires zones or regions to be defined
 * when the simulation starts in order to work. Regions can overlap and be added or removed at runtime, but at least one
 * region needs to be defined when the scene is created.
 * <p>
 * If objects that do no overlap any region are inserted into the scene, they will not be added to the broad-phase and
 * thus collisions will be disabled for them. A PxBroadPhaseCallback out-of-bounds notification will be sent for each one
 * of those objects.
 * <p>
 * The total number of regions is limited by PxBroadPhaseCaps::mMaxNbRegions.
 * <p>
 * The number of regions has a direct impact on performance and memory usage, so it is recommended to experiment with
 * various settings to find the best combination for your game. A good default setup is to start with global bounds
 * around the whole world, and subdivide these bounds into 4*4 regions. The PxBroadPhaseExt::createRegionsFromWorldBounds
 * function can do that for you.
 */
public class PxBroadPhaseRegion extends NativeObject {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxBroadPhaseRegion wrapPointer(long address) {
        return address != 0L ? new PxBroadPhaseRegion(address) : null;
    }
    
    public static PxBroadPhaseRegion arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxBroadPhaseRegion(long address) {
        super(address);
    }

    // Constructors

    public PxBroadPhaseRegion() {
        address = _PxBroadPhaseRegion();
    }
    private static native long _PxBroadPhaseRegion();

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * Region's bounds
     */
    public PxBounds3 getMBounds() {
        checkNotNull();
        return PxBounds3.wrapPointer(_getMBounds(address));
    }
    private static native long _getMBounds(long address);

    /**
     * Region's bounds
     */
    public void setMBounds(PxBounds3 value) {
        checkNotNull();
        _setMBounds(address, value.getAddress());
    }
    private static native void _setMBounds(long address, long value);

    /**
     * Region's user-provided data
     */
    public NativeObject getMUserData() {
        checkNotNull();
        return NativeObject.wrapPointer(_getMUserData(address));
    }
    private static native long _getMUserData(long address);

    /**
     * Region's user-provided data
     */
    public void setMUserData(NativeObject value) {
        checkNotNull();
        _setMUserData(address, value.getAddress());
    }
    private static native void _setMUserData(long address, long value);

}
