package physx.physics;

/**
 * Collection of flags describing the filter actions to take for a collision pair.
 * <p>
 * <b>See also:</b> PxFilterFlags PxSimulationFilterShader PxSimulationFilterCallback
 */
public enum PxFilterFlagEnum {

    /**
     * Ignore the collision pair as long as the bounding volumes of the pair objects overlap.
     * <p>
     * Killed pairs will be ignored by the simulation and won't run through the filter again until one
     * of the following occurs:
     * <p>
     * \li The bounding volumes of the two objects overlap again (after being separated)
     * \li The user enforces a re-filtering (see #PxScene::resetFiltering())
     * <p>
     * <b>See also:</b> PxScene::resetFiltering()
     */
    eKILL(geteKILL()),
    /**
     * Ignore the collision pair as long as the bounding volumes of the pair objects overlap or until filtering relevant data changes for one of the collision objects.
     * <p>
     * Suppressed pairs will be ignored by the simulation and won't make another filter request until one
     * of the following occurs:
     * <p>
     * \li Same conditions as for killed pairs (see #eKILL)
     * \li The filter data or the filter object attributes change for one of the collision objects
     * <p>
     * <b>See also:</b> PxFilterData PxFilterObjectAttributes
     */
    eSUPPRESS(geteSUPPRESS()),
    /**
     * Invoke the filter callback (#PxSimulationFilterCallback::pairFound()) for this collision pair.
     * <p>
     * <b>See also:</b> PxSimulationFilterCallback
     */
    eCALLBACK(geteCALLBACK()),
    /**
     * Track this collision pair with the filter callback mechanism.
     * <p>
     * When the bounding volumes of the collision pair lose contact, the filter callback #PxSimulationFilterCallback::pairLost()
     * will be invoked. Furthermore, the filter status of the collision pair can be adjusted through #PxSimulationFilterCallback::statusChange()
     * once per frame (until a pairLost() notification occurs).
     * <p>
     * <b>See also:</b> PxSimulationFilterCallback
     */
    eNOTIFY(geteNOTIFY()),
    /**
     * Provided default to get standard behavior:
     * <p>
     * The application configure the pair's collision properties once when bounding volume overlap is found and
     * doesn't get asked again about that pair until overlap status or filter properties changes, or re-filtering is requested.
     * <p>
     * No notification is provided when bounding volume overlap is lost
     * <p>
     * The pair will not be killed or suppressed, so collision detection will be processed
     */
    eDEFAULT(geteDEFAULT());
    public final int value;
    
    PxFilterFlagEnum(int value) {
        this.value = value;
    }

    private static native int _geteKILL();
    private static int geteKILL() {
        de.fabmax.physxjni.Loader.load();
        return _geteKILL();
    }

    private static native int _geteSUPPRESS();
    private static int geteSUPPRESS() {
        de.fabmax.physxjni.Loader.load();
        return _geteSUPPRESS();
    }

    private static native int _geteCALLBACK();
    private static int geteCALLBACK() {
        de.fabmax.physxjni.Loader.load();
        return _geteCALLBACK();
    }

    private static native int _geteNOTIFY();
    private static int geteNOTIFY() {
        de.fabmax.physxjni.Loader.load();
        return _geteNOTIFY();
    }

    private static native int _geteDEFAULT();
    private static int geteDEFAULT() {
        de.fabmax.physxjni.Loader.load();
        return _geteDEFAULT();
    }

    public static PxFilterFlagEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxFilterFlagEnum: " + value);
    }

}
