package physx.cooking;

import physx.common.PxTypedStridedData_PxU16Const;
import physx.geometry.PxSimpleTriangleMesh;

/**
 * Descriptor class for #PxTriangleMesh.
 * <p>
 * Note that this class is derived from PxSimpleTriangleMesh which contains the members that describe the basic mesh.
 * The mesh data is *copied* when an PxTriangleMesh object is created from this descriptor. After the call the
 * user may discard the triangle data.
 * @see physx.geometry.PxTriangleMesh
 * @see physx.geometry.PxTriangleMeshGeometry
 * @see physx.physics.PxShape
 */
public class PxTriangleMeshDesc extends PxSimpleTriangleMesh {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxTriangleMeshDesc wrapPointer(long address) {
        return address != 0L ? new PxTriangleMeshDesc(address) : null;
    }
    
    public static PxTriangleMeshDesc arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxTriangleMeshDesc(long address) {
        super(address);
    }

    // Placed Constructors

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @return Stack allocated object of PxTriangleMeshDesc
     */
    public static PxTriangleMeshDesc createAt(long address) {
        __placement_new_PxTriangleMeshDesc(address);
        PxTriangleMeshDesc createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @return Stack allocated object of PxTriangleMeshDesc
     */
    public static <T> PxTriangleMeshDesc createAt(T allocator, Allocator<T> allocate) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxTriangleMeshDesc(address);
        PxTriangleMeshDesc createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxTriangleMeshDesc(long address);

    // Constructors

    /**
     * Constructor sets to default.
     */
    public PxTriangleMeshDesc() {
        address = _PxTriangleMeshDesc();
    }
    private static native long _PxTriangleMeshDesc();

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * Optional pointer to first material index, or NULL. There are PxSimpleTriangleMesh::numTriangles indices in total.
     * Caller may add materialIndexStride bytes to the pointer to access the next triangle.
     * <p>
     * When a triangle mesh collides with another object, a material is required at the collision point.
     * If materialIndices is NULL, then the material of the PxShape instance is used.
     * Otherwise, if the point of contact is on a triangle with index i, then the material index is determined as: 
     * PxMaterialTableIndex index = *(PxMaterialTableIndex *)(((PxU8*)materialIndices) + materialIndexStride * i);
     * <p>
     * If the contact point falls on a vertex or an edge, a triangle adjacent to the vertex or edge is selected, and its index
     * used to look up a material. The selection is arbitrary but consistent over time. 
     * <p>
     * <b>Default:</b> NULL
     */
    public PxTypedStridedData_PxU16Const getMaterialIndices() {
        checkNotNull();
        return PxTypedStridedData_PxU16Const.wrapPointer(_getMaterialIndices(address));
    }
    private static native long _getMaterialIndices(long address);

    /**
     * Optional pointer to first material index, or NULL. There are PxSimpleTriangleMesh::numTriangles indices in total.
     * Caller may add materialIndexStride bytes to the pointer to access the next triangle.
     * <p>
     * When a triangle mesh collides with another object, a material is required at the collision point.
     * If materialIndices is NULL, then the material of the PxShape instance is used.
     * Otherwise, if the point of contact is on a triangle with index i, then the material index is determined as: 
     * PxMaterialTableIndex index = *(PxMaterialTableIndex *)(((PxU8*)materialIndices) + materialIndexStride * i);
     * <p>
     * If the contact point falls on a vertex or an edge, a triangle adjacent to the vertex or edge is selected, and its index
     * used to look up a material. The selection is arbitrary but consistent over time. 
     * <p>
     * <b>Default:</b> NULL
     */
    public void setMaterialIndices(PxTypedStridedData_PxU16Const value) {
        checkNotNull();
        _setMaterialIndices(address, value.getAddress());
    }
    private static native void _setMaterialIndices(long address, long value);

}
