package physx.extensions;

import physx.NativeObject;
import physx.common.PxCollection;
import physx.physics.PxScene;

public class PxCollectionExt extends NativeObject {

    protected PxCollectionExt() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxCollectionExt wrapPointer(long address) {
        return address != 0L ? new PxCollectionExt(address) : null;
    }
    
    public static PxCollectionExt arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxCollectionExt(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Removes and releases all object from a collection.
     * <p>
     * The Collection itself is not released.
     * <p>
     * If the releaseExclusiveShapes flag is not set to true, release() will not be called on exclusive shapes.
     * <p>
     * It is assumed that the application holds a reference to each of the objects in the collection, with the exception of objects that are not releasable
     * (PxBase::isReleasable()). In general, objects that violate this assumption need to be removed from the collection prior to calling releaseObjects.
     * <p>
     * <b>Note:</b> when a shape is created with PxRigidActorExt::createExclusiveShape(), the only counted reference is held by the actor. 
     * If such a shape and its actor are present in the collection, the reference count will be decremented once when the actor is released, and once when the 
     * shape is released, resulting in undefined behavior. Shape reference counts can be incremented with PxShape::acquireReference().
     * @param collection to remove and release all object from.
     */
    public static void releaseObjects(PxCollection collection) {
        _releaseObjects(collection.getAddress());
    }
    private static native void _releaseObjects(long collection);

    /**
     * Removes and releases all object from a collection.
     * <p>
     * The Collection itself is not released.
     * <p>
     * If the releaseExclusiveShapes flag is not set to true, release() will not be called on exclusive shapes.
     * <p>
     * It is assumed that the application holds a reference to each of the objects in the collection, with the exception of objects that are not releasable
     * (PxBase::isReleasable()). In general, objects that violate this assumption need to be removed from the collection prior to calling releaseObjects.
     * <p>
     * <b>Note:</b> when a shape is created with PxRigidActorExt::createExclusiveShape(), the only counted reference is held by the actor. 
     * If such a shape and its actor are present in the collection, the reference count will be decremented once when the actor is released, and once when the 
     * shape is released, resulting in undefined behavior. Shape reference counts can be incremented with PxShape::acquireReference().
     * @param collection to remove and release all object from.
     * @param releaseExclusiveShapes if this parameter is set to false, release() will not be called on exclusive shapes.
     */
    public static void releaseObjects(PxCollection collection, boolean releaseExclusiveShapes) {
        _releaseObjects(collection.getAddress(), releaseExclusiveShapes);
    }
    private static native void _releaseObjects(long collection, boolean releaseExclusiveShapes);

    /**
     * Removes objects of a given type from a collection, potentially adding them to another collection.
     * <p>
     * \param[in,out] collection Collection from which objects are removed
     * @param concreteType PxConcreteType of sdk objects that should be removed 
     */
    public static void remove(PxCollection collection, short concreteType) {
        _remove(collection.getAddress(), concreteType);
    }
    private static native void _remove(long collection, short concreteType);

    /**
     * Removes objects of a given type from a collection, potentially adding them to another collection.
     * <p>
     * \param[in,out] collection Collection from which objects are removed
     * @param concreteType PxConcreteType of sdk objects that should be removed 
     * \param[in,out] to Optional collection to which the removed objects are added
     */
    public static void remove(PxCollection collection, short concreteType, PxCollection to) {
        _remove(collection.getAddress(), concreteType, to.getAddress());
    }
    private static native void _remove(long collection, short concreteType, long to);

    /**
     * Collects all objects from a PxScene.
     * <p>
     * This function creates a new collection from all objects that were added to the specified 
     * PxScene. Instances of the following types are included: PxActor, PxAggregate, 
     * PxArticulationReducedCoordinate and PxJoint (other PxConstraint types are not included).
     * <p>
     * This is a helper function to ease the creation of collections for serialization. 
     * The function PxSerialization.complete() can be used to complete the collection with required objects prior to 
     * serialization.
     * @param scene The PxScene instance from which objects are collected. See #PxScene
     * @return Collection to which objects are added. See #PxCollection
     * @see PxSerialization#complete
     */
    public static PxCollection createCollection(PxScene scene) {
        return PxCollection.wrapPointer(_createCollection(scene.getAddress()));
    }
    private static native long _createCollection(long scene);

}
