package physx.extensions;

/**
 * Used to specify which axes of a D6 joint are driven. 
 * <p>
 * Each drive is an implicit force-limited damped spring:
 * <p>
 * force = spring * (target position - position) + damping * (targetVelocity - velocity)
 * <p>
 * Alternatively, the spring may be configured to generate a specified acceleration instead of a force.
 * <p>
 * A linear axis is affected by drive only if the corresponding drive flag is set. There are two possible models
 * for angular drive: swing/twist, which may be used to drive one or more angular degrees of freedom, or slerp,
 * which may only be used to drive all three angular degrees simultaneously.
 * <p>
 * <b>See also:</b> PxD6Joint
 */
public enum PxD6DriveEnum {

    /**
     * drive along the X-axis
     */
    eX(geteX()),
    /**
     * drive along the Y-axis
     */
    eY(geteY()),
    /**
     * drive along the Z-axis
     */
    eZ(geteZ()),
    /**
     * rotational drive around the Y- and Z-axis
     */
    eSWING(geteSWING()),
    /**
     * rotational drive around the X-axis
     */
    eTWIST(geteTWIST()),
    /**
     * drive of all three angular degrees along a SLERP-path (note: takes precedence over eSWING/eTWIST)
     */
    eSLERP(geteSLERP());
    public final int value;
    
    PxD6DriveEnum(int value) {
        this.value = value;
    }

    private static native int _geteX();
    private static int geteX() {
        de.fabmax.physxjni.Loader.load();
        return _geteX();
    }

    private static native int _geteY();
    private static int geteY() {
        de.fabmax.physxjni.Loader.load();
        return _geteY();
    }

    private static native int _geteZ();
    private static int geteZ() {
        de.fabmax.physxjni.Loader.load();
        return _geteZ();
    }

    private static native int _geteSWING();
    private static int geteSWING() {
        de.fabmax.physxjni.Loader.load();
        return _geteSWING();
    }

    private static native int _geteTWIST();
    private static int geteTWIST() {
        de.fabmax.physxjni.Loader.load();
        return _geteTWIST();
    }

    private static native int _geteSLERP();
    private static int geteSLERP() {
        de.fabmax.physxjni.Loader.load();
        return _geteSLERP();
    }

    public static PxD6DriveEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxD6DriveEnum: " + value);
    }

}
