package physx.extensions;

import physx.common.PxBase;

/**
 * A joint that connects an existing revolute joint to an existing prismatic joint,
 * and constrains their relative angular/linear velocity and position with respect to each other.
 * @see PxJoint
 */
public class PxRackAndPinionJoint extends PxJoint {

    protected PxRackAndPinionJoint() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxRackAndPinionJoint wrapPointer(long address) {
        return address != 0L ? new PxRackAndPinionJoint(address) : null;
    }
    
    public static PxRackAndPinionJoint arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxRackAndPinionJoint(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Set the hinge &amp; prismatic joints connected by the rack &amp; pinion joint.
     * <p>
     * The passed hinge joint can be either PxRevoluteJoint, PxD6Joint or PxArticulationJointReducedCoordinate.
     * The passed prismatic joint can be either PxPrismaticJoint or PxD6Joint.
     * <p>
     * Note that these joints are only used to compute the positional error correction term,
     * used to adjust potential drift between jointed actors. The rack &amp; pinion joint can run without
     * calling this function, but in that case some visible overlap may develop over time between
     * the teeth of the rack &amp; pinion meshes.
     * <p>
     * <b>Note:</b> Calling this function resets the internal positional error correction term.
     * @param hinge  The hinge joint (pinion)
     * @param prismatic The prismatic joint (rack)
     * @return  true if success
     */
    public boolean setJoints(PxBase hinge, PxBase prismatic) {
        checkNotNull();
        return _setJoints(address, hinge.getAddress(), prismatic.getAddress());
    }
    private static native boolean _setJoints(long address, long hinge, long prismatic);

    /**
     * Set the desired ratio directly.
     * <p>
     * <b>Note:</b> You may need to use a negative gear ratio if the joint frames of involved actors are not oriented in the same direction.
     * <p>
     * <b>Note:</b> Calling this function resets the internal positional error correction term.
     * @param ratio Desired ratio between the hinge and the prismatic.
     */
    public void setRatio(float ratio) {
        checkNotNull();
        _setRatio(address, ratio);
    }
    private static native void _setRatio(long address, float ratio);

    /**
     * Get the ratio.
     * @return  Current ratio
     */
    public float getRatio() {
        checkNotNull();
        return _getRatio(address);
    }
    private static native float _getRatio(long address);

    /**
     * Set the desired ratio indirectly.
     * <p>
     * This is a simple helper function that computes the ratio from passed data:
     * <p>
     * ratio = (PI*2*nbRackTeeth)/(rackLength*nbPinionTeeth)
     * <p>
     * <b>Note:</b> Calling this function resets the internal positional error correction term.
     * @param nbRackTeeth  Number of teeth on the rack (cannot be zero)
     * @param nbPinionTeeth Number of teeth on the pinion (cannot be zero)
     * @param rackLength  Length of the rack
     * @return  true if success
     */
    public boolean setData(int nbRackTeeth, int nbPinionTeeth, float rackLength) {
        checkNotNull();
        return _setData(address, nbRackTeeth, nbPinionTeeth, rackLength);
    }
    private static native boolean _setData(long address, int nbRackTeeth, int nbPinionTeeth, float rackLength);

}
