package physx.geometry;

import physx.NativeObject;
import physx.common.PxBounds3;
import physx.common.PxTransform;
import physx.common.PxVec3;
import physx.physics.PxHitFlags;
import physx.physics.PxRaycastHit;
import physx.physics.PxSweepHit;

/**
 * Collection of geometry object queries (sweeps, raycasts, overlaps, ...).
 */
public class PxGeometryQuery extends NativeObject {

    protected PxGeometryQuery() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxGeometryQuery wrapPointer(long address) {
        return address != 0L ? new PxGeometryQuery(address) : null;
    }
    
    public static PxGeometryQuery arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxGeometryQuery(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Sweep a specified geometry object in space and test for collision with a given object.
     * <p>
     * The following combinations are supported.
     * <p>
     * \li PxSphereGeometry vs. {PxSphereGeometry, PxPlaneGeometry, PxCapsuleGeometry, PxBoxGeometry, PxConvexMeshGeometry, PxTriangleMeshGeometry, PxHeightFieldGeometry}
     * \li PxCapsuleGeometry vs. {PxSphereGeometry, PxPlaneGeometry, PxCapsuleGeometry, PxBoxGeometry, PxConvexMeshGeometry, PxTriangleMeshGeometry, PxHeightFieldGeometry}
     * \li PxBoxGeometry vs. {PxSphereGeometry, PxPlaneGeometry, PxCapsuleGeometry, PxBoxGeometry, PxConvexMeshGeometry, PxTriangleMeshGeometry, PxHeightFieldGeometry}
     * \li PxConvexMeshGeometry vs. {PxSphereGeometry, PxPlaneGeometry, PxCapsuleGeometry, PxBoxGeometry, PxConvexMeshGeometry, PxTriangleMeshGeometry, PxHeightFieldGeometry}
     * @param unitDir   Normalized direction along which object geom0 should be swept
     * @param maxDist   Maximum sweep distance, has to be in the [0, inf) range
     * @param geom0   The geometry object to sweep. Supported geometries are #PxSphereGeometry, #PxCapsuleGeometry, #PxBoxGeometry and #PxConvexMeshGeometry
     * @param pose0   Pose of the geometry object to sweep
     * @param geom1   The geometry object to test the sweep against
     * @param pose1   Pose of the geometry object to sweep against
     * @param sweepHit  The sweep hit information. Only valid if this method returns true.
     * @return True if the swept geometry object geom0 hits the object geom1
     * @see physx.physics.PxGeomSweepHit
     * @see PxGeometry
     * @see physx.common.PxTransform
     */
    public static boolean sweep(PxVec3 unitDir, float maxDist, PxGeometry geom0, PxTransform pose0, PxGeometry geom1, PxTransform pose1, PxSweepHit sweepHit) {
        return _sweep(unitDir.getAddress(), maxDist, geom0.getAddress(), pose0.getAddress(), geom1.getAddress(), pose1.getAddress(), sweepHit.getAddress());
    }
    private static native boolean _sweep(long unitDir, float maxDist, long geom0, long pose0, long geom1, long pose1, long sweepHit);

    /**
     * Sweep a specified geometry object in space and test for collision with a given object.
     * <p>
     * The following combinations are supported.
     * <p>
     * \li PxSphereGeometry vs. {PxSphereGeometry, PxPlaneGeometry, PxCapsuleGeometry, PxBoxGeometry, PxConvexMeshGeometry, PxTriangleMeshGeometry, PxHeightFieldGeometry}
     * \li PxCapsuleGeometry vs. {PxSphereGeometry, PxPlaneGeometry, PxCapsuleGeometry, PxBoxGeometry, PxConvexMeshGeometry, PxTriangleMeshGeometry, PxHeightFieldGeometry}
     * \li PxBoxGeometry vs. {PxSphereGeometry, PxPlaneGeometry, PxCapsuleGeometry, PxBoxGeometry, PxConvexMeshGeometry, PxTriangleMeshGeometry, PxHeightFieldGeometry}
     * \li PxConvexMeshGeometry vs. {PxSphereGeometry, PxPlaneGeometry, PxCapsuleGeometry, PxBoxGeometry, PxConvexMeshGeometry, PxTriangleMeshGeometry, PxHeightFieldGeometry}
     * @param unitDir   Normalized direction along which object geom0 should be swept
     * @param maxDist   Maximum sweep distance, has to be in the [0, inf) range
     * @param geom0   The geometry object to sweep. Supported geometries are #PxSphereGeometry, #PxCapsuleGeometry, #PxBoxGeometry and #PxConvexMeshGeometry
     * @param pose0   Pose of the geometry object to sweep
     * @param geom1   The geometry object to test the sweep against
     * @param pose1   Pose of the geometry object to sweep against
     * @param sweepHit  The sweep hit information. Only valid if this method returns true.
     * @param hitFlags   Specify which properties per hit should be computed and written to result hit array. Combination of #PxHitFlag flags
     * @return True if the swept geometry object geom0 hits the object geom1
     * @see physx.physics.PxGeomSweepHit
     * @see PxGeometry
     * @see physx.common.PxTransform
     */
    public static boolean sweep(PxVec3 unitDir, float maxDist, PxGeometry geom0, PxTransform pose0, PxGeometry geom1, PxTransform pose1, PxSweepHit sweepHit, PxHitFlags hitFlags) {
        return _sweep(unitDir.getAddress(), maxDist, geom0.getAddress(), pose0.getAddress(), geom1.getAddress(), pose1.getAddress(), sweepHit.getAddress(), hitFlags.getAddress());
    }
    private static native boolean _sweep(long unitDir, float maxDist, long geom0, long pose0, long geom1, long pose1, long sweepHit, long hitFlags);

    /**
     * Sweep a specified geometry object in space and test for collision with a given object.
     * <p>
     * The following combinations are supported.
     * <p>
     * \li PxSphereGeometry vs. {PxSphereGeometry, PxPlaneGeometry, PxCapsuleGeometry, PxBoxGeometry, PxConvexMeshGeometry, PxTriangleMeshGeometry, PxHeightFieldGeometry}
     * \li PxCapsuleGeometry vs. {PxSphereGeometry, PxPlaneGeometry, PxCapsuleGeometry, PxBoxGeometry, PxConvexMeshGeometry, PxTriangleMeshGeometry, PxHeightFieldGeometry}
     * \li PxBoxGeometry vs. {PxSphereGeometry, PxPlaneGeometry, PxCapsuleGeometry, PxBoxGeometry, PxConvexMeshGeometry, PxTriangleMeshGeometry, PxHeightFieldGeometry}
     * \li PxConvexMeshGeometry vs. {PxSphereGeometry, PxPlaneGeometry, PxCapsuleGeometry, PxBoxGeometry, PxConvexMeshGeometry, PxTriangleMeshGeometry, PxHeightFieldGeometry}
     * @param unitDir   Normalized direction along which object geom0 should be swept
     * @param maxDist   Maximum sweep distance, has to be in the [0, inf) range
     * @param geom0   The geometry object to sweep. Supported geometries are #PxSphereGeometry, #PxCapsuleGeometry, #PxBoxGeometry and #PxConvexMeshGeometry
     * @param pose0   Pose of the geometry object to sweep
     * @param geom1   The geometry object to test the sweep against
     * @param pose1   Pose of the geometry object to sweep against
     * @param sweepHit  The sweep hit information. Only valid if this method returns true.
     * @param hitFlags   Specify which properties per hit should be computed and written to result hit array. Combination of #PxHitFlag flags
     * @param inflation  Surface of the swept shape is additively extruded in the normal direction, rounding corners and edges.
     * @return True if the swept geometry object geom0 hits the object geom1
     * @see physx.physics.PxGeomSweepHit
     * @see PxGeometry
     * @see physx.common.PxTransform
     */
    public static boolean sweep(PxVec3 unitDir, float maxDist, PxGeometry geom0, PxTransform pose0, PxGeometry geom1, PxTransform pose1, PxSweepHit sweepHit, PxHitFlags hitFlags, float inflation) {
        return _sweep(unitDir.getAddress(), maxDist, geom0.getAddress(), pose0.getAddress(), geom1.getAddress(), pose1.getAddress(), sweepHit.getAddress(), hitFlags.getAddress(), inflation);
    }
    private static native boolean _sweep(long unitDir, float maxDist, long geom0, long pose0, long geom1, long pose1, long sweepHit, long hitFlags, float inflation);

    /**
     * Overlap test for two geometry objects.
     * <p>
     * All combinations are supported except:
     * \li PxPlaneGeometry vs. {PxPlaneGeometry, PxTriangleMeshGeometry, PxHeightFieldGeometry}
     * \li PxTriangleMeshGeometry vs. PxHeightFieldGeometry
     * \li PxHeightFieldGeometry vs. PxHeightFieldGeometry
     * \li Anything involving PxParticleSystemGeometry, PxTetrahedronMeshGeometry, or PxConvexCoreGeometry
     * @param geom0   The first geometry object
     * @param pose0   Pose of the first geometry object
     * @param geom1   The second geometry object
     * @param pose1   Pose of the second geometry object
     * @return True if the two geometry objects overlap
     * @see PxGeometry
     * @see physx.common.PxTransform
     */
    public static boolean overlap(PxGeometry geom0, PxTransform pose0, PxGeometry geom1, PxTransform pose1) {
        return _overlap(geom0.getAddress(), pose0.getAddress(), geom1.getAddress(), pose1.getAddress());
    }
    private static native boolean _overlap(long geom0, long pose0, long geom1, long pose1);

    /**
     * Raycast test against a geometry object.
     * <p>
     * All geometry types are supported except PxParticleSystemGeometry and PxTetrahedronMeshGeometry.
     * @param origin   The origin of the ray to test the geometry object against
     * @param unitDir   Normalized direction of the ray to test the geometry object against
     * @param geom    The geometry object to test the ray against
     * @param pose    Pose of the geometry object
     * @param maxDist   Maximum ray length, has to be in the [0, inf) range
     * @param hitFlags   Specification of the kind of information to retrieve on hit. Combination of #PxHitFlag flags
     * @param maxHits   max number of returned hits = size of 'rayHits' buffer
     * @param rayHits   Raycast hits information
     * @return Number of hits between the ray and the geometry object
     * @see physx.physics.PxGeomRaycastHit
     * @see PxGeometry
     * @see physx.common.PxTransform
     */
    public static int raycast(PxVec3 origin, PxVec3 unitDir, PxGeometry geom, PxTransform pose, float maxDist, PxHitFlags hitFlags, int maxHits, PxRaycastHit rayHits) {
        return _raycast(origin.getAddress(), unitDir.getAddress(), geom.getAddress(), pose.getAddress(), maxDist, hitFlags.getAddress(), maxHits, rayHits.getAddress());
    }
    private static native int _raycast(long origin, long unitDir, long geom, long pose, float maxDist, long hitFlags, int maxHits, long rayHits);

    /**
     * Computes distance between a point and a geometry object.
     * <p>
     * Currently supported geometry objects: box, sphere, capsule, convex, mesh.
     * <p>
     * <b>Note:</b> For meshes, only the BVH34 midphase data-structure is supported.
     * @param point   The point P
     * @param geom    The geometry object
     * @param pose    Pose of the geometry object
     * @return Square distance between the point and the geom object, or 0.0 if the point is inside the object, or -1.0 if an error occured (geometry type is not supported, or invalid pose)
     * @see PxGeometry
     * @see physx.common.PxTransform
     */
    public static float pointDistance(PxVec3 point, PxGeometry geom, PxTransform pose) {
        return _pointDistance(point.getAddress(), geom.getAddress(), pose.getAddress());
    }
    private static native float _pointDistance(long point, long geom, long pose);

    /**
     * Computes distance between a point and a geometry object.
     * <p>
     * Currently supported geometry objects: box, sphere, capsule, convex, mesh.
     * <p>
     * <b>Note:</b> For meshes, only the BVH34 midphase data-structure is supported.
     * @param point   The point P
     * @param geom    The geometry object
     * @param pose    Pose of the geometry object
     * @param closestPoint Optionally returned closest point to P on the geom object. Only valid when returned distance is strictly positive.
     * @return Square distance between the point and the geom object, or 0.0 if the point is inside the object, or -1.0 if an error occured (geometry type is not supported, or invalid pose)
     * @see PxGeometry
     * @see physx.common.PxTransform
     */
    public static float pointDistance(PxVec3 point, PxGeometry geom, PxTransform pose, PxVec3 closestPoint) {
        return _pointDistance(point.getAddress(), geom.getAddress(), pose.getAddress(), closestPoint.getAddress());
    }
    private static native float _pointDistance(long point, long geom, long pose, long closestPoint);

    /**
     * computes the bounds for a geometry object
     * @param bounds  Returned computed bounds
     * @param geom   The geometry object
     * @param pose   Pose of the geometry object
     * @see PxGeometry
     * @see physx.common.PxTransform
     */
    public static void computeGeomBounds(PxBounds3 bounds, PxGeometry geom, PxTransform pose) {
        _computeGeomBounds(bounds.getAddress(), geom.getAddress(), pose.getAddress());
    }
    private static native void _computeGeomBounds(long bounds, long geom, long pose);

    /**
     * computes the bounds for a geometry object
     * @param bounds  Returned computed bounds
     * @param geom   The geometry object
     * @param pose   Pose of the geometry object
     * @param inflation Scale factor for computed bounds. The geom's extents are multiplied by this value.
     * @see PxGeometry
     * @see physx.common.PxTransform
     */
    public static void computeGeomBounds(PxBounds3 bounds, PxGeometry geom, PxTransform pose, float inflation) {
        _computeGeomBounds(bounds.getAddress(), geom.getAddress(), pose.getAddress(), inflation);
    }
    private static native void _computeGeomBounds(long bounds, long geom, long pose, float inflation);

    /**
     * Checks if provided geometry is valid.
     * @param geom The geometry object.
     * @return True if geometry is valid.
     * @see PxGeometry
     */
    public static boolean isValid(PxGeometry geom) {
        return _isValid(geom.getAddress());
    }
    private static native boolean _isValid(long geom);

}
