package physx.geometry;


/**
 * Tetrahedron mesh geometry class.
 * <p>
 * This class wraps a tetrahedron mesh such that it can be used in contexts where a PxGeometry type is needed.
 */
public class PxTetrahedronMeshGeometry extends PxGeometry {

    protected PxTetrahedronMeshGeometry() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxTetrahedronMeshGeometry wrapPointer(long address) {
        return address != 0L ? new PxTetrahedronMeshGeometry(address) : null;
    }
    
    public static PxTetrahedronMeshGeometry arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxTetrahedronMeshGeometry(long address) {
        super(address);
    }

    // Constructors

    /**
     * Constructor. By default creates an empty object with a NULL mesh and identity scale.
     */
    public PxTetrahedronMeshGeometry(PxTetrahedronMesh mesh) {
        address = _PxTetrahedronMeshGeometry(mesh.getAddress());
    }
    private static native long _PxTetrahedronMeshGeometry(long mesh);

    // Attributes

    /**
     * A reference to the mesh object.
     */
    public PxTetrahedronMesh getTetrahedronMesh() {
        checkNotNull();
        return PxTetrahedronMesh.wrapPointer(_getTetrahedronMesh(address));
    }
    private static native long _getTetrahedronMesh(long address);

    /**
     * A reference to the mesh object.
     */
    public void setTetrahedronMesh(PxTetrahedronMesh value) {
        checkNotNull();
        _setTetrahedronMesh(address, value.getAddress());
    }
    private static native void _setTetrahedronMesh(long address, long value);

    // Functions

    /**
     * Returns true if the geometry is valid.
     * @return  True if the current settings are valid for shape creation.
     * <p>
     * <b>Note:</b> A valid tetrahedron mesh has a positive scale value in each direction (scale.scale.x &gt; 0, scale.scale.y &gt; 0, scale.scale.z &gt; 0).
     * It is illegal to call PxPhysics::createShape with a tetrahedron mesh that has zero extents in any direction.
     */
    public boolean isValid() {
        checkNotNull();
        return _isValid(address);
    }
    private static native boolean _isValid(long address);

}
