package physx.physics;

import physx.NativeObject;
import physx.common.PxVec3;

/**
 * Defines a spatial tendon attachment point on a link.
 */
public class PxArticulationAttachment extends NativeObject {

    protected PxArticulationAttachment() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxArticulationAttachment wrapPointer(long address) {
        return address != 0L ? new PxArticulationAttachment(address) : null;
    }
    
    public static PxArticulationAttachment arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxArticulationAttachment(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * user can assign this to whatever, usually to create a 1:1 relationship with a user object.
     */
    public NativeObject getUserData() {
        checkNotNull();
        return NativeObject.wrapPointer(_getUserData(address));
    }
    private static native long _getUserData(long address);

    /**
     * user can assign this to whatever, usually to create a 1:1 relationship with a user object.
     */
    public void setUserData(NativeObject value) {
        checkNotNull();
        _setUserData(address, value.getAddress());
    }
    private static native void _setUserData(long address, long value);

    // Functions

    /**
     * Sets the spring rest length for the sub-tendon from the root to this leaf attachment.
     * <p>
     * Setting this on non-leaf attachments has no effect.
     * @param restLength The rest length of the spring.
     * <b>Default:</b> 0
     * @see #isLeaf
     */
    public void setRestLength(float restLength) {
        checkNotNull();
        _setRestLength(address, restLength);
    }
    private static native void _setRestLength(long address, float restLength);

    /**
     * Gets the spring rest length for the sub-tendon from the root to this leaf attachment.
     * @return The rest length.
     * @see #setRestLength
     */
    public float getRestLength() {
        checkNotNull();
        return _getRestLength(address);
    }
    private static native float _getRestLength(long address);

    /**
     * Sets the low and high limit on the length of the sub-tendon from the root to this leaf attachment.
     * <p>
     * Setting this on non-leaf attachments has no effect.
     * @param parameters Struct with the low and high limit.
     * <b>Default:</b> (PX_MAX_F32, -PX_MAX_F32) (i.e. an invalid configuration that can only work if stiffness is zero)
     * @see #isLeaf
     */
    public void setLimitParameters(PxArticulationTendonLimit parameters) {
        checkNotNull();
        _setLimitParameters(address, parameters.getAddress());
    }
    private static native void _setLimitParameters(long address, long parameters);

    /**
     * Gets the low and high limit on the length of the sub-tendon from the root to this leaf attachment.
     * @return Struct with the low and high limit.
     * @see #setLimitParameters
     */
    public PxArticulationTendonLimit getLimitParameters() {
        checkNotNull();
        return PxArticulationTendonLimit.wrapPointer(_getLimitParameters(address));
    }
    private static native long _getLimitParameters(long address);

    /**
     * Sets the attachment's relative offset in the link actor frame.
     * @param offset The relative offset in the link actor frame.
     * @see #getRelativeOffset
     */
    public void setRelativeOffset(PxVec3 offset) {
        checkNotNull();
        _setRelativeOffset(address, offset.getAddress());
    }
    private static native void _setRelativeOffset(long address, long offset);

    /**
     * Gets the attachment's relative offset in the link actor frame.
     * @return The relative offset in the link actor frame.
     * @see #setRelativeOffset
     */
    public PxVec3 getRelativeOffset() {
        checkNotNull();
        return PxVec3.wrapPointer(_getRelativeOffset(address));
    }
    private static native long _getRelativeOffset(long address);

    /**
     * Sets the attachment coefficient.
     * @param coefficient The scale that the distance between this attachment and its parent is multiplied by when summing up the spatial tendon's length.
     * @see #getCoefficient
     */
    public void setCoefficient(float coefficient) {
        checkNotNull();
        _setCoefficient(address, coefficient);
    }
    private static native void _setCoefficient(long address, float coefficient);

    /**
     * Gets the attachment coefficient.
     * @return The scale that the distance between this attachment and its parent is multiplied by when summing up the spatial tendon's length.
     * @see #setCoefficient
     */
    public float getCoefficient() {
        checkNotNull();
        return _getCoefficient(address);
    }
    private static native float _getCoefficient(long address);

    /**
     * Gets the articulation link.
     * @return The articulation link that this attachment is attached to.
     */
    public PxArticulationLink getLink() {
        checkNotNull();
        return PxArticulationLink.wrapPointer(_getLink(address));
    }
    private static native long _getLink(long address);

    /**
     * Gets the parent attachment.
     * @return The parent attachment.
     */
    public PxArticulationAttachment getParent() {
        checkNotNull();
        return PxArticulationAttachment.wrapPointer(_getParent(address));
    }
    private static native long _getParent(long address);

    /**
     * Indicates that this attachment is a leaf, and thus defines a sub-tendon from the root to this attachment.
     * @return True: This attachment is a leaf and has zero children; False: Not a leaf.
     */
    public boolean isLeaf() {
        checkNotNull();
        return _isLeaf(address);
    }
    private static native boolean _isLeaf(long address);

    /**
     * Gets the spatial tendon that the attachment is a part of.
     * @return The tendon.
     * @see PxArticulationSpatialTendon
     */
    public PxArticulationSpatialTendon getTendon() {
        checkNotNull();
        return PxArticulationSpatialTendon.wrapPointer(_getTendon(address));
    }
    private static native long _getTendon(long address);

    /**
     * Releases the attachment.
     * <p>
     * <b>Note:</b> Releasing the attachment is not allowed while the articulation is in a scene. In order to
     * release the attachment, remove and then re-add the articulation to the scene.
     */
    public void release() {
        checkNotNull();
        _release(address);
    }
    private static native void _release(long address);

}
