package physx.physics;

/**
 * Enum for selecting the friction algorithm used for simulation.
 * <p>
 * #PxFrictionType::ePATCH is the default friction logic (Couloumb type friction model). Friction gets computed per contact patch.
 * Up to two contact points lying in the contact patch area are selected as friction anchors to which friction impulses are applied. If there
 * are more than two contact points, to select anchors from, the anchors are selected using a heuristic that tries to maximize the distance
 * between the anchors within the contact patch area. For each contact patch, two perpendicular axes of the contact patch plane are selected.
 * A 1D-constraint along each of the two axes is used to implement friction at a friction anchor point. Note that the two axes are processed
 * separately when the PGS solver type is selected. This can lead to asymmetries when transitioning from dynamic to static friction and vice
 * versa in certain edge cases. The TGS solver type, on the other hand, works with the combined impulse along the two axes and as such avoids
 * this potential problem, but this is slightly more computationally expensive. Another difference between TGS and PGS is that TGS applies
 * friction throughout all position and all velocity iterations, while PGS by default applies friction throughout the last 3 position iterations
 * and all velocity iterations (unless #PxSceneFlag::eENABLE_FRICTION_EVERY_ITERATION is used).
 * <p>
 * #PxFrictionType::eFRICTION_COUNT is the total number of friction models supported by the SDK.
 */
public enum PxFrictionTypeEnum {

    /**
     * Select default patch-friction model.
     */
    ePATCH(getePATCH()),
    eONE_DIRECTIONAL(geteONE_DIRECTIONAL()),
    eTWO_DIRECTIONAL(geteTWO_DIRECTIONAL()),
    eFRICTION_COUNT(geteFRICTION_COUNT());
    public final int value;
    
    PxFrictionTypeEnum(int value) {
        this.value = value;
    }

    private static native int _getePATCH();
    private static int getePATCH() {
        de.fabmax.physxjni.Loader.load();
        return _getePATCH();
    }

    private static native int _geteONE_DIRECTIONAL();
    private static int geteONE_DIRECTIONAL() {
        de.fabmax.physxjni.Loader.load();
        return _geteONE_DIRECTIONAL();
    }

    private static native int _geteTWO_DIRECTIONAL();
    private static int geteTWO_DIRECTIONAL() {
        de.fabmax.physxjni.Loader.load();
        return _geteTWO_DIRECTIONAL();
    }

    private static native int _geteFRICTION_COUNT();
    private static int geteFRICTION_COUNT() {
        de.fabmax.physxjni.Loader.load();
        return _geteFRICTION_COUNT();
    }

    public static PxFrictionTypeEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxFrictionTypeEnum: " + value);
    }

}
