package physx.physics;

import physx.NativeObject;

/**
 * Scene query filtering callbacks.
 * <p>
 * Custom filtering logic for scene query intersection candidates. If an intersection candidate object passes the data based filter
 * (see #PxQueryFilterData), filtering callbacks are executed if requested (see #PxQueryFilterData.flags)
 * <p>
 * \li If #PxQueryFlag::ePREFILTER is set, the preFilter function runs before exact intersection tests.
 * If this function returns #PxQueryHitType::eTOUCH or #PxQueryHitType::eBLOCK, exact testing is performed to
 * determine the intersection location.
 * <p>
 * The preFilter function may overwrite the copy of queryFlags it receives as an argument to specify any of #PxHitFlag::eMODIFIABLE_FLAGS
 * on a per-shape basis. Changes apply only to the shape being filtered, and changes to other flags are ignored.
 * <p>
 * \li If #PxQueryFlag::ePREFILTER is not set, precise intersection testing is performed using the original query's filterData.flags.
 * <p>
 * \li If #PxQueryFlag::ePOSTFILTER is set, the postFilter function is called for each intersection to determine the touch/block status.
 * This overrides any touch/block status previously returned from the preFilter function for this shape.
 * <p>
 * Filtering calls are not guaranteed to be sorted along the ray or sweep direction.
 * @see PxQueryFlags
 * @see PxHitFlags
 */
public class PxQueryFilterCallback extends NativeObject {

    protected PxQueryFilterCallback() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxQueryFilterCallback wrapPointer(long address) {
        return address != 0L ? new PxQueryFilterCallback(address) : null;
    }
    
    public static PxQueryFilterCallback arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxQueryFilterCallback(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

}
