package de.flapdoodle.embed.mongo.transitions;

import de.flapdoodle.embed.mongo.commands.MongoRestoreArguments;
import de.flapdoodle.embed.mongo.commands.ServerAddress;
import de.flapdoodle.embed.process.types.ProcessArguments;
import de.flapdoodle.reverse.StateID;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link MongoRestoreProcessArguments}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableMongoRestoreProcessArguments.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableMongoRestoreProcessArguments
    extends MongoRestoreProcessArguments {
  private final StateID<ProcessArguments> destination;
  private final StateID<ServerAddress> serverAddress;
  private final StateID<MongoRestoreArguments> arguments;

  private ImmutableMongoRestoreProcessArguments(ImmutableMongoRestoreProcessArguments.Builder builder) {
    if (builder.destination != null) {
      initShim.destination(builder.destination);
    }
    if (builder.serverAddress != null) {
      initShim.serverAddress(builder.serverAddress);
    }
    if (builder.arguments != null) {
      initShim.arguments(builder.arguments);
    }
    this.destination = initShim.destination();
    this.serverAddress = initShim.serverAddress();
    this.arguments = initShim.arguments();
    this.initShim = null;
  }

  private ImmutableMongoRestoreProcessArguments(
      StateID<ProcessArguments> destination,
      StateID<ServerAddress> serverAddress,
      StateID<MongoRestoreArguments> arguments) {
    this.destination = destination;
    this.serverAddress = serverAddress;
    this.arguments = arguments;
    this.initShim = null;
  }

  private static final byte STAGE_INITIALIZING = -1;
  private static final byte STAGE_UNINITIALIZED = 0;
  private static final byte STAGE_INITIALIZED = 1;
  private transient volatile InitShim initShim = new InitShim();

  private final class InitShim {
    private byte destinationBuildStage = STAGE_UNINITIALIZED;
    private StateID<ProcessArguments> destination;

    StateID<ProcessArguments> destination() {
      if (destinationBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (destinationBuildStage == STAGE_UNINITIALIZED) {
        destinationBuildStage = STAGE_INITIALIZING;
        this.destination = Objects.requireNonNull(ImmutableMongoRestoreProcessArguments.super.destination(), "destination");
        destinationBuildStage = STAGE_INITIALIZED;
      }
      return this.destination;
    }

    void destination(StateID<ProcessArguments> destination) {
      this.destination = destination;
      destinationBuildStage = STAGE_INITIALIZED;
    }

    private byte serverAddressBuildStage = STAGE_UNINITIALIZED;
    private StateID<ServerAddress> serverAddress;

    StateID<ServerAddress> serverAddress() {
      if (serverAddressBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (serverAddressBuildStage == STAGE_UNINITIALIZED) {
        serverAddressBuildStage = STAGE_INITIALIZING;
        this.serverAddress = Objects.requireNonNull(ImmutableMongoRestoreProcessArguments.super.serverAddress(), "serverAddress");
        serverAddressBuildStage = STAGE_INITIALIZED;
      }
      return this.serverAddress;
    }

    void serverAddress(StateID<ServerAddress> serverAddress) {
      this.serverAddress = serverAddress;
      serverAddressBuildStage = STAGE_INITIALIZED;
    }

    private byte argumentsBuildStage = STAGE_UNINITIALIZED;
    private StateID<MongoRestoreArguments> arguments;

    StateID<MongoRestoreArguments> arguments() {
      if (argumentsBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (argumentsBuildStage == STAGE_UNINITIALIZED) {
        argumentsBuildStage = STAGE_INITIALIZING;
        this.arguments = Objects.requireNonNull(ImmutableMongoRestoreProcessArguments.super.arguments(), "arguments");
        argumentsBuildStage = STAGE_INITIALIZED;
      }
      return this.arguments;
    }

    void arguments(StateID<MongoRestoreArguments> arguments) {
      this.arguments = arguments;
      argumentsBuildStage = STAGE_INITIALIZED;
    }

    private String formatInitCycleMessage() {
      List<String> attributes = new ArrayList<>();
      if (destinationBuildStage == STAGE_INITIALIZING) attributes.add("destination");
      if (serverAddressBuildStage == STAGE_INITIALIZING) attributes.add("serverAddress");
      if (argumentsBuildStage == STAGE_INITIALIZING) attributes.add("arguments");
      return "Cannot build MongoRestoreProcessArguments, attribute initializers form cycle " + attributes;
    }
  }

  /**
   * @return The value of the {@code destination} attribute
   */
  @Override
  public StateID<ProcessArguments> destination() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.destination()
        : this.destination;
  }

  /**
   * @return The value of the {@code serverAddress} attribute
   */
  @Override
  public StateID<ServerAddress> serverAddress() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.serverAddress()
        : this.serverAddress;
  }

  /**
   * @return The value of the {@code arguments} attribute
   */
  @Override
  public StateID<MongoRestoreArguments> arguments() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.arguments()
        : this.arguments;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongoRestoreProcessArguments#destination() destination} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for destination
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableMongoRestoreProcessArguments withDestination(StateID<ProcessArguments> value) {
    if (this.destination == value) return this;
    StateID<ProcessArguments> newValue = Objects.requireNonNull(value, "destination");
    return new ImmutableMongoRestoreProcessArguments(newValue, this.serverAddress, this.arguments);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongoRestoreProcessArguments#serverAddress() serverAddress} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for serverAddress
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableMongoRestoreProcessArguments withServerAddress(StateID<ServerAddress> value) {
    if (this.serverAddress == value) return this;
    StateID<ServerAddress> newValue = Objects.requireNonNull(value, "serverAddress");
    return new ImmutableMongoRestoreProcessArguments(this.destination, newValue, this.arguments);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongoRestoreProcessArguments#arguments() arguments} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for arguments
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableMongoRestoreProcessArguments withArguments(StateID<MongoRestoreArguments> value) {
    if (this.arguments == value) return this;
    StateID<MongoRestoreArguments> newValue = Objects.requireNonNull(value, "arguments");
    return new ImmutableMongoRestoreProcessArguments(this.destination, this.serverAddress, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableMongoRestoreProcessArguments} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableMongoRestoreProcessArguments
        && equalTo(0, (ImmutableMongoRestoreProcessArguments) another);
  }

  private boolean equalTo(int synthetic, ImmutableMongoRestoreProcessArguments another) {
    return destination.equals(another.destination)
        && serverAddress.equals(another.serverAddress)
        && arguments.equals(another.arguments);
  }

  /**
   * Computes a hash code from attributes: {@code destination}, {@code serverAddress}, {@code arguments}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + destination.hashCode();
    h += (h << 5) + serverAddress.hashCode();
    h += (h << 5) + arguments.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code MongoRestoreProcessArguments} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "MongoRestoreProcessArguments{"
        + "destination=" + destination
        + ", serverAddress=" + serverAddress
        + ", arguments=" + arguments
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link MongoRestoreProcessArguments} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable MongoRestoreProcessArguments instance
   */
  public static ImmutableMongoRestoreProcessArguments copyOf(MongoRestoreProcessArguments instance) {
    if (instance instanceof ImmutableMongoRestoreProcessArguments) {
      return (ImmutableMongoRestoreProcessArguments) instance;
    }
    return ImmutableMongoRestoreProcessArguments.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableMongoRestoreProcessArguments ImmutableMongoRestoreProcessArguments}.
   * <pre>
   * ImmutableMongoRestoreProcessArguments.builder()
   *    .destination(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.types.ProcessArguments&amp;gt;) // optional {@link MongoRestoreProcessArguments#destination() destination}
   *    .serverAddress(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.mongo.commands.ServerAddress&amp;gt;) // optional {@link MongoRestoreProcessArguments#serverAddress() serverAddress}
   *    .arguments(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.mongo.commands.MongoRestoreArguments&amp;gt;) // optional {@link MongoRestoreProcessArguments#arguments() arguments}
   *    .build();
   * </pre>
   * @return A new ImmutableMongoRestoreProcessArguments builder
   */
  public static ImmutableMongoRestoreProcessArguments.Builder builder() {
    return new ImmutableMongoRestoreProcessArguments.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableMongoRestoreProcessArguments ImmutableMongoRestoreProcessArguments}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private StateID<ProcessArguments> destination;
    private StateID<ServerAddress> serverAddress;
    private StateID<MongoRestoreArguments> arguments;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code MongoRestoreProcessArguments} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(MongoRestoreProcessArguments instance) {
      Objects.requireNonNull(instance, "instance");
      destination(instance.destination());
      serverAddress(instance.serverAddress());
      arguments(instance.arguments());
      return this;
    }

    /**
     * Initializes the value for the {@link MongoRestoreProcessArguments#destination() destination} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link MongoRestoreProcessArguments#destination() destination}.</em>
     * @param destination The value for destination 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder destination(StateID<ProcessArguments> destination) {
      this.destination = Objects.requireNonNull(destination, "destination");
      return this;
    }

    /**
     * Initializes the value for the {@link MongoRestoreProcessArguments#serverAddress() serverAddress} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link MongoRestoreProcessArguments#serverAddress() serverAddress}.</em>
     * @param serverAddress The value for serverAddress 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder serverAddress(StateID<ServerAddress> serverAddress) {
      this.serverAddress = Objects.requireNonNull(serverAddress, "serverAddress");
      return this;
    }

    /**
     * Initializes the value for the {@link MongoRestoreProcessArguments#arguments() arguments} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link MongoRestoreProcessArguments#arguments() arguments}.</em>
     * @param arguments The value for arguments 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder arguments(StateID<MongoRestoreArguments> arguments) {
      this.arguments = Objects.requireNonNull(arguments, "arguments");
      return this;
    }

    /**
     * Builds a new {@link ImmutableMongoRestoreProcessArguments ImmutableMongoRestoreProcessArguments}.
     * @return An immutable instance of MongoRestoreProcessArguments
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableMongoRestoreProcessArguments build() {
      return new ImmutableMongoRestoreProcessArguments(this);
    }
  }
}
