package de.flapdoodle.embed.mongo.commands;

import java.util.Objects;
import java.util.Optional;

/**
 * Immutable implementation of {@link MongosArguments}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableMongosArguments.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableMongosArguments extends MongosArguments {
  private final boolean verbose;
  private final String configDB;
  private final String replicaSet;

  private ImmutableMongosArguments(ImmutableMongosArguments.Builder builder) {
    this.configDB = builder.configDB;
    this.replicaSet = builder.replicaSet;
    this.verbose = builder.verboseIsSet()
        ? builder.verbose
        : super.verbose();
  }

  private ImmutableMongosArguments(boolean verbose, String configDB, String replicaSet) {
    this.verbose = verbose;
    this.configDB = configDB;
    this.replicaSet = replicaSet;
  }

  /**
   * @return The value of the {@code verbose} attribute
   */
  @Override
  public boolean verbose() {
    return verbose;
  }

  /**
   * @return The value of the {@code configDB} attribute
   */
  @Override
  public Optional<String> configDB() {
    return Optional.ofNullable(configDB);
  }

  /**
   * @return The value of the {@code replicaSet} attribute
   */
  @Override
  public Optional<String> replicaSet() {
    return Optional.ofNullable(replicaSet);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongosArguments#verbose() verbose} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for verbose
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableMongosArguments withVerbose(boolean value) {
    if (this.verbose == value) return this;
    return new ImmutableMongosArguments(value, this.configDB, this.replicaSet);
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link MongosArguments#configDB() configDB} attribute.
   * @param value The value for configDB
   * @return A modified copy of {@code this} object
   */
  public final ImmutableMongosArguments withConfigDB(String value) {
    String newValue = Objects.requireNonNull(value, "configDB");
    if (Objects.equals(this.configDB, newValue)) return this;
    return new ImmutableMongosArguments(this.verbose, newValue, this.replicaSet);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link MongosArguments#configDB() configDB} attribute.
   * An equality check is used on inner nullable value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for configDB
   * @return A modified copy of {@code this} object
   */
  public final ImmutableMongosArguments withConfigDB(Optional<String> optional) {
    String value = optional.orElse(null);
    if (Objects.equals(this.configDB, value)) return this;
    return new ImmutableMongosArguments(this.verbose, value, this.replicaSet);
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link MongosArguments#replicaSet() replicaSet} attribute.
   * @param value The value for replicaSet
   * @return A modified copy of {@code this} object
   */
  public final ImmutableMongosArguments withReplicaSet(String value) {
    String newValue = Objects.requireNonNull(value, "replicaSet");
    if (Objects.equals(this.replicaSet, newValue)) return this;
    return new ImmutableMongosArguments(this.verbose, this.configDB, newValue);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link MongosArguments#replicaSet() replicaSet} attribute.
   * An equality check is used on inner nullable value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for replicaSet
   * @return A modified copy of {@code this} object
   */
  public final ImmutableMongosArguments withReplicaSet(Optional<String> optional) {
    String value = optional.orElse(null);
    if (Objects.equals(this.replicaSet, value)) return this;
    return new ImmutableMongosArguments(this.verbose, this.configDB, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableMongosArguments} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableMongosArguments
        && equalTo(0, (ImmutableMongosArguments) another);
  }

  private boolean equalTo(int synthetic, ImmutableMongosArguments another) {
    return verbose == another.verbose
        && Objects.equals(configDB, another.configDB)
        && Objects.equals(replicaSet, another.replicaSet);
  }

  /**
   * Computes a hash code from attributes: {@code verbose}, {@code configDB}, {@code replicaSet}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Boolean.hashCode(verbose);
    h += (h << 5) + Objects.hashCode(configDB);
    h += (h << 5) + Objects.hashCode(replicaSet);
    return h;
  }

  /**
   * Prints the immutable value {@code MongosArguments} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder("MongosArguments{");
    builder.append("verbose=").append(verbose);
    if (configDB != null) {
      builder.append(", ");
      builder.append("configDB=").append(configDB);
    }
    if (replicaSet != null) {
      builder.append(", ");
      builder.append("replicaSet=").append(replicaSet);
    }
    return builder.append("}").toString();
  }

  /**
   * Creates an immutable copy of a {@link MongosArguments} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable MongosArguments instance
   */
  public static ImmutableMongosArguments copyOf(MongosArguments instance) {
    if (instance instanceof ImmutableMongosArguments) {
      return (ImmutableMongosArguments) instance;
    }
    return ImmutableMongosArguments.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableMongosArguments ImmutableMongosArguments}.
   * <pre>
   * ImmutableMongosArguments.builder()
   *    .verbose(boolean) // optional {@link MongosArguments#verbose() verbose}
   *    .configDB(String) // optional {@link MongosArguments#configDB() configDB}
   *    .replicaSet(String) // optional {@link MongosArguments#replicaSet() replicaSet}
   *    .build();
   * </pre>
   * @return A new ImmutableMongosArguments builder
   */
  public static ImmutableMongosArguments.Builder builder() {
    return new ImmutableMongosArguments.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableMongosArguments ImmutableMongosArguments}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long OPT_BIT_VERBOSE = 0x1L;
    private long optBits;

    private boolean verbose;
    private String configDB;
    private String replicaSet;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code MongosArguments} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(MongosArguments instance) {
      Objects.requireNonNull(instance, "instance");
      verbose(instance.verbose());
      Optional<String> configDBOptional = instance.configDB();
      if (configDBOptional.isPresent()) {
        configDB(configDBOptional);
      }
      Optional<String> replicaSetOptional = instance.replicaSet();
      if (replicaSetOptional.isPresent()) {
        replicaSet(replicaSetOptional);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link MongosArguments#verbose() verbose} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link MongosArguments#verbose() verbose}.</em>
     * @param verbose The value for verbose 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder verbose(boolean verbose) {
      this.verbose = verbose;
      optBits |= OPT_BIT_VERBOSE;
      return this;
    }

    /**
     * Initializes the optional value {@link MongosArguments#configDB() configDB} to configDB.
     * @param configDB The value for configDB
     * @return {@code this} builder for chained invocation
     */
    public final Builder configDB(String configDB) {
      this.configDB = Objects.requireNonNull(configDB, "configDB");
      return this;
    }

    /**
     * Initializes the optional value {@link MongosArguments#configDB() configDB} to configDB.
     * @param configDB The value for configDB
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder configDB(Optional<String> configDB) {
      this.configDB = configDB.orElse(null);
      return this;
    }

    /**
     * Initializes the optional value {@link MongosArguments#replicaSet() replicaSet} to replicaSet.
     * @param replicaSet The value for replicaSet
     * @return {@code this} builder for chained invocation
     */
    public final Builder replicaSet(String replicaSet) {
      this.replicaSet = Objects.requireNonNull(replicaSet, "replicaSet");
      return this;
    }

    /**
     * Initializes the optional value {@link MongosArguments#replicaSet() replicaSet} to replicaSet.
     * @param replicaSet The value for replicaSet
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder replicaSet(Optional<String> replicaSet) {
      this.replicaSet = replicaSet.orElse(null);
      return this;
    }

    /**
     * Builds a new {@link ImmutableMongosArguments ImmutableMongosArguments}.
     * @return An immutable instance of MongosArguments
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableMongosArguments build() {
      return new ImmutableMongosArguments(this);
    }

    private boolean verboseIsSet() {
      return (optBits & OPT_BIT_VERBOSE) != 0;
    }
  }
}
