/*
 * Decompiled with CFR 0.152.
 */
package de.gsi.dataset.utils;

import de.gsi.dataset.DataSet;
import de.gsi.dataset.DataSetError;
import de.gsi.dataset.DataSetMetaData;
import de.gsi.dataset.spi.AbstractDataSet;
import de.gsi.dataset.spi.DefaultDataSet;
import de.gsi.dataset.spi.DoubleDataSet;
import de.gsi.dataset.spi.DoubleErrorDataSet;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.PushbackInputStream;
import java.nio.ByteBuffer;
import java.nio.DoubleBuffer;
import java.nio.file.Path;
import java.security.InvalidParameterException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Map;
import java.util.TimeZone;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataSetUtils {
    private static final int SWITCH_TO_BINARY_KEY = 254;
    private static final Logger LOGGER = LoggerFactory.getLogger(DataSetUtils.class);
    private static final String DEFAULT_TIME_FORMAT = "yyyyMMdd_HHmmss";

    private DataSetUtils() {
    }

    public static AbstractDataSet<?> copyDataSet(DataSet ds) {
        if (ds instanceof DoubleDataSet) {
            DoubleDataSet dds = (DoubleDataSet)ds;
            DoubleDataSet d = new DoubleDataSet(dds.getName(), dds.getXValues(), dds.getYValues(), true);
            d.setStyle(dds.getStyle());
            d.getDataLabelMap().putAll(dds.getDataLabelMap());
            d.getDataStyleMap().putAll(dds.getDataStyleMap());
            return d;
        }
        if (ds instanceof DefaultDataSet) {
            DefaultDataSet dds = (DefaultDataSet)ds;
            DefaultDataSet d = new DefaultDataSet(dds.getName());
            d.setStyle(dds.getStyle());
            d.getData().clear();
            d.getData().addAll(dds.getData());
            d.getDataLabelMap().putAll(dds.getDataLabelMap());
            d.getDataStyleMap().putAll(dds.getDataStyleMap());
            return d;
        }
        double[] x = new double[ds.getDataCount()];
        double[] y = new double[ds.getDataCount()];
        for (int i = 0; i < ds.getDataCount(); ++i) {
            x[i] = ds.getX(i);
            y[i] = ds.getY(i);
        }
        DoubleDataSet d = new DoubleDataSet(ds.getName(), x, y);
        d.setStyle(ds.getStyle());
        for (int i = 0; i < ds.getDataCount(); ++i) {
            String label;
            String style = ds.getStyle(i);
            if (style != null) {
                d.addDataStyle(i, style);
            }
            if ((label = ds.getDataLabel(i)) == null) continue;
            d.addDataLabel(i, label);
        }
        return d;
    }

    public static double error(DataSet dataSet, ErrType eType, int index) {
        return DataSetUtils.error(dataSet, eType, index, 0.0, false);
    }

    public static double error(DataSet dataSet, ErrType eType, double x) {
        return DataSetUtils.error(dataSet, eType, -1, x, true);
    }

    protected static double error(DataSet dataSet, ErrType eType, int index, double x, boolean interpolate) {
        if (!(dataSet instanceof DataSetError)) {
            return 0.0;
        }
        DataSetError ds = (DataSetError)dataSet;
        if (interpolate) {
            switch (eType) {
                case EXN: {
                    return ds.getXErrorNegative(x);
                }
                case EXP: {
                    return ds.getXErrorPositive(x);
                }
                case EYN: {
                    return ds.getYErrorNegative(x);
                }
                case EYP: {
                    return ds.getYErrorPositive(x);
                }
            }
        } else {
            switch (eType) {
                case EXN: {
                    return ds.getXErrorNegative(index);
                }
                case EXP: {
                    return ds.getXErrorPositive(index);
                }
                case EYN: {
                    return ds.getYErrorNegative(index);
                }
                case EYP: {
                    return ds.getYErrorPositive(index);
                }
            }
        }
        return 0.0;
    }

    protected static double[] cropToLength(double[] in, int length) {
        if (in.length == length) {
            return in;
        }
        return Arrays.copyOf(in, length);
    }

    public static double[] errors(DataSet dataSet, ErrType eType) {
        int nDim = dataSet.getDataCount();
        if (!(dataSet instanceof DataSetError)) {
            return new double[nDim];
        }
        DataSetError ds = (DataSetError)dataSet;
        switch (eType) {
            case EXN: {
                return DataSetUtils.cropToLength(ds.getXErrorsNegative(), nDim);
            }
            case EXP: {
                return DataSetUtils.cropToLength(ds.getXErrorsPositive(), nDim);
            }
            case EYN: {
                return DataSetUtils.cropToLength(ds.getYErrorsNegative(), nDim);
            }
        }
        return DataSetUtils.cropToLength(ds.getYErrorsPositive(), nDim);
    }

    public static String getISODate(long time_ms, String format) {
        long time = TimeUnit.MILLISECONDS.toMillis(time_ms);
        TimeZone tz = TimeZone.getTimeZone("UTC");
        SimpleDateFormat df = new SimpleDateFormat(format);
        df.setTimeZone(tz);
        return df.format(new Date(time));
    }

    private static OutputStream openDatasetFileOutput(File file, Compression compression) throws IOException {
        switch (compression) {
            case NONE: {
                return new FileOutputStream(file);
            }
            case GZIP: {
                return new GZIPOutputStream(new FileOutputStream(file));
            }
            case ZIP: {
                ZipOutputStream zipOStream = new ZipOutputStream(new FileOutputStream(file));
                String filename = file.getName();
                String zipentryname = filename.toLowerCase().endsWith(".zip") ? filename.substring(0, filename.length() - 4) : filename;
                zipOStream.putNextEntry(new ZipEntry(zipentryname));
                return zipOStream;
            }
        }
        throw new IllegalArgumentException("Unknown Compression format: " + compression.toString());
    }

    private static SplitCharByteInputStream openDatasetFileInput(File file, Compression compression) throws IOException {
        InputStream istream;
        switch (compression) {
            case ZIP: {
                ZipInputStream zipIStream = new ZipInputStream(new FileInputStream(file));
                if (zipIStream.getNextEntry() == null) {
                    throw new ZipException("Corrupt zip archive has no entries");
                }
                istream = zipIStream;
                break;
            }
            case GZIP: {
                istream = new GZIPInputStream(new FileInputStream(file));
                break;
            }
            case NONE: {
                istream = new FileInputStream(file);
                break;
            }
            default: {
                throw new IOException("Unimplemented Compression");
            }
        }
        return new SplitCharByteInputStream(new PushbackInputStream(istream, 8192));
    }

    public static String getFileName(DataSet dataSet, String fileName) {
        Pattern placeholder = Pattern.compile("\\{([^\\{\\}]*)\\}");
        Matcher matcher = placeholder.matcher(fileName);
        StringBuffer result = new StringBuffer();
        while (matcher.find()) {
            String value;
            String[] substitutionparams = matcher.group(1).split(";");
            if (substitutionparams.length == 0) {
                throw new IllegalArgumentException("fileName contains empty placeholder: " + matcher.group());
            }
            switch (substitutionparams[0]) {
                case "systemTime": {
                    value = Long.toString(System.currentTimeMillis());
                    break;
                }
                case "dataSetName": {
                    value = dataSet.getName();
                    break;
                }
                case "xMin": {
                    value = Double.toString(dataSet.getXMin());
                    break;
                }
                case "xMax": {
                    value = Double.toString(dataSet.getXMax());
                    break;
                }
                case "yMin": {
                    value = Double.toString(dataSet.getYMin());
                    break;
                }
                case "yMax": {
                    value = Double.toString(dataSet.getYMax());
                    break;
                }
                default: {
                    if (!(dataSet instanceof DataSetMetaData)) {
                        throw new IllegalArgumentException("fileName placeholder references meta data but dataSet is not instanceof DataSetMetaData");
                    }
                    DataSetMetaData metaDataSet = (DataSetMetaData)((Object)dataSet);
                    value = metaDataSet.getMetaInfo().get(substitutionparams[0]);
                    if (value != null) break;
                    throw new IllegalArgumentException("fileName placeholder references nonexisting metaData field: " + substitutionparams[0]);
                }
            }
            if (substitutionparams.length != 1 && !substitutionparams[1].equals("string")) {
                switch (substitutionparams[1]) {
                    case "date": {
                        String format = substitutionparams.length < 3 ? DEFAULT_TIME_FORMAT : substitutionparams[2];
                        value = DataSetUtils.getISODate(Long.valueOf(value), format);
                        break;
                    }
                    case "int": 
                    case "long": {
                        String format = substitutionparams.length < 3 ? "%d" : substitutionparams[2];
                        value = String.format(format, Long.valueOf(value));
                        break;
                    }
                    case "float": 
                    case "double": {
                        String format = substitutionparams.length < 3 ? "%e" : substitutionparams[2];
                        value = String.format(format, Double.valueOf(value));
                        break;
                    }
                    default: {
                        throw new IllegalArgumentException("fileName contains placeholder with illegal type: " + substitutionparams[1]);
                    }
                }
            }
            matcher.appendReplacement(result, value);
        }
        matcher.appendTail(result);
        return result.toString();
    }

    private static Compression evaluateAutoCompression(String fileName) {
        if (fileName.toLowerCase().endsWith(".gz")) {
            return Compression.GZIP;
        }
        if (fileName.toLowerCase().endsWith(".zip")) {
            return Compression.ZIP;
        }
        return Compression.NONE;
    }

    public static String writeDataSetToFile(DataSet dataSet, Path path, String fileName, boolean binary) {
        return DataSetUtils.writeDataSetToFile(dataSet, path, fileName, Compression.AUTO, binary);
    }

    public static String writeDataSetToFile(DataSet dataSet, Path path, String fileName) {
        return DataSetUtils.writeDataSetToFile(dataSet, path, fileName, Compression.AUTO, false);
    }

    public static String writeDataSetToFile(DataSet dataSet, Path path, String fileName, Compression compression) {
        return DataSetUtils.writeDataSetToFile(dataSet, path, fileName, compression, false);
    }

    public static String writeDataSetToFile(DataSet dataSet, Path path, String fileName, Compression compression, boolean binary) {
        if (compression == Compression.AUTO) {
            compression = DataSetUtils.evaluateAutoCompression(fileName);
        }
        if (dataSet == null) {
            throw new IllegalArgumentException("dataSet must not be null or empty");
        }
        if (fileName == null || fileName.isEmpty()) {
            throw new IllegalArgumentException("fileName must not be null or empty");
        }
        try {
            String realFileName = DataSetUtils.getFileName(dataSet, fileName);
            String longFileName = path.toFile() + "/" + realFileName;
            File file = new File(longFileName);
            if (file.getParentFile().mkdirs()) {
                LOGGER.info("needed to create directory for file: " + longFileName);
            }
            ByteArrayOutputStream byteOutput = new ByteArrayOutputStream(8192);
            try (OutputStream outputfile = DataSetUtils.openDatasetFileOutput(file, compression);){
                DataSetUtils.writeDataSetToByteArray(dataSet, byteOutput, binary);
                byteOutput.writeTo(outputfile);
            }
            catch (IOException e) {
                LOGGER.error("could not write to file: '" + fileName + "'", (Throwable)e);
            }
            LOGGER.debug("write data set '" + dataSet.getName() + "' to " + longFileName);
            return longFileName;
        }
        catch (Exception e) {
            LOGGER.error("could not write to file: '" + fileName + "'", (Throwable)e);
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void writeDataSetToByteArray(DataSet dataSet, ByteArrayOutputStream byteOutput, boolean binary) {
        if (dataSet == null) {
            throw new IllegalArgumentException("dataSet must not be null or empty");
        }
        if (byteOutput == null) {
            throw new IllegalArgumentException("byteOutput must not be null or empty");
        }
        byteOutput.reset();
        try {
            dataSet.lock();
            byteOutput.write(("#file producer : " + DataSetUtils.class.getCanonicalName() + "\n").getBytes());
            DataSetUtils.writeHeaderDataToStream(byteOutput, dataSet);
            DataSetUtils.writeMetaDataToStream(byteOutput, dataSet);
            if (binary) {
                DataSetUtils.writeNumericBinaryDataToStream(byteOutput, dataSet);
            } else {
                DataSetUtils.writeNumericDataToStream(byteOutput, dataSet);
            }
        }
        catch (IOException e) {
            LOGGER.error("could not write to ByteArrayOutputStream", (Throwable)e);
            byteOutput.reset();
        }
        finally {
            dataSet.unlock();
        }
    }

    private static void writeNumericBinaryDataToStream(OutputStream outputStream, DataSet dataSet) {
        int nsamples = dataSet.getDataCount();
        StringBuffer buffer = new StringBuffer();
        buffer.append("$binary").append("\n");
        buffer.append("$x;float64[];").append(Long.toString(nsamples)).append("\n");
        buffer.append("$y;float64[];").append(Long.toString(nsamples)).append("\n");
        buffer.append("$eyn;float64[];").append(Long.toString(nsamples)).append("\n");
        buffer.append("$eyp;float64[];").append(Long.toString(nsamples)).append("\n");
        try {
            outputStream.write(buffer.toString().getBytes());
        }
        catch (IOException e) {
            LOGGER.error("WriteNumericDataToBinaryFile failed to write header description: ", (Throwable)e);
        }
        ByteBuffer xByte = ByteBuffer.allocate(8 * nsamples);
        DoubleBuffer xDouble = xByte.asDoubleBuffer();
        xDouble.put(dataSet.getXValues());
        ByteBuffer yByte = ByteBuffer.allocate(8 * nsamples);
        DoubleBuffer yDouble = yByte.asDoubleBuffer();
        yDouble.put(dataSet.getYValues());
        ByteBuffer eynByte = ByteBuffer.allocate(8 * nsamples);
        DoubleBuffer eynDouble = eynByte.asDoubleBuffer();
        eynDouble.put(DataSetUtils.errors(dataSet, ErrType.EYN));
        ByteBuffer eypByte = ByteBuffer.allocate(8 * nsamples);
        DoubleBuffer eypDouble = eypByte.asDoubleBuffer();
        eypDouble.put(DataSetUtils.errors(dataSet, ErrType.EYP));
        try {
            outputStream.write(254);
            outputStream.write(xByte.array());
            outputStream.write(yByte.array());
            outputStream.write(eynByte.array());
            outputStream.write(eypByte.array());
        }
        catch (IOException e) {
            LOGGER.error("WriteNumericDataToBinaryFile failed to write binary body: ", (Throwable)e);
        }
    }

    protected static void writeHeaderDataToStream(OutputStream outputStream, DataSet dataSet) {
        try {
            StringBuffer buffer = new StringBuffer();
            buffer.append("#dataSetName : ").append(dataSet.getName()).append('\n');
            buffer.append("#xMin : ").append(dataSet.getXMin()).append('\n');
            buffer.append("#xMax : ").append(dataSet.getXMax()).append('\n');
            buffer.append("#yMin : ").append(dataSet.getYMin()).append('\n');
            buffer.append("#yMax : ").append(dataSet.getYMax()).append('\n');
            try {
                buffer.append("#integral : ").append(DataSetUtils.integralSimple(dataSet)).append('\n');
                buffer.append("#mean : ").append(DataSetUtils.mean(dataSet.getYValues())).append('\n');
                buffer.append("#rms : ").append(DataSetUtils.rootMeanSquare(dataSet.getYValues())).append('\n');
            }
            catch (Exception e) {
                LOGGER.error("writeHeaderDataToFile - compute Math error for dataSet = '" + dataSet.getName() + "'", (Throwable)e);
            }
            outputStream.write(buffer.toString().getBytes());
        }
        catch (Exception e) {
            LOGGER.error("writeHeaderDataToFile - error for dataSet = '" + dataSet.getName() + "'", (Throwable)e);
        }
    }

    protected static void writeMetaDataToStream(OutputStream outputStream, DataSet dataSet) {
        if (!(dataSet instanceof DataSetMetaData)) {
            return;
        }
        DataSetMetaData metaDataSet = (DataSetMetaData)((Object)dataSet);
        try {
            StringBuffer buffer = new StringBuffer();
            for (String info : metaDataSet.getInfoList()) {
                buffer.append("#info : ").append(info).append('\n');
            }
            for (String warning : metaDataSet.getWarningList()) {
                buffer.append("#warning : ").append(warning).append('\n');
            }
            for (String error : metaDataSet.getErrorList()) {
                buffer.append("#error : ").append(error).append('\n');
            }
            Map<String, String> map = metaDataSet.getMetaInfo();
            for (String key : metaDataSet.getMetaInfo().keySet()) {
                buffer.append("#metaKey -").append(key).append(" : ").append(map.get(key)).append('\n');
            }
            outputStream.write(buffer.toString().getBytes());
        }
        catch (Exception e) {
            LOGGER.error("writeMetaDataToFile - error for dataSet = '" + dataSet.getName() + "'", (Throwable)e);
        }
    }

    protected static void writeNumericDataToStream(OutputStream outputFile, DataSet dataSet) {
        try {
            int nSamples = dataSet.getDataCount();
            StringBuffer buffer = new StringBuffer();
            buffer.append("#nSamples : " + nSamples).append("\n");
            buffer.append("$index, x, y, eyn, eyp");
            buffer.append("\n");
            for (int i = 0; i < nSamples; ++i) {
                buffer.append(i);
                buffer.append(',');
                buffer.append(dataSet.getX(i));
                buffer.append(',');
                buffer.append(dataSet.getY(i));
                buffer.append(',');
                buffer.append(DataSetUtils.error(dataSet, ErrType.EYN, i));
                buffer.append(',');
                buffer.append(DataSetUtils.error(dataSet, ErrType.EYP, i));
                buffer.append('\n');
            }
            outputFile.write(buffer.toString().getBytes());
        }
        catch (Exception e) {
            LOGGER.error("writeNumericDataToFile - error for dataSet = '" + dataSet.getName() + "'", (Throwable)e);
        }
    }

    protected static String getKey(String line, String replace) {
        if (line == null) {
            return null;
        }
        String[] split = line.split(" : ");
        if (split.length < 2) {
            return null;
        }
        return split[0].replaceAll(replace, "");
    }

    protected static String getValue(String line) {
        if (line == null) {
            return null;
        }
        String[] split = line.split(" : ");
        if (split.length < 2) {
            return null;
        }
        return split[1];
    }

    public static DataSet readDataSetFromFile(String fileName) {
        return DataSetUtils.readDataSetFromFile(fileName, Compression.AUTO);
    }

    public static DataSet readDataSetFromFile(String fileName, Compression compression) {
        if (fileName == null || fileName.isEmpty()) {
            throw new IllegalArgumentException("fileName must not be null or empty");
        }
        if (compression == Compression.AUTO) {
            compression = DataSetUtils.evaluateAutoCompression(fileName);
        }
        DoubleErrorDataSet dataSet = null;
        try {
            File file = new File(fileName);
            try (SplitCharByteInputStream inputFile = DataSetUtils.openDatasetFileInput(file, compression);){
                dataSet = DataSetUtils.readDataSetFromStream(inputFile);
            }
            catch (IOException e) {
                LOGGER.error("could not open/parse file: '" + fileName + "'", (Throwable)e);
            }
        }
        catch (Exception e) {
            LOGGER.error("could not open/parse file: '" + fileName + "'", (Throwable)e);
            return dataSet;
        }
        return dataSet;
    }

    public static DoubleErrorDataSet readDataSetFromByteArray(byte[] byteArray) {
        if (byteArray == null || byteArray.length == 0) {
            throw new InvalidParameterException("null byteArray");
        }
        if (byteArray.length == 0) {
            throw new InvalidParameterException("byteArray with zero length");
        }
        DoubleErrorDataSet dataSet = null;
        try {
            ByteArrayInputStream istream = new ByteArrayInputStream(byteArray);
            try (SplitCharByteInputStream inputFile = new SplitCharByteInputStream(new PushbackInputStream(istream, 8192));){
                dataSet = DataSetUtils.readDataSetFromStream(inputFile);
            }
            catch (IOException e) {
                LOGGER.error("could not open/parse byte array size = " + byteArray.length, (Throwable)e);
            }
        }
        catch (Exception e) {
            LOGGER.error("could not open/parse byte array size = " + byteArray.length, (Throwable)e);
            return dataSet;
        }
        return dataSet;
    }

    public static DoubleErrorDataSet readDataSetFromStream(SplitCharByteInputStream inputStream) {
        boolean binary = false;
        DoubleErrorDataSet dataSet = null;
        try (BufferedReader inputReader = new BufferedReader(new InputStreamReader(inputStream));){
            String dataSetName = "unknown data set";
            int nDataCountEstimate = 0;
            ArrayList<String> info = new ArrayList<String>();
            ArrayList<String> warning = new ArrayList<String>();
            ArrayList<String> error = new ArrayList<String>();
            ConcurrentHashMap<String, String> metaInfoMap = new ConcurrentHashMap<String, String>();
            String line = inputReader.readLine();
            while ((line = inputReader.readLine()) != null) {
                if (line.contains("$")) {
                    if (!line.startsWith("$binary")) break;
                    binary = true;
                    break;
                }
                if (line.contains("#dataSetName")) {
                    dataSetName = DataSetUtils.getValue(line);
                }
                if (line.contains("#nSamples")) {
                    nDataCountEstimate = Integer.parseInt(DataSetUtils.getValue(line));
                }
                if (line.contains("#info")) {
                    info.add(DataSetUtils.getValue(line));
                }
                if (line.contains("#warning")) {
                    warning.add(DataSetUtils.getValue(line));
                }
                if (line.contains("#error")) {
                    error.add(DataSetUtils.getValue(line));
                }
                if (!line.contains("#metaKey -")) continue;
                String key = DataSetUtils.getKey(line, "#metaKey -");
                String value = DataSetUtils.getValue(line);
                metaInfoMap.put(key, value);
            }
            dataSet = new DoubleErrorDataSet(dataSetName, nDataCountEstimate);
            dataSet.getMetaInfo().putAll(metaInfoMap);
            dataSet.getInfoList();
            if (binary) {
                DataSetUtils.readNumericDataFromBinaryFile(inputReader, inputStream, dataSet);
            } else {
                DataSetUtils.readNumericDataFromFile(inputReader, dataSet);
            }
        }
        catch (IOException e) {
            LOGGER.error("could not open/parse inputStream", (Throwable)e);
            return dataSet;
        }
        return dataSet;
    }

    private static void readNumericDataFromBinaryFile(BufferedReader inputReader, SplitCharByteInputStream inputFile, DoubleErrorDataSet dataSet) throws IOException {
        class DataEntry {
            public String name;
            public String type;
            public int nsamples;
            public DoubleBuffer data = null;

            DataEntry() {
            }
        }
        String line;
        ArrayList<1> toRead = new ArrayList<1>();
        while ((line = inputReader.readLine()) != null) {
            final String[] tokens = line.substring(1).split(";");
            toRead.add(new DataEntry(){
                {
                    this.name = tokens[0];
                    this.type = tokens[1];
                    this.nsamples = Integer.valueOf(tokens[2]);
                }
            });
        }
        if (!inputFile.reachedSplit()) {
            LOGGER.error("File seems to be corrupted, Split marker not found");
        } else {
            inputFile.switchToBinary();
            int[] valindex = new int[]{-1, -1, -1, -1};
            block13: for (int i = 0; i < toRead.size(); ++i) {
                DataEntry dataentry = (DataEntry)toRead.get(i);
                LOGGER.debug("Read data: " + dataentry.name);
                ByteBuffer byteData = ByteBuffer.allocate(dataentry.nsamples * 8);
                dataentry.data = byteData.asDoubleBuffer();
                for (int alreadyRead = 0; alreadyRead < dataentry.nsamples * 8; alreadyRead += inputFile.read(byteData.array(), alreadyRead, dataentry.nsamples * 8 - alreadyRead)) {
                }
                switch (dataentry.name) {
                    case "x": {
                        valindex[0] = i;
                        continue block13;
                    }
                    case "y": {
                        valindex[1] = i;
                        continue block13;
                    }
                    case "eyn": {
                        valindex[2] = i;
                        continue block13;
                    }
                    case "eyp": {
                        valindex[3] = i;
                        continue block13;
                    }
                    default: {
                        LOGGER.debug("Got unused variable " + dataentry.name + " of type " + dataentry.type);
                    }
                }
            }
            double[] x = new double[((DataEntry)toRead.get((int)valindex[0])).nsamples];
            ((DataEntry)toRead.get((int)valindex[0])).data.get(x);
            double[] y = new double[((DataEntry)toRead.get((int)valindex[1])).nsamples];
            ((DataEntry)toRead.get((int)valindex[1])).data.get(y);
            double[] eyn = new double[((DataEntry)toRead.get((int)valindex[2])).nsamples];
            ((DataEntry)toRead.get((int)valindex[2])).data.get(eyn);
            double[] eyp = new double[((DataEntry)toRead.get((int)valindex[3])).nsamples];
            ((DataEntry)toRead.get((int)valindex[3])).data.get(eyp);
            dataSet.set(x, y, eyn, eyp, false);
        }
    }

    protected static void readNumericDataFromFile(BufferedReader inputFile, DoubleErrorDataSet dataSet) {
        try {
            String line = inputFile.readLine();
            while (line != null) {
                String[] parse = line.split(",");
                if (parse.length != 0) {
                    long index = Long.parseLong(parse[0]);
                    double x = Double.parseDouble(parse[1]);
                    double y = Double.parseDouble(parse[2]);
                    double eyn = parse.length < 5 ? 0.0 : Double.parseDouble(parse[3]);
                    double eyp = parse.length < 5 ? 0.0 : Double.parseDouble(parse[4]);
                    dataSet.add(x, y, eyn, eyp);
                }
                line = inputFile.readLine();
            }
        }
        catch (Exception e) {
            LOGGER.error("readNumericDataFromFile could not parse numeric data for: '" + dataSet.getName() + "'", (Throwable)e);
        }
    }

    protected static double integralSimple(DataSet function) {
        double integral1 = 0.0;
        double integral2 = 0.0;
        if (function.getDataCount() <= 1) {
            return 0.0;
        }
        for (int i = 1; i < function.getDataCount(); ++i) {
            double step = function.getX(i) - function.getX(i - 1);
            double val1 = function.getY(i - 1);
            double val2 = function.getY(i);
            integral1 += step * val1;
            integral2 += step * val2;
        }
        return 0.5 * (integral1 + integral2);
    }

    protected static synchronized double mean(double[] data) {
        if (data.length <= 0) {
            return Double.NaN;
        }
        double norm = 1.0 / (double)data.length;
        double val = 0.0;
        for (int i = 0; i < data.length; ++i) {
            val += norm * data[i];
        }
        return val;
    }

    protected static synchronized double rootMeanSquare(double[] data) {
        if (data.length <= 0) {
            return Double.NaN;
        }
        double norm = 1.0 / (double)data.length;
        double val1 = 0.0;
        double val2 = 0.0;
        for (int i = 0; i < data.length; ++i) {
            val1 += data[i];
            val2 += data[i] * data[i];
        }
        return Math.sqrt(Math.abs((val2 *= norm) - (val1 *= norm) * val1));
    }

    private static class SplitCharByteInputStream
    extends FilterInputStream {
        private static final byte MARKER = -2;
        private boolean binary = false;
        private boolean hasMarker = false;
        private final PushbackInputStream pbin;

        public SplitCharByteInputStream(PushbackInputStream in) {
            super(in);
            this.pbin = in;
        }

        @Override
        public int read() throws IOException {
            int b = this.in.read();
            if (b == -2) {
                this.pbin.unread(b);
                b = -1;
            }
            return b;
        }

        @Override
        public int read(byte[] b) throws IOException {
            int nread;
            if (!this.hasMarker) {
                nread = this.in.read(b, 0, b.length);
                for (int i = 0; i < b.length; ++i) {
                    if (b[i] != -2) continue;
                    this.pbin.unread(b, i + 1, nread - i - 1);
                    this.hasMarker = true;
                    return i;
                }
            } else {
                if (!this.binary) {
                    return -1;
                }
                nread = this.in.read(b, 0, b.length);
            }
            return nread;
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            int nread;
            if (!this.hasMarker) {
                nread = this.in.read(b, off, len);
                for (int i = off; i < off + nread; ++i) {
                    if (b[i] != -2) continue;
                    this.pbin.unread(b, i + 1, off + nread - i - 1);
                    this.hasMarker = true;
                    return i;
                }
            } else {
                if (!this.binary) {
                    return -1;
                }
                nread = this.in.read(b, off, len);
            }
            return nread;
        }

        public boolean reachedSplit() {
            return this.hasMarker && !this.binary;
        }

        public void switchToBinary() {
            if (this.hasMarker) {
                if (!this.binary) {
                    this.binary = true;
                } else {
                    LOGGER.warn("Allready in binary mode");
                }
            } else {
                LOGGER.warn("Char/Byte split marker not reached yet");
            }
        }
    }

    public static enum Compression {
        AUTO,
        GZIP,
        ZIP,
        NONE;

    }

    public static enum ErrType {
        EXN,
        EXP,
        EYN,
        EYP;

    }
}

