package de.joshuaschnabel.config;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.util.HashSet;
import java.util.Set;

import de.joshuaschnabel.config.json.JsonConfig;
import de.joshuaschnabel.config.json.NestedJsonConfig;

/**
 * This class prepares passed parameters, local settings, and external settings.
 * 
 * @author Joshua Schnabel
 *
 */
public class Config
{
	private String			appname;
	private String			classPath;
	private JsonConfig	config;
	private Set<String>	flags	= new HashSet<>();

	/**
	 * Description: {@link Config}
	 *
	 * @param appname
	 * @param classPath
	 * @param args
	 * @throws IOException
	 * @throws URISyntaxException
	 */
	public Config(String appname, String classPath, String[] args) throws IOException, URISyntaxException
	{
		this.appname = appname;
		this.classPath = classPath;

		loadConfig(args);
	}

	private void loadConfig(String[] args) throws IOException, URISyntaxException
	{
		JsonConfig defaultConfig = loadLocalConfig(new JsonConfig());
		JsonConfig externalConfig = loadExternalConfig(defaultConfig);
		JsonConfig argumentConfig = loadRunntimeArgs(args, externalConfig);
		JsonConfig fixedConfig = loadFixedConfig(argumentConfig);

		this.config = fixedConfig;
	}

	private JsonConfig loadRunntimeArgs(String[] args, JsonConfig jsonConfig)
	{
		JsonConfig argConfig = new NestedJsonConfig(jsonConfig);

		for (String arg : args)
		{
			if (arg.startsWith("--"))
			{
				if (arg.contains("="))
				{
					String[] splits = arg.split("=");
					argConfig.setProperty(splits[0].substring(2, splits[0].length()), splits[1]);
				}
				else
				{
					addFlag(arg.substring(2, arg.length()));
				}
			}
			else if (arg.startsWith("-"))
			{
				char[] splits = arg.toCharArray();
				for (int i = 1; i < splits.length; i++)
				{
					char c = splits[i];
					addFlag(String.valueOf(c));
				}

			}
		}

		if (!argConfig.isEmpty())
		{
			return argConfig;
		}
		return jsonConfig;
	}

	private void addFlag(String substring)
	{
		this.flags.add(substring);
	}

	private JsonConfig loadExternalConfig(JsonConfig localConfig) throws URISyntaxException, IOException
	{
		String path = Config.class.getProtectionDomain().getCodeSource().getLocation().toURI().getPath();

		File externalNestedJsonConfigFile = new File(path + "/" + this.appname + ".json");

		JsonConfig exteralConfig = null;
		if (externalNestedJsonConfigFile.exists())
		{
			try (InputStream stream = new FileInputStream(externalNestedJsonConfigFile))
			{
				exteralConfig = new NestedJsonConfig(localConfig);
				exteralConfig.load(stream);
			}
			catch (@SuppressWarnings("unused") FileNotFoundException e)
			{
				return localConfig;
			}
			return exteralConfig;
		}
		return localConfig;
	}

	private JsonConfig loadLocalConfig(JsonConfig argumentNestedJsonConfig) throws IOException
	{
		return loadConfigFromClasspath(this.classPath + "/" + this.appname + ".json", argumentNestedJsonConfig);
	}

	private JsonConfig loadFixedConfig(JsonConfig argumentNestedJsonConfig) throws IOException
	{
		return loadConfigFromClasspath(this.classPath + "/" + this.appname + "_fixed.json", argumentNestedJsonConfig);
	}

	private static JsonConfig loadConfigFromClasspath(String path, JsonConfig argumentConfig) throws IOException
	{
		JsonConfig config = new NestedJsonConfig(argumentConfig);

		config.load(Config.class.getClass().getResourceAsStream(path));
		return config;
	}

	/**
	 * Get the value of a property
	 * 
	 * @param name
	 *           of the property
	 * @return the value of the property
	 */
	public String getValue(String name)
	{
		return this.config.getProperty(name);
	}

	/**
	 * Get the value of a flag
	 * 
	 * @param name
	 *           of the flag
	 * @return the value of the flag
	 */
	public boolean isFlagSet(String name)
	{
		return this.flags.contains(name);
	}
}
