/**
 * JsonConfig.java
 *
 * Copyright 2016 Joshua Schnabel (github@joshua-schnabel.de)
 * 
 * Created: 29.10.2016 10:35:01
 * Part of: jsConfig
 * 
 * For licence informations check the LICENCE file!
 */
package de.joshuaschnabel.config.json;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.Arrays;

import com.eclipsesource.json.Json;
import com.eclipsesource.json.JsonObject;
import com.eclipsesource.json.WriterConfig;

import de.joshuaschnabel.config.json.objects.ConfigEntry;
import de.joshuaschnabel.config.json.objects.ConfigFolder;
import de.joshuaschnabel.config.json.objects.ConfigObject;
import de.joshuaschnabel.config.json.serializer.JsonSerializer;

/**
 * This class offers the possibility to store settings in a json file.
 * 
 * @author Joshua Schnabel
 *
 */
public class JsonConfig
{
	private static final String	ROOT	= "root";
	private ConfigFolder				root;

	/**
	 * 
	 *
	 * @see JsonConfig
	 */
	public JsonConfig()
	{
		this.root = new ConfigFolder(ROOT);
	}

	/**
	 * Loads settings from stream
	 * 
	 * @param is
	 *           - InputStream
	 * @throws IOException
	 */
	public void load(InputStream is) throws IOException
	{
		load(new InputStreamReader(is, Charset.forName("UTF-8")));
	}

	/**
	 * Loads settings from reader
	 * 
	 * @param ir
	 *           - Reader
	 * @throws IOException
	 */
	public void load(Reader ir) throws IOException
	{
		this.root = JsonSerializer.deserialize(Json.parse(ir).asObject());
	}

	/**
	 * Save settings to stream
	 * 
	 * @param os
	 *           - OutputStream
	 * @throws IOException
	 */
	public void save(OutputStream os) throws IOException
	{
		save(new OutputStreamWriter(os, Charset.forName("UTF-8")));
	}

	/**
	 * Save settings to Writer
	 * 
	 * @param ow
	 *           - Writer
	 * @throws IOException
	 */
	public void save(Writer ow) throws IOException
	{
		JsonObject jsonValue = JsonSerializer.serialize(this.root);
		jsonValue.writeTo(ow, WriterConfig.MINIMAL);
		ow.close();
	}

	/**
	 * Stores a value to a name.
	 * 
	 * @param path
	 *           - path of the value
	 * @param value
	 *           the value
	 */
	public void setProperty(String path, String value)
	{
		String[] names = path.split("/");

		String entryName = names[names.length - 1];

		ConfigFolder folder = findFolder(names);

		if (folder == null) folder = createFolder(names);

		ConfigObject element = folder.getByName(entryName);

		if (element == null)
			folder.add(new ConfigEntry(entryName, value));
		else
			((ConfigEntry) element).setValue(value);
	}

	/**
	 * @param names
	 * @return
	 */
	private ConfigFolder findFolder(String[] names)
	{
		return findFolderRecursiv(this.root, Arrays.copyOfRange(names, 0, Math.max(names.length - 1, 0)));
	}

	private ConfigFolder findFolderRecursiv(ConfigFolder currentElement, String[] names)
	{
		if (names.length > 0)
		{
			String name = names[0];
			ConfigObject element = currentElement.getByName(name);
			if (element != null && element.is(ConfigFolder.class))
			{
				return findFolderRecursiv((ConfigFolder) element, Arrays.copyOfRange(names, 1, names.length));
			}
			return null;
		}
		return currentElement;
	}

	private ConfigFolder createFolder(String[] names)
	{
		return createFolderRecursiv(this.root, Arrays.copyOfRange(names, 0, Math.max(names.length - 1, 0)));
	}

	private ConfigFolder createFolderRecursiv(ConfigFolder currentElement, String[] names)
	{
		if (names.length > 0)
		{
			String name = names[0];

			// if (currentElement != null && currentElement.is(ConfigFolder.class))
			// {
			if (!currentElement.existsByName(name))
			{
				currentElement.add(new ConfigFolder(name));
			}

			ConfigObject element = currentElement.getByName(name);

			if (!element.is(ConfigFolder.class))
			{
				throw new IllegalStateException("Folder with Name " + name + " in " + getName(names) + " couldn't be obend! Element is no Folder!");
			}
			return createFolderRecursiv((ConfigFolder) element, Arrays.copyOfRange(names, 1, names.length));
			// }
		}
		return currentElement;
	}

	private static String getName(String[] names)
	{
		String name = "/";
		for (int j = 0; j < names.length; j++)
		{
			name += names[j] + "/";
		}
		return name;
	}

	/**
	 * Get a value
	 * 
	 * @param path
	 *           - to the value
	 * @return the value
	 */
	public String getProperty(String path)
	{
		String[] names = path.split("/");
		String entryName = names[names.length - 1];

		ConfigFolder folder = findFolder(names);
		if (folder != null)
		{
			ConfigObject element = folder.getByName(entryName);
			if (element != null && element.is(ConfigEntry.class))
			{
				return ((ConfigEntry) element).getValue();
			}
		}
		return null;
	}

	/**
	 * Get a value
	 * 
	 * @param path
	 *           - to the value
	 * @param defaultValue
	 *           - defaultValue if the path doesn't exists
	 * @return the value
	 */
	public String getProperty(String path, String defaultValue)
	{
		String value = getProperty(path);
		if (value == null) return defaultValue;
		return value;
	}

	/**
	 * 
	 * 
	 * @return {@link Boolean#TRUE} if there are no entries
	 */
	public boolean isEmpty()
	{
		return this.root.entriesSize() == 0;
	}

}
