package de.julielab.gene.candidateretrieval;

import de.julielab.geneexpbase.TermNormalizer;
import de.julielab.geneexpbase.candidateretrieval.SynHit;
import de.julielab.geneexpbase.genemodel.GeneMention;
import de.julielab.geneexpbase.genemodel.Specifier;
import org.jetbrains.annotations.NotNull;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static de.julielab.geneexpbase.genemodel.SpecifierType.*;

public class GeneRecordHit extends SynHit {
    private String symbol;
    private String nomenclature;
    private String chromosome;
    private String mapLocation;
    private double synonymSimilarityScore;
    private String synonymField;
    private String[] fullNames;
    private String[] synonyms;
    private String[] otherDesignations;
    private String[] uniprotNames;
    private String[] bioThesaurusNames;
    private String[] xrefs;
    private String[] geneRifs;
    private String[] interactions;
    private String[] descriptions;
    private String[] summaries;
    private String[] goDescriptors;
    private Set<String> nameTokens;
    private transient TermNormalizer termNormalizerLatinNumbers = new TermNormalizer();
    private String ecNumber;

    public GeneRecordHit(String syn, double score, String id, String source) {
        super(syn, score, id, source);
        setId(id);
    }

    /**
     * Returns a comparator that sorts SynHits first according to their equality to the gene name and then by score.This helps to overcome the issue that Lucene's float scores sometimes
     * fail to put an exact equal match to the top of the results.
     *
     * @param geneName            A gene name to compare to.
     * @param sortByMatchingField
     * @return A comparator that can be used to sort lists of SynHits.
     */
    public static Comparator<SynHit> getNormalizedExactMatchThenLuceneScoreComparator(final String geneName, boolean sortByMatchingField) {
//        Comparator<SynHit> comparator = Comparator.comparing(sh -> sh.getSynonym().equals(geneName));
        Comparator<SynHit> comparator = Comparator.comparing(sh -> sh.getSynonym().equals(geneName));
        comparator = comparator.thenComparingDouble(sh -> sh.getLuceneScore()).reversed();
        if (sortByMatchingField)
            comparator = comparator.thenComparingInt(sh -> {
                if (sh instanceof GeneRecordHit)
                    return -1 * GeneRecordQueryGenerator.ALL_FIELDS_LIST.indexOf(((GeneRecordHit) sh).getSynonymField());
                else return 0;
            });
        return comparator;
    }

    public Stream<String> getCommonTokens(GeneMention gm) {
        Set<String> nameTokenSet = gm.getNameTokenSet();

        Set<String> grhTokens = getNormalizedGeneNameTokenSet();

        return grhTokens.stream().filter(nameTokenSet::contains);
    }

    public String getChromosome() {
        return chromosome;
    }

    public void setChromosome(String chromosome) {
        this.chromosome = chromosome;
    }

    public String getMapLocation() {
        return mapLocation;
    }

    public void setMapLocation(String mapLocation) {
        this.mapLocation = mapLocation;
    }

    public String getSymbol() {
        return symbol;
    }

    public void setSymbol(String symbol) {
        this.symbol = symbol;
    }

    public String getSymbolFromNomenclature() {
        return nomenclature;
    }

    public void setNomenclature(String nomenclature) {
        this.nomenclature = nomenclature;
    }

    public String[] getFullNames() {
        return fullNames;
    }

    public void setFullNames(String[] fullNames) {
        this.fullNames = fullNames;
    }

    public String[] getSynonyms() {
        return synonyms;
    }

    public void setSynonyms(String[] synonyms) {
        this.synonyms = synonyms;
    }

    public String[] getOtherDesignations() {
        return otherDesignations;
    }

    public void setOtherDesignations(String[] otherDesignations) {
        this.otherDesignations = otherDesignations;
    }

    public String[] getUniprotNames() {
        return uniprotNames;
    }

    public void setUniprotNames(String[] uniprotNames) {
        this.uniprotNames = uniprotNames;
    }

    public String[] getBioThesaurusNames() {
        return bioThesaurusNames;
    }

    public void setBioThesaurusNames(String[] bioThesaurusNames) {
        this.bioThesaurusNames = bioThesaurusNames;
    }

    public String[] getXrefs() {
        return xrefs;
    }

    public void setXrefs(String[] xrefs) {
        this.xrefs = xrefs;
    }

    public double getSynonymSimilarityScore() {
        return synonymSimilarityScore;
    }

    public void setSynonymSimilarityScore(double synonymTokenJaroScore) {
        this.synonymSimilarityScore = synonymTokenJaroScore;
    }

    /**
     * @return The field of the synonym that matched the searched gene name best.
     */
    public String getSynonymField() {
        return synonymField;
    }

    /**
     * @param synonymField The field of the synonym that matched the searched gene name best.
     */
    public void setSynonymField(String synonymField) {
        this.synonymField = synonymField;
    }

    public String[] getGeneRifs() {
        return geneRifs;
    }

    public void setGeneRifs(String[] geneRifs) {
        this.geneRifs = geneRifs;
    }

    public String[] getInteractions() {
        return interactions;
    }

    public void setInteractions(String[] interactions) {
        this.interactions = interactions;
    }

    public String[] getDescriptions() {
        return descriptions;
    }

    public void setDescriptions(String[] descriptions) {
        this.descriptions = descriptions;
    }

    public String[] getSummaries() {
        return summaries;
    }

    public void setSummaries(String[] summaries) {
        this.summaries = summaries;
    }

    public String[] getGoDescriptors() {
        return goDescriptors;
    }

    public void setGoDescriptors(String[] goDescriptors) {
        this.goDescriptors = goDescriptors;
    }

    public Set<String> getNormalizedGeneNameTokenSet() {
        if (nameTokens == null) {
            Stream<String> grhTokens = getNormalizedGeneNameTokens();
            nameTokens = grhTokens.filter(Objects::nonNull).collect(Collectors.toSet());
        }
        return nameTokens;
    }

    @NotNull
    private Stream<String> getNormalizedGeneNameTokens() {
        if (termNormalizerLatinNumbers == null)
            termNormalizerLatinNumbers = new TermNormalizer();
        Function<String, Stream<String>> strTokensFunc = str -> str != null ? Arrays.stream(termNormalizerLatinNumbers.normalize(str).split("\\s+")) : Stream.empty();
        Stream<String> grhTokens = getGeneNameTokens(strTokensFunc);
        return grhTokens;
    }

    @NotNull
    private Stream<String> getGeneNameTokens(Function<String, Stream<String>> tokenizerFunc) {
        return getGeneNames().flatMap(tokenizerFunc);
    }

    public Stream<String> getGeneNames() {
        Stream<String> grhTokens = Stream.of(getSymbol());
        grhTokens = Stream.concat(grhTokens, Arrays.stream(getFullNames()));
        grhTokens = Stream.concat(grhTokens, Stream.of(getSymbolFromNomenclature()));
        grhTokens = Stream.concat(grhTokens, Arrays.stream(getSynonyms()));
        grhTokens = Stream.concat(grhTokens, Arrays.stream(getOtherDesignations()));
        grhTokens = Stream.concat(grhTokens, Arrays.stream(getUniprotNames()));
        grhTokens = Stream.concat(grhTokens, Arrays.stream(getBioThesaurusNames()));
        return grhTokens;
    }


    public Map<Specifier, Integer> getMajoritySpecifiers() {
        Map<Specifier, Integer> counts = new HashMap<>();
        Iterator<String> iterator = getGeneNames().iterator();
        while (iterator.hasNext()) {
            String name = iterator.next();
            termNormalizerLatinNumbers.getGreekCharacters(name).map(s -> new Specifier(GREEK, s)).forEach(s -> counts.merge(s, 1, Integer::sum));
            termNormalizerLatinNumbers.getRomanNumbers(name).map(s -> new Specifier(ROMAN, s)).forEach(s -> counts.merge(s, 1, Integer::sum));
            termNormalizerLatinNumbers.getNumbers(name).map(s -> new Specifier(NUMERICAL, s)).forEach(s -> counts.merge(s, 1, Integer::sum));
        }
        Specifier maxGreek = null;
        Specifier maxRoman = null;
        Specifier maxNum = null;
        int maxGreekCount = 0;
        int maxRomanCount = 0;
        int maxNumCount = 0;
        for (Specifier key : counts.keySet()) {
            int count = counts.get(key);
            switch (key.getSpecifierType()) {
                case NUMERICAL:
                    if (maxNum == null || maxNumCount < count)
                        maxNum = key;
                    maxNumCount = count;
                    break;
                case ROMAN:
                    if (maxRoman == null || maxRomanCount < count)
                        maxRoman = key;
                    maxRomanCount = count;
                    break;
                case GREEK:
                    if (maxGreek == null || maxGreekCount < count)
                        maxGreek = key;
                    maxGreekCount = count;
                    break;
            }
        }
        Map<Specifier, Integer> ret = new HashMap<>();
        if (maxGreek != null)
            ret.put(maxGreek, maxGreekCount);
        if (maxRoman != null)
            ret.put(maxRoman, maxRomanCount);
        if (maxNum != null)
            ret.put(maxNum, maxNumCount);
        return counts;
    }

    public void setEcNumber(String ecNumber) {
        this.ecNumber = ecNumber;
    }

    public String getEcNumber() {
        return ecNumber;
    }
}
