package de.julielab.gene.candidateretrieval;

import de.julielab.geneexpbase.CandidateFilter;
import de.julielab.geneexpbase.candidateretrieval.CandidateCacheKey;
import de.julielab.geneexpbase.candidateretrieval.QueryGenerator;
import de.julielab.geneexpbase.genemodel.GeneName;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause.Occur;
import org.apache.lucene.search.*;
import org.apache.lucene.search.BooleanQuery.Builder;
import org.jetbrains.annotations.NotNull;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static de.julielab.gene.candidateretrieval.Configuration.PARAM_DISMAX_TIE_BREAKER;
import static de.julielab.gene.candidateretrieval.SynonymIndexFieldNames.ID_FIELD;
import static de.julielab.gene.candidateretrieval.SynonymIndexFieldNames.TAX_ID_FIELD;
import static org.apache.lucene.search.BooleanClause.Occur.*;

public class GeneRecordQueryGenerator extends QueryGenerator {
    public static final String[] ALL_FIELDS = new String[]{SynonymIndexFieldNames.SYMBOL, SynonymIndexFieldNames.SYMBOL_FROM_NOMCENCLATURE, SynonymIndexFieldNames.SYNONYMS, SynonymIndexFieldNames.FULL_NAMES, SynonymIndexFieldNames.OTHER_DESIGNATIONS, SynonymIndexFieldNames.CHROMOSOME, SynonymIndexFieldNames.DESCRIPTION, SynonymIndexFieldNames.GENERIF, SynonymIndexFieldNames.GODESC, SynonymIndexFieldNames.INTERACTION, SynonymIndexFieldNames.MAPLOCATION, SynonymIndexFieldNames.SUMMARY, SynonymIndexFieldNames.PROTEIN_NAMES, SynonymIndexFieldNames.UNIPROT_NAMES, SynonymIndexFieldNames.BIO_THESAURUS};
    public static final String[] SYNONYM_FIELDS = new String[]{SynonymIndexFieldNames.SYMBOL, SynonymIndexFieldNames.SYMBOL_FROM_NOMCENCLATURE, SynonymIndexFieldNames.SYNONYMS/*, XREFS*/, SynonymIndexFieldNames.FULL_NAMES, SynonymIndexFieldNames.OTHER_DESIGNATIONS, SynonymIndexFieldNames.MAPLOCATION, SynonymIndexFieldNames.PROTEIN_NAMES, SynonymIndexFieldNames.UNIPROT_NAMES, SynonymIndexFieldNames.BIO_THESAURUS};
    public static final String[] SYNONYM_FIELDS_NOT_CHROMOSOMAL = new String[]{SynonymIndexFieldNames.SYMBOL, SynonymIndexFieldNames.SYMBOL_FROM_NOMCENCLATURE, SynonymIndexFieldNames.SYNONYMS, SynonymIndexFieldNames.FULL_NAMES, SynonymIndexFieldNames.OTHER_DESIGNATIONS, SynonymIndexFieldNames.PROTEIN_NAMES, SynonymIndexFieldNames.UNIPROT_NAMES, SynonymIndexFieldNames.BIO_THESAURUS};
    public static final List<String> ALL_FIELDS_LIST = Arrays.asList(ALL_FIELDS);
    // Whether to use dis max queries instead of disjunctions in a CNF. Only used when flatDisjunction is off.
    private final boolean useDisMax;
    // If true, create one large disjunction of all tokens on all fields (instead of disjunctions for each token on every field and those in conjunction, CNF). False by default.
    private final boolean flatDisjunction;
    // Whether to match the whole gene name on the fields containing the complete strings, non-tokenized. True by default
    private final boolean searchExactMatches;
    // Whether to generate a synonym-centric query via GeneRecordSynonymsQueryGenerator and add it in a SHOULD clause
    private final boolean useSynonymMatchesAsRelevanceSignal;
    private final boolean useNpContextAsRelevanceSignal;
    private final boolean useAlternativeNamesAsRelevanceSignal;
    private final boolean useContextGenesAsRelevanceSignal;
    private final boolean useAppositionsAsRelevanceSignal;
    private final int relaxation;
    private GeneRecordSynonymsQueryGenerator synonymsQueryGeneratorApprox;
    private GeneRecordSynonymsQueryGenerator synonymsQueryGeneratorExact;

    public GeneRecordQueryGenerator() {
        this(false);
    }

    public GeneRecordQueryGenerator(boolean useDisMax) {
        this(useDisMax, false);
    }

    public GeneRecordQueryGenerator(boolean useDisMax, boolean flatDisjunction) {
        this(useDisMax, flatDisjunction, true, false);
    }

    /**
     * @param useDisMax          Whether to use dis max queries instead of disjunctions in a CNF. Only used when flatDisjunction is false. Defaults to false.
     * @param flatDisjunction    If true, create one large disjunction of all tokens on all fields (instead of disjunctions for each token on every field and those in conjunction, CNF). False by default.
     * @param searchExactMatches Whether to match the whole gene name on the fields containing the complete strings, non-tokenized. True by default.
     */
    public GeneRecordQueryGenerator(boolean useDisMax, boolean flatDisjunction, boolean searchExactMatches, boolean useSynonymMatchesAsRelevanceSignal) {
        this(useDisMax, flatDisjunction, searchExactMatches, useSynonymMatchesAsRelevanceSignal, false);
    }

    public GeneRecordQueryGenerator(boolean useDisMax, boolean flatDisjunction, boolean searchExactMatches, boolean useSynonymMatchesAsRelevanceSignal, boolean useNpContextAsRelevanceSignal) {
        this(useDisMax, flatDisjunction, searchExactMatches, useSynonymMatchesAsRelevanceSignal, useNpContextAsRelevanceSignal, false, false);
    }

    public GeneRecordQueryGenerator(boolean useDisMax, boolean flatDisjunction, boolean searchExactMatches, boolean useSynonymMatchesAsRelevanceSignal, boolean useNpContextAsRelevanceSignal, boolean useAlternativeNamesAsRelevanceSignal, boolean useAppositionsAsRelevanceSignal) {
        this(useDisMax, flatDisjunction, searchExactMatches, useSynonymMatchesAsRelevanceSignal, useNpContextAsRelevanceSignal, useAlternativeNamesAsRelevanceSignal, useAppositionsAsRelevanceSignal, 0);
    }

    public GeneRecordQueryGenerator(boolean useDisMax, boolean flatDisjunction, boolean searchExactMatches, boolean useSynonymMatchesAsRelevanceSignal, boolean useNpContextAsRelevanceSignal, boolean useAlternativeNamesAsRelevanceSignal, boolean useAppositionsAsRelevanceSignal, int relaxation) {
        this(useDisMax, flatDisjunction, searchExactMatches, useSynonymMatchesAsRelevanceSignal, useNpContextAsRelevanceSignal, useAlternativeNamesAsRelevanceSignal, useAppositionsAsRelevanceSignal, false, relaxation);
    }

    public GeneRecordQueryGenerator(boolean useDisMax, boolean flatDisjunction, boolean searchExactMatches, boolean useSynonymMatchesAsRelevanceSignal, boolean useNpContextAsRelevanceSignal, boolean useAlternativeNamesAsRelevanceSignal, boolean useAppositionsAsRelevanceSignal, boolean useContextGenesAsRelevanceSignal) {
        this(useDisMax, flatDisjunction, searchExactMatches, useSynonymMatchesAsRelevanceSignal, useNpContextAsRelevanceSignal, useAlternativeNamesAsRelevanceSignal, useAppositionsAsRelevanceSignal, useContextGenesAsRelevanceSignal, 0);
    }

    public GeneRecordQueryGenerator(boolean useDisMax, boolean flatDisjunction, boolean searchExactMatches, boolean useSynonymMatchesAsRelevanceSignal, boolean useNpContextAsRelevanceSignal, boolean useAlternativeNamesAsRelevanceSignal, boolean useAppositionsAsRelevanceSignal, boolean useContextGenesAsRelevanceSignal, int relaxation) {
        this.useDisMax = useDisMax;
        this.flatDisjunction = flatDisjunction;
        this.searchExactMatches = searchExactMatches;
        this.useSynonymMatchesAsRelevanceSignal = useSynonymMatchesAsRelevanceSignal;
        this.useNpContextAsRelevanceSignal = useNpContextAsRelevanceSignal;
        this.useAlternativeNamesAsRelevanceSignal = useAlternativeNamesAsRelevanceSignal;
        this.useContextGenesAsRelevanceSignal = useContextGenesAsRelevanceSignal;
        this.useAppositionsAsRelevanceSignal = useAppositionsAsRelevanceSignal;
        this.relaxation = relaxation;
        if (this.useSynonymMatchesAsRelevanceSignal) {
            synonymsQueryGeneratorApprox = new GeneRecordSynonymsQueryGenerator(false);
            synonymsQueryGeneratorExact = new GeneRecordSynonymsQueryGenerator(true);
        }
    }

    public Query generateQuery(CandidateCacheKey key)
            throws BooleanQuery.TooManyClauses {
        Builder mainQb = new Builder();

        Map<String, Float> fieldWeights = key.getFieldWeights();
        boolean fieldweightsPresent = fieldWeights != null;
        float dismaxTieBreaker = fieldweightsPresent ? fieldWeights.get(PARAM_DISMAX_TIE_BREAKER) : 0.3f;

        GeneName geneName = key.getGeneName();
        Set<String> normalizedGeneNames = new HashSet<>();
        Builder namesBuilder = new Builder();
        List<GeneName> names = Stream.concat(Stream.of(geneName), geneName.getAlternatives().stream()).collect(Collectors.toList());
        for (GeneName name : names) {
            // make a query of the form "token1[All fields] AND token2[All fields] AND ..."
            String normalizedName = name.getNormalizedText();
            normalizedGeneNames.add(normalizedName);
            // We wrap the main query - the CNF, Dismax-CNF or flat disjunction - into a boolean query of its own.
            // This way we can set the minimumMustMatch parameter to gradually relax the query if necessary.
            Builder mainTokenDisjunction = new Builder();
            String[] tokens = normalizedName.split("\\s+");
            if (!flatDisjunction) {
                int effectiveRelaxation = relaxation != Integer.MAX_VALUE ? relaxation : tokens.length;
                int max = Math.max(1, tokens.length - effectiveRelaxation);
                mainTokenDisjunction.setMinimumNumberShouldMatch(max);
            }
            if (!flatDisjunction) {
                for (String token : tokens) {
                    Builder fieldsDisjunctionBuilder = new Builder();
                    String[] fieldsToSearch = CandidateFilter.isNumberGreekOrLatin(token) ? SYNONYM_FIELDS : ALL_FIELDS;
                    Stream<Query> termQueryStream = Arrays.stream(fieldsToSearch).map(field -> new Term(field, token)).map(TermQuery::new);
                    if (fieldWeights != null)
                        termQueryStream = termQueryStream.map(TermQuery.class::cast).map(tq -> {
                            float weight = fieldWeights.getOrDefault(tq.getTerm().field(), 1f);
                            return new BoostQuery(tq, weight);
                        });
                    if (!useDisMax) {
                        termQueryStream.forEach(tq -> fieldsDisjunctionBuilder.add(tq, SHOULD));
                        mainTokenDisjunction.add(fieldsDisjunctionBuilder.build(), SHOULD);
                    } else {
                        DisjunctionMaxQuery dismaxQuery = new DisjunctionMaxQuery(termQueryStream.collect(Collectors.toList()), dismaxTieBreaker);
                        mainTokenDisjunction.add(dismaxQuery, SHOULD);
                    }
                }
            } else {
                Builder flatDisjunctionBuilder = buildFlatDisjunction(fieldWeights, normalizedName);
                // The disjunction must match, so any token on any field
                mainTokenDisjunction.add(flatDisjunctionBuilder.build(), MUST);
            }
            namesBuilder.add(mainTokenDisjunction.build(), SHOULD);
        }
        mainQb.add(namesBuilder.build(), MUST);
        // A compulsory filter clause saying that at least one token must be found in a candidate's name to be a match.
        // This avoids candidates resulting purely from context fields (Generif, interactions etc)
//        Builder query4allNames = new Builder();
//        for (GeneName name : namesStream) {
//            String normalizedName = name.getNormalizedText();
//            normalizedGeneNames.add(normalizedName);
//            // This is basically "term1 OR term 2 OR term3 OR ..." with a minimumMustMatch of 1 so that at least
//            // one term must be a match on a synonym field
//            Builder query4allTokens = new Builder().setMinimumNumberShouldMatch(1);
//            String[] tokens = normalizedName.split("\\s+");
//            for (String token : tokens) {
//                if (CandidateFilter.isNumberGreekOrLatin(token))
//                    continue;
//
//                // This is a query for the current token on all the synonym fields.
//                Builder query4token = new Builder();
//                Stream<Query> termQueryStream = Arrays.stream(SYNONYM_FIELDS_NOT_CHROMOSOMAL).map(field -> new Term(field, token)).map(TermQuery::new);
//                termQueryStream.forEach(tq -> query4token.add(tq, SHOULD));
//                query4allTokens.add(query4token.build(), SHOULD);
//            }
//            query4allNames.add(query4allTokens.build(), SHOULD);
//        }
//        mainQb.add(query4allNames.build(), FILTER);
//        if (searchExactMatches) {
//        String normalizedGeneName = geneName.getNormalizedText();
//            // Add exact matches as additional relevance signals
//            Query exactSynonymQuery;
//            Stream<Query> exactSynonymQueries = Arrays.stream(SYNONYM_FIELDS).map(field -> new Term(field + "_exact", normalizedGeneName)).map(TermQuery::new);
//            if (fieldWeights != null)
//                exactSynonymQueries = exactSynonymQueries.map(TermQuery.class::cast).map(tq -> {
//                    float weight = fieldWeights.getOrDefault(tq.getTerm().field(), 1f);
//                    return new BoostQuery(tq, weight);
//                });
//            if (!useDisMax) {
//                Builder exactSynonymBuilder = new Builder();
//                exactSynonymQueries.forEach(q -> exactSynonymBuilder.add(q, SHOULD));
//                exactSynonymQuery = exactSynonymBuilder.build();
//            } else {
//                exactSynonymQuery = new DisjunctionMaxQuery(exactSynonymQueries.collect(Collectors.toList()), dismaxTieBreaker);
//            }
//            mainQb.add(exactSynonymQuery, SHOULD);
//        }
        if (useSynonymMatchesAsRelevanceSignal) {
            // Add another relevance signal by explicitly searching on the names
            Query q = synonymsQueryGeneratorApprox.generateQuery(key);
            mainQb.add(q, SHOULD);
            q = synonymsQueryGeneratorExact.generateQuery(key);
            mainQb.add(q, SHOULD);
        }
        for (GeneName appositionContext : geneName.getAppositionContexts()) {
            mainQb.add(buildFlatDisjunction(fieldWeights, appositionContext.getNormalizedText()).build(), SHOULD);
        }
//        if (useNpContextAsRelevanceSignal && geneName.getNominalPhraseContext() != null && !geneName.getNormalizedText().equals(geneName.getNominalPhraseContext())) {
//            // Add another relevance signal by searching for the nominal phrase tokens of the gene
//        if (!normalizedGeneNames.contains(geneName.getNominalPhraseContext().getNormalizedText())) {
//            BooleanQuery q = buildFlatDisjunction(fieldWeights, geneName.getNominalPhraseContext().getNormalizedText()).build();
//            mainQb.add(q, SHOULD);
//        }
//        }
//        if (useAlternativeNamesAsRelevanceSignal) {
//            for (GeneName alternative : geneName.getAlternatives()) {
//                mainQb.add(buildFlatDisjunction(fieldWeights, alternative.getNormalizedText()).build(), SHOULD);
//            }
//        }
        if (key.getContextNames() != null) {
            for (GeneName contextName : key.getContextNames()) {
                if (!normalizedGeneNames.contains(contextName.getNormalizedText()))
                    mainQb.add(buildFlatDisjunction(fieldWeights, contextName.getNormalizedText()).build(), SHOULD);
            }
        }
        if (key.getGeneIdsFilter() != null && !key.getGeneIdsFilter().isEmpty()) {
            Builder filterBuilder = new Builder();
            key.getGeneIdsFilter().stream().map(id -> new Term(ID_FIELD, id)).map(TermQuery::new).forEach(tq -> filterBuilder.add(tq, SHOULD));
            mainQb.add(filterBuilder.build(), Occur.FILTER);
        }
        if (key.getIdPrefixFilter() != null && !key.getIdPrefixFilter().isBlank()) {
            mainQb.add(new PrefixQuery(new Term(ID_FIELD, key.getIdPrefixFilter())), FILTER);
        }
        if (!StringUtils.isBlank(key.getTaxId())) {
            mainQb.add(new TermQuery(new Term(TAX_ID_FIELD, key.getTaxId())), Occur.FILTER);
        }

        return mainQb.build();
    }

    @NotNull
    private Builder buildFlatDisjunction(Map<String, Float> fieldWeights, String normalizedGeneName) {
        // Flat disjunction - build term queries for all tokens on all fields in one large disjunction
        Builder flatDisjunctionBuilder = new Builder();
        int flatDisjunctionClauses = 0;
        for (String token : normalizedGeneName.split("\\s+")) {
            String[] fieldsToSearch = CandidateFilter.isNumberGreekOrLatin(token) ? SYNONYM_FIELDS : ALL_FIELDS;
            Builder tokenQueryBuilder = new Builder();
            int tokenQueryClauses = 0;
            for (String field : fieldsToSearch) {
                Query tq = new TermQuery(new Term(field, token));
                if (fieldWeights != null) {
                    float weight = fieldWeights.getOrDefault(field, 1f);
                    tq = new BoostQuery(tq, weight);
                }
                tokenQueryBuilder.add(tq, SHOULD);
                ++tokenQueryClauses;
                if (tokenQueryClauses >= BooleanQuery.getMaxClauseCount())
                    break;
            }
            flatDisjunctionBuilder.add(tokenQueryBuilder.build(), SHOULD);
            ++flatDisjunctionClauses;
            if (flatDisjunctionClauses >= BooleanQuery.getMaxClauseCount())
                break;
        }
        return flatDisjunctionBuilder;
    }

//    @Override
//    public Query generateQuery(CandidateCacheKey key)
//            throws BooleanQuery.TooManyClauses {
//        Builder mainQb = new Builder();
//
//        Map<String, Float> fieldWeights = key.getFieldWeights();
//        boolean fieldweightsPresent = fieldWeights != null;
//        float dismaxTieBreaker = fieldweightsPresent ? fieldWeights.get(Configuration.PARAM_DISMAX_TIE_BREAKER) : 0.3f;
//
//        GeneName geneName = key.getGeneName();
//        Set<String> normalizedGeneNames = new HashSet<>();
//        Builder namesBuilder = new Builder();
//        List<GeneName> namesStream = Stream.concat(Stream.of(geneName), geneName.getAlternatives().stream()).collect(Collectors.toList());
//        for (GeneName name : namesStream) {
//            // make a query of the form "token1[All fields] AND token2[All fields] AND ..."
//            String normalizedName = name.getNormalizedText();
//            normalizedGeneNames.add(normalizedName);
//            // We wrap the main query - the CNF, Dismax-CNF or flat disjunction - into a boolean query of its own.
//            // This way we can set the minimumMustMatch parameter to gradually relax the query if necessary.
//            Builder mainTokenDisjunction = new Builder();
//            String[] tokens = normalizedName.split("\\s+");
//            if (!flatDisjunction) {
//                int effectiveRelaxation = relaxation != Integer.MAX_VALUE ? relaxation : tokens.length;
//                int max = Math.max(1, tokens.length - effectiveRelaxation);
//                mainTokenDisjunction.setMinimumNumberShouldMatch(max);
//                for (String token : tokens) {
//                    Builder fieldsDisjunctionBuilder = new Builder();
//                    String[] fieldsToSearch = CandidateFilter.isNumberGreekOrLatin(token) ? SYNONYM_FIELDS : ALL_FIELDS;
//                    Stream<Query> termQueryStream = Arrays.stream(fieldsToSearch).map(field -> new Term(field, token)).map(TermQuery::new);
//                    if (fieldWeights != null)
//                        termQueryStream = termQueryStream.map(TermQuery.class::cast).map(tq -> {
//                            float weight = fieldWeights.getOrDefault(tq.getTerm().field(), 1f);
//                            return new BoostQuery(tq, weight);
//                        });
//                    if (!useDisMax) {
//                        termQueryStream.forEach(tq -> fieldsDisjunctionBuilder.add(tq, Occur.SHOULD));
//                        mainTokenDisjunction.add(fieldsDisjunctionBuilder.build(), Occur.SHOULD);
//                    } else {
//                        DisjunctionMaxQuery dismaxQuery = new DisjunctionMaxQuery(termQueryStream.collect(Collectors.toList()), dismaxTieBreaker);
//                        mainTokenDisjunction.add(dismaxQuery, Occur.SHOULD);
//                    }
//                }
//                namesBuilder.add(mainTokenDisjunction.build(), Occur.SHOULD);
//                mainQb.add(namesBuilder.build(), Occur.MUST);
//            } else {
//                // We wrap the whole disjunction in a MUST clause. Otherwise, when combined with a filter,
//                // some random results with a score of 0 will be returned.
//                Builder flatDisjunction = new Builder();
//                buildFlatDisjunction(flatDisjunction, fieldWeights, normalizedName);
//                mainQb.add(flatDisjunction.build(), Occur.MUST);
//            }
//        }
//        // A compulsory filter clause saying that at least one token must be found in a candidate's name to be a match.
//        // This avoids candidates resulting purely from context fields (Generif, interactions etc)
////        Builder query4allNames = new Builder();
////        for (GeneName name : namesStream) {
////            String normalizedName = name.getNormalizedText();
////            normalizedGeneNames.add(normalizedName);
////            // This is basically "term1 OR term 2 OR term3 OR ..." with a minimumMustMatch of 1 so that at least
////            // one term must be a match on a synonym field
////            Builder query4allTokens = new Builder().setMinimumNumberShouldMatch(1);
////            String[] tokens = normalizedName.split("\\s+");
////            for (String token : tokens) {
////                if (CandidateFilter.isNumberGreekOrLatin(token))
////                    continue;
////
////                // This is a query for the current token on all the synonym fields.
////                Builder query4token = new Builder();
////                Stream<Query> termQueryStream = Arrays.stream(SYNONYM_FIELDS_NOT_CHROMOSOMAL).map(field -> new Term(field, token)).map(TermQuery::new);
////                termQueryStream.forEach(tq -> query4token.add(tq, SHOULD));
////                query4allTokens.add(query4token.build(), SHOULD);
////            }
////            query4allNames.add(query4allTokens.build(), SHOULD);
////        }
////        mainQb.add(query4allNames.build(), FILTER);
////        if (searchExactMatches) {
////        String normalizedGeneName = geneName.getNormalizedText();
////            // Add exact matches as additional relevance signals
////            Query exactSynonymQuery;
////            Stream<Query> exactSynonymQueries = Arrays.stream(SYNONYM_FIELDS).map(field -> new Term(field + "_exact", normalizedGeneName)).map(TermQuery::new);
////            if (fieldWeights != null)
////                exactSynonymQueries = exactSynonymQueries.map(TermQuery.class::cast).map(tq -> {
////                    float weight = fieldWeights.getOrDefault(tq.getTerm().field(), 1f);
////                    return new BoostQuery(tq, weight);
////                });
////            if (!useDisMax) {
////                Builder exactSynonymBuilder = new Builder();
////                exactSynonymQueries.forEach(q -> exactSynonymBuilder.add(q, SHOULD));
////                exactSynonymQuery = exactSynonymBuilder.build();
////            } else {
////                exactSynonymQuery = new DisjunctionMaxQuery(exactSynonymQueries.collect(Collectors.toList()), dismaxTieBreaker);
////            }
////            mainQb.add(exactSynonymQuery, SHOULD);
////        }
//        if (useSynonymMatchesAsRelevanceSignal) {
//            // Add another relevance signal by explicitly searching on the names
//            Query q = synonymsQueryGeneratorApprox.generateQuery(key);
//            mainQb.add(q, Occur.SHOULD);
//            q = synonymsQueryGeneratorExact.generateQuery(key);
//            mainQb.add(q, Occur.SHOULD);
//        }
//        if (useNpContextAsRelevanceSignal && geneName.getNominalPhraseContext() != null && !geneName.getNormalizedText().equals(geneName.getNominalPhraseContext())) {
//            // Add another relevance signal by searching for the nominal phrase tokens of the gene
//            if (!normalizedGeneNames.contains(geneName.getNominalPhraseContext().getNormalizedText())) {
//                BooleanQuery q = buildFlatDisjunction(fieldWeights, geneName.getNominalPhraseContext().getNormalizedText()).build();
//                mainQb.add(q, Occur.SHOULD);
//            }
//        }
//        if (useAlternativeNamesAsRelevanceSignal) {
//            for (GeneName alternative : geneName.getAlternatives()) {
//                mainQb.add(buildFlatDisjunction(fieldWeights, alternative.getNormalizedText()).build(), Occur.SHOULD);
//            }
//        }
//        if (useAppositionsAsRelevanceSignal) {
//            for (GeneName appositionContext : geneName.getAppositionContexts()) {
//                mainQb.add(buildFlatDisjunction(fieldWeights, appositionContext.getNormalizedText()).build(), Occur.SHOULD);
//            }
//        }
//        if (key.getContextNames() != null) {
//            for (GeneName contextName : key.getContextNames()) {
//                if (!normalizedGeneNames.contains(contextName.getNormalizedText()))
//                    mainQb.add(buildFlatDisjunction(fieldWeights, contextName.getNormalizedText()).build(), Occur.SHOULD);
//            }
//        }
//        if (key.getGeneIdsFilter() != null && !key.getGeneIdsFilter().isEmpty()) {
//            Builder filterBuilder = new Builder();
//            key.getGeneIdsFilter().stream().map(id -> new Term(SynonymIndexFieldNames.ID_FIELD, id)).map(TermQuery::new).forEach(tq -> filterBuilder.add(tq, Occur.SHOULD));
//            mainQb.add(filterBuilder.build(), Occur.FILTER);
//        }
//        if (key.getIdPrefixFilter() != null && !key.getIdPrefixFilter().isBlank()) {
//            mainQb.add(new PrefixQuery(new Term(SynonymIndexFieldNames.ID_FIELD, key.getIdPrefixFilter())), Occur.FILTER);
//        }
//        if (!StringUtils.isBlank(key.getTaxId())) {
//            mainQb.add(new TermQuery(new Term(SynonymIndexFieldNames.TAX_ID_FIELD, key.getTaxId())), Occur.FILTER);
//        }
//        if (!StringUtils.isBlank(key.getTermFilterField()) && !StringUtils.isBlank(key.getTermFilterTerm())) {
//            mainQb.add(new TermQuery(new Term(key.getTermFilterField(), key.getTermFilterTerm())), Occur.FILTER);
//        }
//
//        return mainQb.build();
//    }
//
//    @NotNull
//    private Builder buildFlatDisjunction(Map<String, Float> fieldWeights, String normalizedGeneName) {
//        return buildFlatDisjunction(null, fieldWeights, normalizedGeneName);
//    }
//
//    @NotNull
//    private Builder buildFlatDisjunction(Builder builder, Map<String, Float> fieldWeights, String normalizedGeneName) {
//        // Flat disjunction - build term queries for all tokens on all fields in one large disjunction
//        Builder flatDisjunctionBuilder = builder != null ? builder : new Builder();
//        int flatDisjunctionClauses = 0;
//        for (String token : normalizedGeneName.split("\\s+")) {
//            String[] fieldsToSearch = CandidateFilter.isNumberGreekOrLatin(token) ? SYNONYM_FIELDS : ALL_FIELDS;
//            Builder tokenQueryBuilder = new Builder();
//            int tokenQueryClauses = 0;
//            for (String field : fieldsToSearch) {
//                Query tq = new TermQuery(new Term(field, token));
//                if (fieldWeights != null) {
//                    float weight = fieldWeights.getOrDefault(field, 1f);
//                    tq = new BoostQuery(tq, weight);
//                }
//                tokenQueryBuilder.add(tq, Occur.SHOULD);
//                ++tokenQueryClauses;
//                if (tokenQueryClauses >= BooleanQuery.getMaxClauseCount())
//                    break;
//            }
//            flatDisjunctionBuilder.add(tokenQueryBuilder.build(), Occur.SHOULD);
//            ++flatDisjunctionClauses;
//            if (flatDisjunctionClauses >= BooleanQuery.getMaxClauseCount())
//                break;
//        }
//        return flatDisjunctionBuilder;
//    }

    @Override
    public String getName() {
        return getClass().getSimpleName() + (useDisMax ? "dismax" : "disjunction") + "-" + flatDisjunction + "-" + searchExactMatches + "-" + useSynonymMatchesAsRelevanceSignal + "-" + useNpContextAsRelevanceSignal + "-" + useAlternativeNamesAsRelevanceSignal + "-" + relaxation;
    }

    public boolean isUseContextGenesAsRelevanceSignal() {
        return useContextGenesAsRelevanceSignal;
    }
}
