package de.julielab.gene.candidateretrieval;

import de.julielab.geneexpbase.candidateretrieval.CandidateCacheKey;
import de.julielab.geneexpbase.candidateretrieval.QueryGenerator;
import de.julielab.geneexpbase.genemodel.GeneName;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause.Occur;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.BooleanQuery.Builder;
import org.apache.lucene.search.BoostQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Stream;

public class GeneRecordSynonymsQueryGenerator extends QueryGenerator {
    public static final String[] ALL_FIELDS = new String[]{SynonymIndexFieldNames.SYMBOL, SynonymIndexFieldNames.SYMBOL_FROM_NOMCENCLATURE, SynonymIndexFieldNames.SYNONYMS, SynonymIndexFieldNames.FULL_NAMES, SynonymIndexFieldNames.OTHER_DESIGNATIONS, SynonymIndexFieldNames.CHROMOSOME, SynonymIndexFieldNames.MAPLOCATION, SynonymIndexFieldNames.UNIPROT_NAMES, SynonymIndexFieldNames.BIO_THESAURUS};
    public static final List<Function<GeneRecordHit, String[]>> ALL_FIELD_FUNCTIONS = Arrays.asList(h -> new String[]{h.getSymbol()}, h -> new String[]{h.getSymbolFromNomenclature()}, h -> h.getSynonyms(), h -> h.getFullNames(), h -> h.getOtherDesignations(), h -> new String[]{h.getChromosome()}, h -> new String[]{h.getMapLocation()}, h -> h.getUniprotNames(), h -> h.getBioThesaurusNames());
    private final boolean exact;
    private final String[] fieldsToSearch;
    private final Occur tokenOccur;

    public GeneRecordSynonymsQueryGenerator(boolean exact) {
        this(exact, ALL_FIELDS);
    }

    public GeneRecordSynonymsQueryGenerator(boolean exact, String[] fieldsToSearch) {
        this(exact, fieldsToSearch, Occur.MUST);
    }

    public GeneRecordSynonymsQueryGenerator(boolean exact, String[] fieldsToSearch, Occur tokenOccur) {
        this.exact = exact;
        this.fieldsToSearch = fieldsToSearch;
        this.tokenOccur = tokenOccur;
    }

    @Override
    public Query generateQuery(CandidateCacheKey key)
            throws BooleanQuery.TooManyClauses {
        Map<String, Float> fieldWeights = key.getFieldWeights();
        GeneName geneName = key.getGeneName();
        Builder mainQb = new Builder();

        // make a query of the form "token1[All fields] AND token2[All fields] AND ..."
        String normalizedGeneName = geneName.getNormalizedText();
        String[] tokensToSearch = exact ? new String[]{normalizedGeneName} : normalizedGeneName.split("\\s+");
        Builder searchQuery = new Builder();
        for (String token : tokensToSearch) {
            Builder fieldsDisjunctionBuilder = new Builder();
            Stream<Query> termQueryStream = Arrays.stream(fieldsToSearch).map(field -> new Term(field + (exact ? "_exact" : ""), token)).map(TermQuery::new);
            if (fieldWeights != null)
                termQueryStream = termQueryStream.map(TermQuery.class::cast).map(tq -> {
                    float weight = fieldWeights.getOrDefault(tq.getTerm().field(), 1f);
                    return new BoostQuery(tq, weight);
                });
            termQueryStream.forEach(tq -> fieldsDisjunctionBuilder.add(tq, Occur.SHOULD));
            searchQuery.add(fieldsDisjunctionBuilder.build(), tokenOccur);
        }
        mainQb.add(searchQuery.build(), Occur.MUST);

        if (key.getGeneIdsFilter() != null && !key.getGeneIdsFilter().isEmpty()) {
            Builder filterBuilder = new Builder();
            key.getGeneIdsFilter().stream().map(id -> new Term(SynonymIndexFieldNames.ID_FIELD, id)).map(TermQuery::new).forEach(tq -> filterBuilder.add(tq, Occur.SHOULD));
            mainQb.add(filterBuilder.build(), Occur.FILTER);
        }
        if (!StringUtils.isBlank(key.getTaxId())) {
            mainQb.add(new TermQuery(new Term(SynonymIndexFieldNames.TAX_ID_FIELD, key.getTaxId())), Occur.FILTER);
        }
        return mainQb.build();
    }

    @Override
    public String getName() {
        return getClass().getSimpleName() + Arrays.toString(fieldsToSearch) + "-" + exact;
    }

}
