package de.julielab.gene.candidateretrieval;

import de.julielab.geneexpbase.GeneExpRuntimeException;
import de.julielab.geneexpbase.candidateretrieval.CandidateCacheKey;
import de.julielab.geneexpbase.candidateretrieval.QueryGenerator;
import org.apache.lucene.analysis.custom.CustomAnalyzer;
import org.apache.lucene.analysis.ngram.NGramFilterFactory;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.util.QueryBuilder;

import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Stream;

public class NGramQueryGenerator extends QueryGenerator {
    private transient CustomAnalyzer ngramAnalyzer;
    private final int min;
    private final int max;

    public NGramQueryGenerator(int min, int max) {
        this.min = min;
        this.max = max;
        ngramAnalyzer = init();
    }

    public CustomAnalyzer init() {
        CustomAnalyzer ngramAnalyzer;
        try {
            Map<String, String> ngramFilterSettings = new HashMap<>();
            ngramFilterSettings.put("minGramSize", String.valueOf(min));
            ngramFilterSettings.put("maxGramSize", String.valueOf(max));
            ngramAnalyzer = CustomAnalyzer.builder()
                    .withTokenizer("whitespace")
                    .addTokenFilter(NGramFilterFactory.class, ngramFilterSettings)
                    .build();
        } catch (IOException e) {
            throw new GeneExpRuntimeException(e);
        }
        return ngramAnalyzer;
    }

    @Override
    public Query generateQuery(CandidateCacheKey key) {
        if (ngramAnalyzer == null)
            ngramAnalyzer = init();
        String synonymsString = key.getGeneName().getText();
        Collection<String> geneIds = key.getGeneIdsFilter();
        QueryBuilder qb = new QueryBuilder(ngramAnalyzer);
        Query ngramQuery = qb.createBooleanQuery(SynonymIndexFieldNames.LOOKUP_SYN_FIELD_NGRAMS, synonymsString);
        BooleanQuery.Builder mainQb = new BooleanQuery.Builder().add(ngramQuery, BooleanClause.Occur.MUST);

        if (geneIds != null && ! geneIds.isEmpty()) {
            Stream<TermQuery> idTermQueries = geneIds.stream().map(id -> new TermQuery(new Term(SynonymIndexFieldNames.ID_FIELD, id)));
            BooleanQuery.Builder filterBuilder = new BooleanQuery.Builder();
            idTermQueries.forEach(q -> filterBuilder.add(q, BooleanClause.Occur.SHOULD));
            mainQb.add(filterBuilder.build(), BooleanClause.Occur.FILTER);
        }

        return mainQb.build();
    }

    @Override
    public String getName() {
        return "NGrams";
    }
}
